"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhApiCtrl = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _errorResponse = require("../lib/error-response");

var _json2csv = require("json2csv");

var _logger = require("../lib/logger");

var _csvKeyEquivalence = require("../../common/csv-key-equivalence");

var _apiErrorsEquivalence = require("../lib/api-errors-equivalence");

var _endpoints = _interopRequireDefault(require("../../common/api-info/endpoints"));

var _queue = require("../start/queue");

var _fs = _interopRequireDefault(require("fs"));

var _manageHosts = require("../lib/manage-hosts");

var _updateRegistry = require("../lib/update-registry");

var _jwtDecode = _interopRequireDefault(require("jwt-decode"));

var _cacheApiUserHasRunAs = require("../lib/cache-api-user-has-run-as");

var _cookie = require("../lib/cookie");

var _getConfiguration = require("../lib/get-configuration");

/*
 * Wazuh app - Class for Wazuh-API functions
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
// Require some libraries
class WazuhApiCtrl {
  constructor() {
    (0, _defineProperty2.default)(this, "manageHosts", void 0);
    (0, _defineProperty2.default)(this, "updateRegistry", void 0);
    this.manageHosts = new _manageHosts.ManageHosts();
    this.updateRegistry = new _updateRegistry.UpdateRegistry();
  }

  async getToken(context, request, response) {
    try {
      const {
        force,
        idHost
      } = request.body;
      const {
        username
      } = await context.wazuh.security.getCurrentUser(request, context);

      if (!force && request.headers.cookie && username === (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-user') && idHost === (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-api')) {
        const wzToken = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-token');

        if (wzToken) {
          try {
            // if the current token is not a valid jwt token we ask for a new one
            const decodedToken = (0, _jwtDecode.default)(wzToken);
            const expirationTime = decodedToken.exp - Date.now() / 1000;

            if (wzToken && expirationTime > 0) {
              return response.ok({
                body: {
                  token: wzToken
                }
              });
            }
          } catch (error) {
            (0, _logger.log)('wazuh-api:getToken', error.message || error);
          }
        }
      }

      let token;

      if ((await _cacheApiUserHasRunAs.APIUserAllowRunAs.canUse(idHost)) == _cacheApiUserHasRunAs.API_USER_STATUS_RUN_AS.ENABLED) {
        token = await context.wazuh.api.client.asCurrentUser.authenticate(idHost);
      } else {
        token = await context.wazuh.api.client.asInternalUser.authenticate(idHost);
      }

      ;
      let textSecure = '';

      if (context.wazuh.server.info.protocol === 'https') {
        textSecure = ';Secure';
      }

      return response.ok({
        headers: {
          'set-cookie': [`wz-token=${token};Path=/;HttpOnly${textSecure}`, `wz-user=${username};Path=/;HttpOnly${textSecure}`, `wz-api=${idHost};Path=/;HttpOnly`]
        },
        body: {
          token
        }
      });
    } catch (error) {
      const errorMessage = ((error.response || {}).data || {}).detail || error.message || error;
      (0, _logger.log)('wazuh-api:getToken', errorMessage);
      return (0, _errorResponse.ErrorResponse)(`Error getting the authorization token: ${errorMessage}`, 3000, 500, response);
    }
  }
  /**
   * Returns if the wazuh-api configuration is working
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} status obj or ErrorResponse
   */


  async checkStoredAPI(context, request, response) {
    try {
      // Get config from wazuh.yml
      const id = request.body.id;
      const api = await this.manageHosts.getHostById(id); // Check Manage Hosts

      if (!Object.keys(api).length) {
        throw new Error('Could not find Wazuh API entry on wazuh.yml');
      }

      (0, _logger.log)('wazuh-api:checkStoredAPI', `${id} exists`, 'debug'); // Fetch needed information about the cluster and the manager itself

      const responseManagerInfo = await context.wazuh.api.client.asInternalUser.request('get', `/manager/info`, {}, {
        apiHostID: id,
        forceRefresh: true
      }); // Look for socket-related errors

      if (this.checkResponseIsDown(responseManagerInfo)) {
        return (0, _errorResponse.ErrorResponse)(`ERROR3099 - ${responseManagerInfo.data.detail || 'Wazuh not ready yet'}`, 3099, 500, response);
      } // If we have a valid response from the Wazuh API


      if (responseManagerInfo.status === 200 && responseManagerInfo.data) {
        // Clear and update cluster information before being sent back to frontend
        delete api.cluster_info;
        const responseAgents = await context.wazuh.api.client.asInternalUser.request('GET', `/agents`, {
          params: {
            agents_list: '000'
          }
        }, {
          apiHostID: id
        });

        if (responseAgents.status === 200) {
          const managerName = responseAgents.data.data.affected_items[0].manager;
          const responseClusterStatus = await context.wazuh.api.client.asInternalUser.request('GET', `/cluster/status`, {}, {
            apiHostID: id
          });

          if (responseClusterStatus.status === 200) {
            if (responseClusterStatus.data.data.enabled === 'yes') {
              const responseClusterLocalInfo = await context.wazuh.api.client.asInternalUser.request('GET', `/cluster/local/info`, {}, {
                apiHostID: id
              });

              if (responseClusterLocalInfo.status === 200) {
                const clusterEnabled = responseClusterStatus.data.data.enabled === 'yes';
                api.cluster_info = {
                  status: clusterEnabled ? 'enabled' : 'disabled',
                  manager: managerName,
                  node: responseClusterLocalInfo.data.data.affected_items[0].node,
                  cluster: clusterEnabled ? responseClusterLocalInfo.data.data.affected_items[0].cluster : 'Disabled'
                };
              }
            } else {
              // Cluster mode is not active
              api.cluster_info = {
                status: 'disabled',
                manager: managerName,
                cluster: 'Disabled'
              };
            }
          } else {
            // Cluster mode is not active
            api.cluster_info = {
              status: 'disabled',
              manager: managerName,
              cluster: 'Disabled'
            };
          }

          if (api.cluster_info) {
            // Update cluster information in the wazuh-registry.json
            await this.updateRegistry.updateClusterInfo(id, api.cluster_info); // Hide Wazuh API secret, username, password

            const copied = { ...api
            };
            copied.secret = '****';
            copied.password = '****';
            return response.ok({
              body: {
                statusCode: 200,
                data: copied,
                idChanged: request.body.idChanged || null
              }
            });
          }
        }
      } // If we have an invalid response from the Wazuh API


      throw new Error(responseManagerInfo.data.detail || `${api.url}:${api.port} is unreachable`);
    } catch (error) {
      if (error.code === 'EPROTO') {
        return response.ok({
          body: {
            statusCode: 200,
            data: {
              apiIsDown: true
            }
          }
        });
      } else if (error.code === 'ECONNREFUSED') {
        return response.ok({
          body: {
            statusCode: 200,
            data: {
              apiIsDown: true
            }
          }
        });
      } else {
        try {
          const apis = await this.manageHosts.getHosts();

          for (const api of apis) {
            try {
              const id = Object.keys(api)[0];
              const responseManagerInfo = await context.wazuh.api.client.asInternalUser.request('GET', `/manager/info`, {}, {
                apiHostID: id
              });

              if (this.checkResponseIsDown(responseManagerInfo)) {
                return (0, _errorResponse.ErrorResponse)(`ERROR3099 - ${response.data.detail || 'Wazuh not ready yet'}`, 3099, 500, response);
              }

              if (responseManagerInfo.status === 200) {
                request.body.id = id;
                request.body.idChanged = id;
                return await this.checkStoredAPI(context, request, response);
              }
            } catch (error) {} // eslint-disable-line

          }
        } catch (error) {
          (0, _logger.log)('wazuh-api:checkStoredAPI', error.message || error);
          return (0, _errorResponse.ErrorResponse)(error.message || error, 3020, 500, response);
        }

        (0, _logger.log)('wazuh-api:checkStoredAPI', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 3002, 500, response);
      }
    }
  }
  /**
   * This perfoms a validation of API params
   * @param {Object} body API params
   */


  validateCheckApiParams(body) {
    if (!('username' in body)) {
      return 'Missing param: API USERNAME';
    }

    if (!('password' in body) && !('id' in body)) {
      return 'Missing param: API PASSWORD';
    }

    if (!('url' in body)) {
      return 'Missing param: API URL';
    }

    if (!('port' in body)) {
      return 'Missing param: API PORT';
    }

    if (!body.url.includes('https://') && !body.url.includes('http://')) {
      return 'protocol_error';
    }

    return false;
  }
  /**
   * This check the wazuh-api configuration received in the POST body will work
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} status obj or ErrorResponse
   */


  async checkAPI(context, request, response) {
    try {
      let apiAvailable = null; // const notValid = this.validateCheckApiParams(request.body);
      // if (notValid) return ErrorResponse(notValid, 3003, 500, response);

      (0, _logger.log)('wazuh-api:checkAPI', `${request.body.id} is valid`, 'debug'); // Check if a Wazuh API id is given (already stored API)

      const data = await this.manageHosts.getHostById(request.body.id);

      if (data) {
        apiAvailable = data;
      } else {
        (0, _logger.log)('wazuh-api:checkAPI', `API ${request.body.id} not found`);
        return (0, _errorResponse.ErrorResponse)(`The API ${request.body.id} was not found`, 3029, 500, response);
      }

      const options = {
        apiHostID: request.body.id
      };

      if (request.body.forceRefresh) {
        options["forceRefresh"] = request.body.forceRefresh;
      }

      let responseManagerInfo;

      try {
        responseManagerInfo = await context.wazuh.api.client.asInternalUser.request('GET', `/manager/info`, {}, options);
      } catch (error) {
        var _error$response, _error$response$data;

        return (0, _errorResponse.ErrorResponse)(`ERROR3099 - ${((_error$response = error.response) === null || _error$response === void 0 ? void 0 : (_error$response$data = _error$response.data) === null || _error$response$data === void 0 ? void 0 : _error$response$data.detail) || 'Wazuh not ready yet'}`, 3099, 500, response);
      }

      (0, _logger.log)('wazuh-api:checkAPI', `${request.body.id} credentials are valid`, 'debug');

      if (responseManagerInfo.status === 200 && responseManagerInfo.data) {
        let responseAgents = await context.wazuh.api.client.asInternalUser.request('GET', `/agents`, {
          params: {
            agents_list: '000'
          }
        }, {
          apiHostID: request.body.id
        });

        if (responseAgents.status === 200) {
          const managerName = responseAgents.data.data.affected_items[0].manager;
          let responseCluster = await context.wazuh.api.client.asInternalUser.request('GET', `/cluster/status`, {}, {
            apiHostID: request.body.id
          }); // Check the run_as for the API user and update it

          let apiUserAllowRunAs = _cacheApiUserHasRunAs.API_USER_STATUS_RUN_AS.ALL_DISABLED;
          const responseApiUserAllowRunAs = await context.wazuh.api.client.asInternalUser.request('GET', `/security/users/me`, {}, {
            apiHostID: request.body.id
          });

          if (responseApiUserAllowRunAs.status === 200) {
            const allow_run_as = responseApiUserAllowRunAs.data.data.affected_items[0].allow_run_as;
            if (allow_run_as && apiAvailable && apiAvailable.run_as) // HOST AND USER ENABLED
              apiUserAllowRunAs = _cacheApiUserHasRunAs.API_USER_STATUS_RUN_AS.ENABLED;else if (!allow_run_as && apiAvailable && apiAvailable.run_as) // HOST ENABLED AND USER DISABLED
              apiUserAllowRunAs = _cacheApiUserHasRunAs.API_USER_STATUS_RUN_AS.USER_NOT_ALLOWED;else if (allow_run_as && (!apiAvailable || !apiAvailable.run_as)) // USER ENABLED AND HOST DISABLED
              apiUserAllowRunAs = _cacheApiUserHasRunAs.API_USER_STATUS_RUN_AS.HOST_DISABLED;else if (!allow_run_as && (!apiAvailable || !apiAvailable.run_as)) // HOST AND USER DISABLED
              apiUserAllowRunAs = _cacheApiUserHasRunAs.API_USER_STATUS_RUN_AS.ALL_DISABLED;
          }

          _cacheApiUserHasRunAs.CacheInMemoryAPIUserAllowRunAs.set(request.body.id, apiAvailable.username, apiUserAllowRunAs);

          if (responseCluster.status === 200) {
            (0, _logger.log)('wazuh-api:checkStoredAPI', `Wazuh API response is valid`, 'debug');

            if (responseCluster.data.data.enabled === 'yes') {
              // If cluster mode is active
              let responseClusterLocal = await context.wazuh.api.client.asInternalUser.request('GET', `/cluster/local/info`, {}, {
                apiHostID: request.body.id
              });

              if (responseClusterLocal.status === 200) {
                return response.ok({
                  body: {
                    manager: managerName,
                    node: responseClusterLocal.data.data.affected_items[0].node,
                    cluster: responseClusterLocal.data.data.affected_items[0].cluster,
                    status: 'enabled',
                    allow_run_as: apiUserAllowRunAs
                  }
                });
              }
            } else {
              // Cluster mode is not active
              return response.ok({
                body: {
                  manager: managerName,
                  cluster: 'Disabled',
                  status: 'disabled',
                  allow_run_as: apiUserAllowRunAs
                }
              });
            }
          }
        }
      }
    } catch (error) {
      (0, _logger.log)('wazuh-api:checkAPI', error.message || error);

      if (error && error.response && error.response.status === 401) {
        return (0, _errorResponse.ErrorResponse)(`Unathorized. Please check API credentials. ${error.response.data.message}`, 401, 401, response);
      }

      if (error && error.response && error.response.data && error.response.data.detail) {
        return (0, _errorResponse.ErrorResponse)(error.response.data.detail, error.response.status || 500, error.response.status || 500, response);
      }

      if (error.code === 'EPROTO') {
        return (0, _errorResponse.ErrorResponse)('Wrong protocol being used to connect to the Wazuh API', 3005, 500, response);
      }

      return (0, _errorResponse.ErrorResponse)(error.message || error, 3005, 500, response);
    }
  }

  checkResponseIsDown(response) {
    if (response.status !== 200) {
      // Avoid "Error communicating with socket" like errors
      const socketErrorCodes = [1013, 1014, 1017, 1018, 1019];
      const status = (response.data || {}).status || 1;
      const isDown = socketErrorCodes.includes(status);
      isDown && (0, _logger.log)('wazuh-api:makeRequest', 'Wazuh API is online but Wazuh is not ready yet');
      return isDown;
    }

    return false;
  }
  /**
   * Check main Wazuh daemons status
   * @param {*} context Endpoint context
   * @param {*} api API entry stored in .wazuh
   * @param {*} path Optional. Wazuh API target path.
   */


  async checkDaemons(context, api, path) {
    try {
      const response = await context.wazuh.api.client.asInternalUser.request('GET', '/manager/status', {}, {
        apiHostID: api.id
      });
      const daemons = ((((response || {}).data || {}).data || {}).affected_items || [])[0] || {};
      const isCluster = ((api || {}).cluster_info || {}).status === 'enabled' && typeof daemons['wazuh-clusterd'] !== 'undefined';
      const wazuhdbExists = typeof daemons['wazuh-db'] !== 'undefined';
      const execd = daemons['wazuh-execd'] === 'running';
      const modulesd = daemons['wazuh-modulesd'] === 'running';
      const wazuhdb = wazuhdbExists ? daemons['wazuh-db'] === 'running' : true;
      const clusterd = isCluster ? daemons['wazuh-clusterd'] === 'running' : true;
      const isValid = execd && modulesd && wazuhdb && clusterd;
      isValid && (0, _logger.log)('wazuh-api:checkDaemons', `Wazuh is ready`, 'debug');

      if (path === '/ping') {
        return {
          isValid
        };
      }

      if (!isValid) {
        throw new Error('Wazuh not ready yet');
      }
    } catch (error) {
      (0, _logger.log)('wazuh-api:checkDaemons', error.message || error);
      return Promise.reject(error);
    }
  }

  sleep(timeMs) {
    // eslint-disable-next-line
    return new Promise((resolve, reject) => {
      setTimeout(resolve, timeMs);
    });
  }
  /**
   * Helper method for Dev Tools.
   * https://documentation.wazuh.com/current/user-manual/api/reference.html
   * Depending on the method and the path some parameters should be an array or not.
   * Since we allow the user to write the request using both comma-separated and array as well,
   * we need to check if it should be transformed or not.
   * @param {*} method The request method
   * @param {*} path The Wazuh API path
   */


  shouldKeepArrayAsIt(method, path) {
    // Methods that we must respect a do not transform them
    const isAgentsRestart = method === 'POST' && path === '/agents/restart';
    const isActiveResponse = method === 'PUT' && path.startsWith('/active-response');
    const isAddingAgentsToGroup = method === 'POST' && path.startsWith('/agents/group/'); // Returns true only if one of the above conditions is true

    return isAgentsRestart || isActiveResponse || isAddingAgentsToGroup;
  }
  /**
   * This performs a request over Wazuh API and returns its response
   * @param {String} method Method: GET, PUT, POST, DELETE
   * @param {String} path API route
   * @param {Object} data data and params to perform the request
   * @param {String} id API id
   * @param {Object} response
   * @returns {Object} API response or ErrorResponse
   */


  async makeRequest(context, method, path, data, id, response) {
    const devTools = !!(data || {}).devTools;

    try {
      const api = await this.manageHosts.getHostById(id);

      if (devTools) {
        delete data.devTools;
      }

      if (!Object.keys(api).length) {
        (0, _logger.log)('wazuh-api:makeRequest', 'Could not get host credentials'); //Can not get credentials from wazuh-hosts

        return (0, _errorResponse.ErrorResponse)('Could not get host credentials', 3011, 404, response);
      }

      if (!data) {
        data = {};
      }

      ;

      if (!data.headers) {
        data.headers = {};
      }

      ;
      const options = {
        apiHostID: id
      }; // Set content type application/xml if needed

      if (typeof (data || {}).body === 'string' && (data || {}).origin === 'xmleditor') {
        data.headers['content-type'] = 'application/xml';
        delete data.origin;
      }

      if (typeof (data || {}).body === 'string' && (data || {}).origin === 'json') {
        data.headers['content-type'] = 'application/json';
        delete data.origin;
      }

      if (typeof (data || {}).body === 'string' && (data || {}).origin === 'raw') {
        data.headers['content-type'] = 'application/octet-stream';
        delete data.origin;
      }

      const delay = (data || {}).delay || 0;

      if (delay) {
        (0, _queue.addJobToQueue)({
          startAt: new Date(Date.now() + delay),
          run: async () => {
            try {
              await context.wazuh.api.client.asCurrentUser.request(method, path, data, options);
            } catch (error) {
              (0, _logger.log)('queue:delayApiRequest', `An error ocurred in the delayed request: "${method} ${path}": ${error.message || error}`);
            }

            ;
          }
        });
        return response.ok({
          body: {
            error: 0,
            message: 'Success'
          }
        });
      }

      if (path === '/ping') {
        try {
          const check = await this.checkDaemons(context, api, path);
          return check;
        } catch (error) {
          const isDown = (error || {}).code === 'ECONNREFUSED';

          if (!isDown) {
            (0, _logger.log)('wazuh-api:makeRequest', 'Wazuh API is online but Wazuh is not ready yet');
            return (0, _errorResponse.ErrorResponse)(`ERROR3099 - ${error.message || 'Wazuh not ready yet'}`, 3099, 500, response);
          }
        }
      }

      (0, _logger.log)('wazuh-api:makeRequest', `${method} ${path}`, 'debug'); // Extract keys from parameters

      const dataProperties = Object.keys(data); // Transform arrays into comma-separated string if applicable.
      // The reason is that we are accepting arrays for comma-separated
      // parameters in the Dev Tools

      if (!this.shouldKeepArrayAsIt(method, path)) {
        for (const key of dataProperties) {
          if (Array.isArray(data[key])) {
            data[key] = data[key].join();
          }
        }
      }

      const responseToken = await context.wazuh.api.client.asCurrentUser.request(method, path, data, options);
      const responseIsDown = this.checkResponseIsDown(responseToken);

      if (responseIsDown) {
        return (0, _errorResponse.ErrorResponse)(`ERROR3099 - ${response.body.message || 'Wazuh not ready yet'}`, 3099, 500, response);
      }

      let responseBody = (responseToken || {}).data || {};

      if (!responseBody) {
        responseBody = typeof responseBody === 'string' && path.includes('/files') && method === 'GET' ? ' ' : false;
        response.data = responseBody;
      }

      const responseError = response.status !== 200 ? response.status : false;

      if (!responseError && responseBody) {
        //cleanKeys(response);
        return response.ok({
          body: responseToken.data
        });
      }

      if (responseError && devTools) {
        return response.ok({
          body: response.data
        });
      }

      throw responseError && responseBody.detail ? {
        message: responseBody.detail,
        code: responseError
      } : new Error('Unexpected error fetching data from the Wazuh API');
    } catch (error) {
      if (error && error.response && error.response.status === 401) {
        return (0, _errorResponse.ErrorResponse)(error.message || error, error.code ? `Wazuh API error: ${error.code}` : 3013, 401, response);
      }

      const errorMsg = (error.response || {}).data || error.message;
      (0, _logger.log)('wazuh-api:makeRequest', errorMsg || error);

      if (devTools) {
        return response.ok({
          body: {
            error: '3013',
            message: errorMsg || error
          }
        });
      } else {
        if ((error || {}).code && _apiErrorsEquivalence.ApiErrorEquivalence[error.code]) {
          error.message = _apiErrorsEquivalence.ApiErrorEquivalence[error.code];
        }

        return (0, _errorResponse.ErrorResponse)(errorMsg.detail || error, error.code ? `Wazuh API error: ${error.code}` : 3013, 500, response);
      }
    }
  }
  /**
   * This make a request to API
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} api response or ErrorResponse
   */


  requestApi(context, request, response) {
    const idApi = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-api');

    if (idApi !== request.body.id) {
      // if the current token belongs to a different API id, we relogin to obtain a new token
      return (0, _errorResponse.ErrorResponse)('status code 401', 401, 401, response);
    }

    if (!request.body.method) {
      return (0, _errorResponse.ErrorResponse)('Missing param: method', 3015, 400, response);
    } else if (!request.body.method.match(/^(?:GET|PUT|POST|DELETE)$/)) {
      (0, _logger.log)('wazuh-api:makeRequest', 'Request method is not valid.'); //Method is not a valid HTTP request method

      return (0, _errorResponse.ErrorResponse)('Request method is not valid.', 3015, 400, response);
    } else if (!request.body.path) {
      return (0, _errorResponse.ErrorResponse)('Missing param: path', 3016, 400, response);
    } else if (!request.body.path.startsWith('/')) {
      (0, _logger.log)('wazuh-api:makeRequest', 'Request path is not valid.'); //Path doesn't start with '/'

      return (0, _errorResponse.ErrorResponse)('Request path is not valid.', 3015, 400, response);
    } else {
      return this.makeRequest(context, request.body.method, request.body.path, request.body.body, request.body.id, response);
    }
  }
  /**
   * Get full data on CSV format from a list Wazuh API endpoint
   * @param {Object} ctx
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} csv or ErrorResponse
   */


  async csv(context, request, response) {
    try {
      if (!request.body || !request.body.path) throw new Error('Field path is required');
      if (!request.body.id) throw new Error('Field id is required');
      const filters = Array.isArray(((request || {}).body || {}).filters) ? request.body.filters : [];
      let tmpPath = request.body.path;

      if (tmpPath && typeof tmpPath === 'string') {
        tmpPath = tmpPath[0] === '/' ? tmpPath.substr(1) : tmpPath;
      }

      if (!tmpPath) throw new Error('An error occurred parsing path field');
      (0, _logger.log)('wazuh-api:csv', `Report ${tmpPath}`, 'debug'); // Real limit, regardless the user query

      const params = {
        limit: 500
      };

      if (filters.length) {
        for (const filter of filters) {
          if (!filter.name || !filter.value) continue;
          params[filter.name] = filter.value;
        }
      }

      let itemsArray = [];
      const output = await context.wazuh.api.client.asCurrentUser.request('GET', `/${tmpPath}`, {
        params: params
      }, {
        apiHostID: request.body.id
      });
      const isList = request.body.path.includes('/lists') && request.body.filters && request.body.filters.length && request.body.filters.find(filter => filter._isCDBList);
      const totalItems = (((output || {}).data || {}).data || {}).total_affected_items;

      if (totalItems && !isList) {
        params.offset = 0;
        itemsArray.push(...output.data.data.affected_items);

        while (itemsArray.length < totalItems && params.offset < totalItems) {
          params.offset += params.limit;
          const tmpData = await context.wazuh.api.client.asCurrentUser.request('GET', `/${tmpPath}`, {
            params: params
          }, {
            apiHostID: request.body.id
          });
          itemsArray.push(...tmpData.data.data.affected_items);
        }
      }

      if (totalItems) {
        const {
          path,
          filters
        } = request.body;
        const isArrayOfLists = path.includes('/lists') && !isList;
        const isAgents = path.includes('/agents') && !path.includes('groups');
        const isAgentsOfGroup = path.startsWith('/agents/groups/');
        const isFiles = path.endsWith('/files');
        let fields = Object.keys(output.data.data.affected_items[0]);

        if (isAgents || isAgentsOfGroup) {
          if (isFiles) {
            fields = ['filename', 'hash'];
          } else {
            fields = ['id', 'status', 'name', 'ip', 'group', 'manager', 'node_name', 'dateAdd', 'version', 'lastKeepAlive', 'os.arch', 'os.build', 'os.codename', 'os.major', 'os.minor', 'os.name', 'os.platform', 'os.uname', 'os.version'];
          }
        }

        if (isArrayOfLists) {
          const flatLists = [];

          for (const list of itemsArray) {
            const {
              relative_dirname,
              items
            } = list;
            flatLists.push(...items.map(item => ({
              relative_dirname,
              key: item.key,
              value: item.value
            })));
          }

          fields = ['relative_dirname', 'key', 'value'];
          itemsArray = [...flatLists];
        }

        if (isList) {
          fields = ['key', 'value'];
          itemsArray = output.data.data.affected_items[0].items;
        }

        fields = fields.map(item => ({
          value: item,
          default: '-'
        }));
        const json2csvParser = new _json2csv.Parser({
          fields
        });
        let csv = json2csvParser.parse(itemsArray);

        for (const field of fields) {
          const {
            value
          } = field;

          if (csv.includes(value)) {
            csv = csv.replace(value, _csvKeyEquivalence.KeyEquivalence[value] || value);
          }
        }

        return response.ok({
          headers: {
            'Content-Type': 'text/csv'
          },
          body: csv
        });
      } else if (output && output.data && output.data.data && !output.data.data.total_affected_items) {
        throw new Error('No results');
      } else {
        throw new Error(`An error occurred fetching data from the Wazuh API${output && output.data && output.data.detail ? `: ${output.body.detail}` : ''}`);
      }
    } catch (error) {
      (0, _logger.log)('wazuh-api:csv', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 3034, 500, response);
    }
  } // Get de list of available requests in the API


  getRequestList(context, request, response) {
    //Read a static JSON until the api call has implemented
    return response.ok({
      body: _endpoints.default
    });
  }
  /**
   * This get the timestamp field
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} timestamp field or ErrorResponse
   */


  getTimeStamp(context, request, response) {
    try {
      const source = JSON.parse(_fs.default.readFileSync(this.updateRegistry.file, 'utf8'));

      if (source.installationDate && source.lastRestart) {
        (0, _logger.log)('wazuh-api:getTimeStamp', `Installation date: ${source.installationDate}. Last restart: ${source.lastRestart}`, 'debug');
        return response.ok({
          body: {
            installationDate: source.installationDate,
            lastRestart: source.lastRestart
          }
        });
      } else {
        throw new Error('Could not fetch wazuh-version registry');
      }
    } catch (error) {
      (0, _logger.log)('wazuh-api:getTimeStamp', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || 'Could not fetch wazuh-version registry', 4001, 500, response);
    }
  }
  /**
   * This get the extensions
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} extensions object or ErrorResponse
   */


  async setExtensions(context, request, response) {
    try {
      const {
        id,
        extensions
      } = request.body; // Update cluster information in the wazuh-registry.json

      await this.updateRegistry.updateAPIExtensions(id, extensions);
      return response.ok({
        body: {
          statusCode: 200
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-api:setExtensions', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || 'Could not set extensions', 4001, 500, response);
    }
  }
  /**
   * This get the extensions
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} extensions object or ErrorResponse
   */


  getExtensions(context, request, response) {
    try {
      const source = JSON.parse(_fs.default.readFileSync(this.updateRegistry.file, 'utf8'));
      return response.ok({
        body: {
          extensions: (source.hosts[request.params.id] || {}).extensions || {}
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-api:getExtensions', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || 'Could not fetch wazuh-version registry', 4001, 500, response);
    }
  }
  /**
   * This get the wazuh setup settings
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} setup info or ErrorResponse
   */


  async getSetupInfo(context, request, response) {
    try {
      const source = JSON.parse(_fs.default.readFileSync(this.updateRegistry.file, 'utf8'));
      return response.ok({
        body: {
          statusCode: 200,
          data: !Object.values(source).length ? '' : source
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-api:getSetupInfo', error.message || error);
      return (0, _errorResponse.ErrorResponse)(`Could not get data from wazuh-version registry due to ${error.message || error}`, 4005, 500, response);
    }
  }
  /**
   * Get basic syscollector information for given agent.
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} Basic syscollector information
   */


  async getSyscollector(context, request, response) {
    try {
      const apiHostID = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-api');

      if (!request.params || !apiHostID || !request.params.agent) {
        throw new Error('Agent ID and API ID are required');
      }

      const {
        agent
      } = request.params;
      const data = await Promise.all([context.wazuh.api.client.asInternalUser.request('GET', `/syscollector/${agent}/hardware`, {}, {
        apiHostID
      }), context.wazuh.api.client.asInternalUser.request('GET', `/syscollector/${agent}/os`, {}, {
        apiHostID
      })]);
      const result = data.map(item => (item.data || {}).data || []);
      const [hardwareResponse, osResponse] = result; // Fill syscollector object

      const syscollector = {
        hardware: typeof hardwareResponse === 'object' && Object.keys(hardwareResponse).length ? { ...hardwareResponse.affected_items[0]
        } : false,
        os: typeof osResponse === 'object' && Object.keys(osResponse).length ? { ...osResponse.affected_items[0]
        } : false
      };
      return response.ok({
        body: syscollector
      });
    } catch (error) {
      (0, _logger.log)('wazuh-api:getSyscollector', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 3035, 500, response);
    }
  }
  /**
   * Check if user assigned roles disable Wazuh Plugin
   * @param context 
   * @param request 
   * @param response 
   * @returns {object} Returns { isWazuhDisabled: boolean parsed integer } 
   */


  async isWazuhDisabled(context, request, response) {
    try {
      const disabledRoles = (await (0, _getConfiguration.getConfiguration)())['disabled_roles'] || [];
      const logoSidebar = (await (0, _getConfiguration.getConfiguration)())['customization.logo.sidebar'];
      const data = (await context.wazuh.security.getCurrentUser(request, context)).authContext;
      const isWazuhDisabled = +(data.roles || []).some(role => disabledRoles.includes(role));
      return response.ok({
        body: {
          isWazuhDisabled,
          logoSidebar
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-api:isWazuhDisabled', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 3035, 500, response);
    }
  }

}

exports.WazuhApiCtrl = WazuhApiCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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