"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhReportingCtrl = void 0;

var _path = _interopRequireDefault(require("path"));

var _fs = _interopRequireDefault(require("fs"));

var _wazuhModules = require("../../common/wazuh-modules");

var TimSort = _interopRequireWildcard(require("timsort"));

var _errorResponse = require("../lib/error-response");

var VulnerabilityRequest = _interopRequireWildcard(require("../lib/reporting/vulnerability-request"));

var OverviewRequest = _interopRequireWildcard(require("../lib/reporting/overview-request"));

var RootcheckRequest = _interopRequireWildcard(require("../lib/reporting/rootcheck-request"));

var PCIRequest = _interopRequireWildcard(require("../lib/reporting/pci-request"));

var GDPRRequest = _interopRequireWildcard(require("../lib/reporting/gdpr-request"));

var TSCRequest = _interopRequireWildcard(require("../lib/reporting/tsc-request"));

var AuditRequest = _interopRequireWildcard(require("../lib/reporting/audit-request"));

var SyscheckRequest = _interopRequireWildcard(require("../lib/reporting/syscheck-request"));

var _pciRequirementsPdfmake = _interopRequireDefault(require("../integration-files/pci-requirements-pdfmake"));

var _gdprRequirementsPdfmake = _interopRequireDefault(require("../integration-files/gdpr-requirements-pdfmake"));

var _tscRequirementsPdfmake = _interopRequireDefault(require("../integration-files/tsc-requirements-pdfmake"));

var _processStateEquivalence = _interopRequireDefault(require("../lib/process-state-equivalence"));

var _csvKeyEquivalence = require("../../common/csv-key-equivalence");

var _agentConfiguration = require("../lib/reporting/agent-configuration");

var _printer = require("../lib/reporting/printer");

var _logger = require("../lib/logger");

var _constants = require("../../common/constants");

var _filesystem = require("../lib/filesystem");

var _moment = _interopRequireDefault(require("moment"));

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Wazuh app - Class for Wazuh reporting controller
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
class WazuhReportingCtrl {
  constructor() {}
  /**
   * This do format to filters
   * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
   * @param {String} searchBar search term
   */


  sanitizeKibanaFilters(filters, searchBar) {
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `Started to sanitize filters`, 'info');
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `filters: ${filters.length}, searchBar: ${searchBar}`, 'debug');
    let str = '';
    const agentsFilter = []; //separate agents filter

    filters = filters.filter(filter => {
      if (filter.meta.controlledBy === _constants.AUTHORIZED_AGENTS) {
        agentsFilter.push(filter);
        return false;
      }

      return filter;
    });
    const len = filters.length;

    for (let i = 0; i < len; i++) {
      const {
        negate,
        key,
        value,
        params,
        type
      } = filters[i].meta;
      str += `${negate ? 'NOT ' : ''}`;
      str += `${key}: `;
      str += `${type === 'range' ? `${params.gte}-${params.lt}` : type === 'phrases' ? '(' + params.join(" OR ") + ')' : type === 'exists' ? '*' : !!value ? value : (params || {}).query}`;
      str += `${i === len - 1 ? '' : ' AND '}`;
    }

    if (searchBar) {
      str += ` AND (${searchBar})`;
    }

    const agentsFilterStr = agentsFilter.map(filter => filter.meta.value).join(',');
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `str: ${str}, agentsFilterStr: ${agentsFilterStr}`, 'debug');
    return [str, agentsFilterStr];
  }
  /**
   * This performs the rendering of given header
   * @param {String} printer section target
   * @param {String} section section target
   * @param {Object} tab tab target
   * @param {Boolean} isAgents is agents section
   * @param {String} apiId ID of API
   */


  async renderHeader(context, printer, section, tab, isAgents, apiId) {
    try {
      (0, _logger.log)('reporting:renderHeader', `section: ${section}, tab: ${tab}, isAgents: ${isAgents}, apiId: ${apiId}`, 'debug');

      if (section && typeof section === 'string') {
        if (!['agentConfig', 'groupConfig'].includes(section)) {
          printer.addContent({
            text: _wazuhModules.WAZUH_MODULES[tab].title + ' report',
            style: 'h1'
          });
        } else if (section === 'agentConfig') {
          printer.addContent({
            text: `Agent ${isAgents} configuration`,
            style: 'h1'
          });
        } else if (section === 'groupConfig') {
          printer.addContent({
            text: 'Agents in group',
            style: {
              fontSize: 14,
              color: '#000'
            },
            margin: [0, 20, 0, 0]
          });

          if (section === 'groupConfig' && !Object.keys(isAgents).length) {
            printer.addContent({
              text: 'There are still no agents in this group.',
              style: {
                fontSize: 12,
                color: '#000'
              },
              margin: [0, 10, 0, 0]
            });
          }
        }

        printer.addNewLine();
      }

      if (isAgents && typeof isAgents === 'object') {
        await this.buildAgentsTable(context, printer, isAgents, apiId, section === 'groupConfig' ? tab : false);
      }

      if (isAgents && typeof isAgents === 'string') {
        const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents`, {
          params: {
            agents_list: isAgents
          }
        }, {
          apiHostID: apiId
        });
        const agentData = agentResponse.data.data.affected_items[0];

        if (agentData && agentData.status !== 'active') {
          printer.addContentWithNewLine({
            text: `Warning. Agent is ${agentData.status.toLowerCase()}`,
            style: 'standard'
          });
        }

        await this.buildAgentsTable(context, printer, [isAgents], apiId);

        if (agentData && agentData.group) {
          const agentGroups = agentData.group.join(', ');
          printer.addContentWithNewLine({
            text: `Group${agentData.group.length > 1 ? 's' : ''}: ${agentGroups}`,
            style: 'standard'
          });
        }
      }

      if (_wazuhModules.WAZUH_MODULES[tab] && _wazuhModules.WAZUH_MODULES[tab].description) {
        printer.addContentWithNewLine({
          text: _wazuhModules.WAZUH_MODULES[tab].description,
          style: 'standard'
        });
      }

      return;
    } catch (error) {
      (0, _logger.log)('reporting:renderHeader', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This build the agents table
   * @param {Array<Strings>} ids ids of agents
   * @param {String} apiId API id
   */


  async buildAgentsTable(context, printer, agentIDs, apiId, multi = false) {
    const dateFormat = await context.core.uiSettings.client.get('dateFormat');
    if (!agentIDs || !agentIDs.length) return;
    (0, _logger.log)('reporting:buildAgentsTable', `${agentIDs.length} agents for API ${apiId}`, 'info');

    try {
      let agentRows = [];

      if (multi) {
        try {
          const agentsResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${multi}/agents`, {}, {
            apiHostID: apiId
          });
          const agentsData = agentsResponse && agentsResponse.data && agentsResponse.data.data && agentsResponse.data.data.affected_items;
          agentRows = (agentsData || []).map(agent => ({ ...agent,
            manager: agent.manager || agent.manager_host,
            os: agent.os && agent.os.name && agent.os.version ? `${agent.os.name} ${agent.os.version}` : ''
          }));
        } catch (error) {
          (0, _logger.log)('reporting:buildAgentsTable', `Skip agent due to: ${error.message || error}`, 'debug');
        }
      } else {
        for (const agentID of agentIDs) {
          try {
            const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents`, {
              params: {
                q: `id=${agentID}`
              }
            }, {
              apiHostID: apiId
            });
            const [agent] = agentResponse.data.data.affected_items;
            agentRows.push({ ...agent,
              manager: agent.manager || agent.manager_host,
              os: agent.os && agent.os.name && agent.os.version ? `${agent.os.name} ${agent.os.version}` : '',
              lastKeepAlive: (0, _moment.default)(agent.lastKeepAlive).format(dateFormat),
              dateAdd: (0, _moment.default)(agent.dateAdd).format(dateFormat)
            });
          } catch (error) {
            (0, _logger.log)('reporting:buildAgentsTable', `Skip agent due to: ${error.message || error}`, 'debug');
          }
        }
      }

      printer.addSimpleTable({
        columns: [{
          id: 'id',
          label: 'ID'
        }, {
          id: 'name',
          label: 'Name'
        }, {
          id: 'ip',
          label: 'IP'
        }, {
          id: 'version',
          label: 'Version'
        }, {
          id: 'manager',
          label: 'Manager'
        }, {
          id: 'os',
          label: 'OS'
        }, {
          id: 'dateAdd',
          label: 'Registration date'
        }, {
          id: 'lastKeepAlive',
          label: 'Last keep alive'
        }],
        items: agentRows
      });
    } catch (error) {
      (0, _logger.log)('reporting:buildAgentsTable', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This load more information
   * @param {*} context Endpoint context
   * @param {*} printer printer instance
   * @param {String} section section target
   * @param {Object} tab tab target
   * @param {String} apiId ID of API
   * @param {Number} from Timestamp (ms) from
   * @param {Number} to Timestamp (ms) to
   * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
   * @param {String} pattern
   * @param {Object} agent agent target
   * @returns {Object} Extended information
   */


  async extendedInformation(context, printer, section, tab, apiId, from, to, filters, pattern = _constants.WAZUH_ALERTS_PATTERN, agent = null) {
    try {
      (0, _logger.log)('reporting:extendedInformation', `Section ${section} and tab ${tab}, API is ${apiId}. From ${from} to ${to}. Filters ${filters}. Index pattern ${pattern}`, 'info');

      if (section === 'agents' && !agent) {
        throw new Error('Reporting for specific agent needs an agent ID in order to work properly');
      }

      const agents = await context.wazuh.api.client.asCurrentUser.request('GET', '/agents', {
        params: {
          limit: 1
        }
      }, {
        apiHostID: apiId
      });
      const totalAgents = agents.data.data.total_affected_items;

      if (section === 'overview' && tab === 'vuls') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector metrics', 'debug');
        const vulnerabilitiesLevels = ['Low', 'Medium', 'High', 'Critical'];
        const vulnerabilitiesResponsesCount = (await Promise.all(vulnerabilitiesLevels.map(async vulnerabilitiesLevel => {
          try {
            const count = await VulnerabilityRequest.uniqueSeverityCount(context, from, to, vulnerabilitiesLevel, filters, pattern);
            return count ? `${count} of ${totalAgents} agents have ${vulnerabilitiesLevel.toLocaleLowerCase()} vulnerabilities.` : undefined;
          } catch (error) {}
        }))).filter(vulnerabilitiesResponse => vulnerabilitiesResponse);
        printer.addList({
          title: {
            text: 'Summary',
            style: 'h2'
          },
          list: vulnerabilitiesResponsesCount
        });
        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector top 3 agents by category', 'debug');
        const lowRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Low', filters, pattern);
        const mediumRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Medium', filters, pattern);
        const highRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'High', filters, pattern);
        const criticalRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Critical', filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding overview vulnerability detector top 3 agents by category', 'debug');

        if (criticalRank && criticalRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with critical severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, criticalRank, apiId);
          printer.addNewLine();
        }

        if (highRank && highRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with high severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, highRank, apiId);
          printer.addNewLine();
        }

        if (mediumRank && mediumRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with medium severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, mediumRank, apiId);
          printer.addNewLine();
        }

        if (lowRank && lowRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with low severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, lowRank, apiId);
          printer.addNewLine();
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector top 3 CVEs', 'debug');
        const cveRank = await VulnerabilityRequest.topCVECount(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding overview vulnerability detector top 3 CVEs', 'debug');

        if (cveRank && cveRank.length) {
          printer.addSimpleTable({
            title: {
              text: 'Top 3 CVE',
              style: 'h2'
            },
            columns: [{
              id: 'top',
              label: 'Top'
            }, {
              id: 'cve',
              label: 'CVE'
            }],
            items: cveRank.map(item => ({
              top: cveRank.indexOf(item) + 1,
              cve: item
            }))
          });
        }
      }

      if (section === 'overview' && tab === 'general') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 agents with level 15 alerts', 'debug');
        const level15Rank = await OverviewRequest.topLevel15(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding top 3 agents with level 15 alerts', 'debug');

        if (level15Rank.length) {
          printer.addContent({
            text: 'Top 3 agents with level 15 alerts',
            style: 'h2'
          });
          await this.buildAgentsTable(context, printer, level15Rank, apiId);
        }
      }

      if (section === 'overview' && tab === 'pm') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching most common rootkits', 'debug');
        const top5RootkitsRank = await RootcheckRequest.top5RootkitsDetected(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding most common rootkits', 'debug');

        if (top5RootkitsRank && top5RootkitsRank.length) {
          printer.addContentWithNewLine({
            text: 'Most common rootkits found among your agents',
            style: 'h2'
          }).addContentWithNewLine({
            text: 'Rootkits are a set of software tools that enable an unauthorized user to gain control of a computer system without being detected.',
            style: 'standard'
          }).addSimpleTable({
            items: top5RootkitsRank.map(item => {
              return {
                top: top5RootkitsRank.indexOf(item) + 1,
                name: item
              };
            }),
            columns: [{
              id: 'top',
              label: 'Top'
            }, {
              id: 'name',
              label: 'Rootkit'
            }]
          });
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching hidden pids', 'debug');
        const hiddenPids = await RootcheckRequest.agentsWithHiddenPids(context, from, to, filters, pattern);
        hiddenPids && printer.addContent({
          text: `${hiddenPids} of ${totalAgents} agents have hidden processes`,
          style: 'h3'
        });
        !hiddenPids && printer.addContentWithNewLine({
          text: `No agents have hidden processes`,
          style: 'h3'
        });
        const hiddenPorts = await RootcheckRequest.agentsWithHiddenPorts(context, from, to, filters, pattern);
        hiddenPorts && printer.addContent({
          text: `${hiddenPorts} of ${totalAgents} agents have hidden ports`,
          style: 'h3'
        });
        !hiddenPorts && printer.addContent({
          text: `No agents have hidden ports`,
          style: 'h3'
        });
        printer.addNewLine();
      }

      if (['overview', 'agents'].includes(section) && tab === 'pci') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top PCI DSS requirements', 'debug');
        const topPciRequirements = await PCIRequest.topPCIRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common PCI DSS requirements alerts found',
          style: 'h2'
        });

        for (const item of topPciRequirements) {
          const rules = await PCIRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_pciRequirementsPdfmake.default[item]) {
            const content = typeof _pciRequirementsPdfmake.default[item] === 'string' ? {
              text: _pciRequirementsPdfmake.default[item],
              style: 'standard'
            } : _pciRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }
      }

      if (['overview', 'agents'].includes(section) && tab === 'tsc') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top TSC requirements', 'debug');
        const topTSCRequirements = await TSCRequest.topTSCRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common TSC requirements alerts found',
          style: 'h2'
        });

        for (const item of topTSCRequirements) {
          const rules = await TSCRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_tscRequirementsPdfmake.default[item]) {
            const content = typeof _tscRequirementsPdfmake.default[item] === 'string' ? {
              text: _tscRequirementsPdfmake.default[item],
              style: 'standard'
            } : _tscRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }
      }

      if (['overview', 'agents'].includes(section) && tab === 'gdpr') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top GDPR requirements', 'debug');
        const topGdprRequirements = await GDPRRequest.topGDPRRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common GDPR requirements alerts found',
          style: 'h2'
        });

        for (const item of topGdprRequirements) {
          const rules = await GDPRRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_gdprRequirementsPdfmake.default && _gdprRequirementsPdfmake.default[item]) {
            const content = typeof _gdprRequirementsPdfmake.default[item] === 'string' ? {
              text: _gdprRequirementsPdfmake.default[item],
              style: 'standard'
            } : _gdprRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }

        printer.addNewLine();
      }

      if (section === 'overview' && tab === 'audit') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching agents with high number of failed sudo commands', 'debug');
        const auditAgentsNonSuccess = await AuditRequest.getTop3AgentsSudoNonSuccessful(context, from, to, filters, pattern);

        if (auditAgentsNonSuccess && auditAgentsNonSuccess.length) {
          printer.addContent({
            text: 'Agents with high number of failed sudo commands',
            style: 'h2'
          });
          await this.buildAgentsTable(context, printer, auditAgentsNonSuccess, apiId);
        }

        const auditAgentsFailedSyscall = await AuditRequest.getTop3AgentsFailedSyscalls(context, from, to, filters, pattern);

        if (auditAgentsFailedSyscall && auditAgentsFailedSyscall.length) {
          printer.addSimpleTable({
            columns: [{
              id: 'agent',
              label: 'Agent ID'
            }, {
              id: 'syscall_id',
              label: 'Syscall ID'
            }, {
              id: 'syscall_syscall',
              label: 'Syscall'
            }],
            items: auditAgentsFailedSyscall.map(item => ({
              agent: item.agent,
              syscall_id: item.syscall.id,
              syscall_syscall: item.syscall.syscall
            })),
            title: {
              text: 'Most common failing syscalls',
              style: 'h2'
            }
          });
        }
      }

      if (section === 'overview' && tab === 'fim') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 rules for FIM', 'debug');
        const rules = await SyscheckRequest.top3Rules(context, from, to, filters, pattern);

        if (rules && rules.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 FIM rules',
            style: 'h2'
          }).addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: {
              text: 'Top 3 rules that are generating most alerts.',
              style: 'standard'
            }
          });
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 agents for FIM', 'debug');
        const agents = await SyscheckRequest.top3agents(context, from, to, filters, pattern);

        if (agents && agents.length) {
          printer.addContentWithNewLine({
            text: 'Agents with suspicious FIM activity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'Top 3 agents that have most FIM alerts from level 7 to level 15. Take care about them.',
            style: 'standard'
          });
          await this.buildAgentsTable(context, printer, agents, apiId);
        }
      }

      if (section === 'agents' && tab === 'audit') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching most common failed syscalls`, 'debug');
        const auditFailedSyscall = await AuditRequest.getTopFailedSyscalls(context, from, to, filters, pattern);
        auditFailedSyscall && auditFailedSyscall.length && printer.addSimpleTable({
          columns: [{
            id: 'id',
            label: 'id'
          }, {
            id: 'syscall',
            label: 'Syscall'
          }],
          items: auditFailedSyscall,
          title: 'Most common failing syscalls'
        });
      }

      if (section === 'agents' && tab === 'fim') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching syscheck database for agent ${agent}`, 'debug');
        const lastScanResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/syscheck/${agent}/last_scan`, {}, {
          apiHostID: apiId
        });

        if (lastScanResponse && lastScanResponse.data) {
          const lastScanData = lastScanResponse.data.data.affected_items[0];

          if (lastScanData.start && lastScanData.end) {
            printer.addContent({
              text: `Last file integrity monitoring scan was executed from ${lastScanData.start} to ${lastScanData.end}.`
            });
          } else if (lastScanData.start) {
            printer.addContent({
              text: `File integrity monitoring scan is currently in progress for this agent (started on ${lastScanData.start}).`
            });
          } else {
            printer.addContent({
              text: `File integrity monitoring scan is currently in progress for this agent.`
            });
          }

          printer.addNewLine();
        }

        (0, _logger.log)('reporting:extendedInformation', `Fetching last 10 deleted files for FIM`, 'debug');
        const lastTenDeleted = await SyscheckRequest.lastTenDeletedFiles(context, from, to, filters, pattern);
        lastTenDeleted && lastTenDeleted.length && printer.addSimpleTable({
          columns: [{
            id: 'path',
            label: 'Path'
          }, {
            id: 'date',
            label: 'Date'
          }],
          items: lastTenDeleted,
          title: 'Last 10 deleted files'
        });
        (0, _logger.log)('reporting:extendedInformation', `Fetching last 10 modified files`, 'debug');
        const lastTenModified = await SyscheckRequest.lastTenModifiedFiles(context, from, to, filters, pattern);
        lastTenModified && lastTenModified.length && printer.addSimpleTable({
          columns: [{
            id: 'path',
            label: 'Path'
          }, {
            id: 'date',
            label: 'Date'
          }],
          items: lastTenModified,
          title: 'Last 10 modified files'
        });
      }

      if (section === 'agents' && tab === 'syscollector') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching hardware information for agent ${agent}`, 'debug');
        const requestsSyscollectorLists = [{
          endpoint: `/syscollector/${agent}/hardware`,
          loggerMessage: `Fetching Hardware information for agent ${agent}`,
          list: {
            title: {
              text: 'Hardware information',
              style: 'h2'
            }
          },
          mapResponse: hardware => [hardware.cpu && hardware.cpu.cores && `${hardware.cpu.cores} cores`, hardware.cpu && hardware.cpu.name, hardware.ram && hardware.ram.total && `${Number(hardware.ram.total / 1024 / 1024).toFixed(2)}GB RAM`]
        }, {
          endpoint: `/syscollector/${agent}/os`,
          loggerMessage: `Fetching OS information for agent ${agent}`,
          list: {
            title: {
              text: 'OS information',
              style: 'h2'
            }
          },
          mapResponse: osData => [osData.sysname, osData.version, osData.architecture, osData.release, osData.os && osData.os.name && osData.os.version && `${osData.os.name} ${osData.os.version}`]
        }];
        const syscollectorLists = await Promise.all(requestsSyscollectorLists.map(async requestSyscollector => {
          try {
            (0, _logger.log)('reporting:extendedInformation', requestSyscollector.loggerMessage, 'debug');
            const responseSyscollector = await context.wazuh.api.client.asCurrentUser.request('GET', requestSyscollector.endpoint, {}, {
              apiHostID: apiId
            });
            const [data] = responseSyscollector && responseSyscollector.data && responseSyscollector.data.data && responseSyscollector.data.data.affected_items || [];

            if (data) {
              return { ...requestSyscollector.list,
                list: requestSyscollector.mapResponse(data)
              };
            }
          } catch (error) {
            (0, _logger.log)('reporting:extendedInformation', error.message || error);
          }
        }));

        if (syscollectorLists) {
          syscollectorLists.filter(syscollectorList => syscollectorList).forEach(syscollectorList => printer.addList(syscollectorList));
        }

        const vulnerabilitiesRequests = ['Critical', 'High'];
        const vulnerabilitiesResponsesItems = (await Promise.all(vulnerabilitiesRequests.map(async vulnerabilitiesLevel => {
          try {
            (0, _logger.log)('reporting:extendedInformation', `Fetching top ${vulnerabilitiesLevel} packages`, 'debug');
            return await VulnerabilityRequest.topPackages(context, from, to, vulnerabilitiesLevel, filters, pattern);
          } catch (error) {
            (0, _logger.log)('reporting:extendedInformation', error.message || error);
          }
        }))).filter(vulnerabilitiesResponse => vulnerabilitiesResponse).flat();

        if (vulnerabilitiesResponsesItems && vulnerabilitiesResponsesItems.length) {
          printer.addSimpleTable({
            title: {
              text: 'Vulnerable packages found (last 24 hours)',
              style: 'h2'
            },
            columns: [{
              id: 'package',
              label: 'Package'
            }, {
              id: 'severity',
              label: 'Severity'
            }],
            items: vulnerabilitiesResponsesItems
          });
        }
      }

      if (section === 'agents' && tab === 'vuls') {
        const topCriticalPackages = await VulnerabilityRequest.topPackagesWithCVE(context, from, to, 'Critical', filters, pattern);

        if (topCriticalPackages && topCriticalPackages.length) {
          printer.addContentWithNewLine({
            text: 'Critical severity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'These vulnerabilties are critical, please review your agent. Click on each link to read more about each found vulnerability.',
            style: 'standard'
          });
          const customul = [];

          for (const critical of topCriticalPackages) {
            customul.push({
              text: critical.package,
              style: 'standard'
            });
            customul.push({
              ul: critical.references.map(item => ({
                text: item.substring(0, 80) + '...',
                link: item,
                color: '#1EA5C8'
              }))
            });
          }

          printer.addContentWithNewLine({
            ul: customul
          });
        }

        const topHighPackages = await VulnerabilityRequest.topPackagesWithCVE(context, from, to, 'High', filters, pattern);

        if (topHighPackages && topHighPackages.length) {
          printer.addContentWithNewLine({
            text: 'High severity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'Click on each link to read more about each found vulnerability.',
            style: 'standard'
          });
          const customul = [];

          for (const critical of topHighPackages) {
            customul.push({
              text: critical.package,
              style: 'standard'
            });
            customul.push({
              ul: critical.references.map(item => ({
                text: item,
                color: '#1EA5C8'
              }))
            });
          }

          customul && customul.length && printer.addContent({
            ul: customul
          });
          printer.addNewLine();
        }
      }

      return false;
    } catch (error) {
      (0, _logger.log)('reporting:extendedInformation', error.message || error);
      return Promise.reject(error);
    }
  }

  getConfigRows(data, labels) {
    (0, _logger.log)('reporting:getConfigRows', `Building configuration rows`, 'info');
    const result = [];

    for (let prop in data || []) {
      if (Array.isArray(data[prop])) {
        data[prop].forEach((x, idx) => {
          if (typeof x === 'object') data[prop][idx] = JSON.stringify(x);
        });
      }

      result.push([(labels || {})[prop] || _csvKeyEquivalence.KeyEquivalence[prop] || prop, data[prop] || '-']);
    }

    return result;
  }

  getConfigTables(data, section, tab, array = []) {
    (0, _logger.log)('reporting:getConfigTables', `Building configuration tables`, 'info');
    let plainData = {};
    const nestedData = [];
    const tableData = [];

    if (data.length === 1 && Array.isArray(data)) {
      tableData[section.config[tab].configuration] = data;
    } else {
      for (let key in data) {
        if (typeof data[key] !== 'object' && !Array.isArray(data[key]) || Array.isArray(data[key]) && typeof data[key][0] !== 'object') {
          plainData[key] = Array.isArray(data[key]) && typeof data[key][0] !== 'object' ? data[key].map(x => {
            return typeof x === 'object' ? JSON.stringify(x) : x + '\n';
          }) : data[key];
        } else if (Array.isArray(data[key]) && typeof data[key][0] === 'object') {
          tableData[key] = data[key];
        } else {
          if (section.isGroupConfig && ['pack', 'content'].includes(key)) {
            tableData[key] = [data[key]];
          } else {
            nestedData.push(data[key]);
          }
        }
      }
    }

    array.push({
      title: (section.options || {}).hideHeader ? '' : (section.tabs || [])[tab] || (section.isGroupConfig ? ((section.labels || [])[0] || [])[tab] : ''),
      columns: ['', ''],
      type: 'config',
      rows: this.getConfigRows(plainData, (section.labels || [])[0])
    });

    for (let key in tableData) {
      const columns = Object.keys(tableData[key][0]);
      columns.forEach((col, i) => {
        columns[i] = col[0].toUpperCase() + col.slice(1);
      });
      const rows = tableData[key].map(x => {
        let row = [];

        for (let key in x) {
          row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
            return x + '\n';
          }) : JSON.stringify(x[key]));
        }

        while (row.length < columns.length) {
          row.push('-');
        }

        return row;
      });
      array.push({
        title: ((section.labels || [])[0] || [])[key] || '',
        type: 'table',
        columns,
        rows
      });
    }

    nestedData.forEach(nest => {
      this.getConfigTables(nest, section, tab + 1, array);
    });
    return array;
  }
  /**
   * Create a report for the modules
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {*} reports list or ErrorResponse
   */


  async createReportsModules(context, request, response) {
    try {
      (0, _logger.log)('reporting:createReportsModules', `Report started`, 'info');
      const {
        array,
        agents,
        browserTimezone,
        searchBar,
        filters,
        time,
        tables,
        name,
        section,
        indexPatternTitle,
        apiId
      } = request.body;
      const {
        moduleID
      } = request.params;
      const {
        from,
        to
      } = time || {}; // Init

      const printer = new _printer.ReportPrinter();
      const {
        username: userID
      } = await context.wazuh.security.getCurrentUser(request, context);
      (0, _filesystem.createDataDirectoryIfNotExists)();
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID));
      await this.renderHeader(context, printer, section, moduleID, agents, apiId);
      const [sanitizedFilters, agentsFilter] = filters ? this.sanitizeKibanaFilters(filters, searchBar) : [false, false];

      if (time && sanitizedFilters) {
        printer.addTimeRangeAndFilters(from, to, sanitizedFilters, browserTimezone);
      }

      if (time) {
        await this.extendedInformation(context, printer, section, moduleID, apiId, new Date(from).getTime(), new Date(to).getTime(), sanitizedFilters, indexPatternTitle, agents);
      }

      printer.addVisualizations(array, agents, moduleID);

      if (tables) {
        printer.addTables(tables);
      } //add authorized agents


      if (agentsFilter) {
        printer.addAgentsFilters(agentsFilter);
      }

      await printer.print(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID, name));
      return response.ok({
        body: {
          success: true,
          message: `Report ${name} was created`
        }
      });
    } catch (error) {
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
    }
  }
  /**
   * Create a report for the groups
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {*} reports list or ErrorResponse
   */


  async createReportsGroups(context, request, response) {
    try {
      (0, _logger.log)('reporting:createReportsGroups', `Report started`, 'info');
      const {
        name,
        components,
        apiId
      } = request.body;
      const {
        groupID
      } = request.params; // Init

      const printer = new _printer.ReportPrinter();
      const {
        username: userID
      } = await context.wazuh.security.getCurrentUser(request, context);
      (0, _filesystem.createDataDirectoryIfNotExists)();
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID));
      let tables = [];
      const equivalences = {
        localfile: 'Local files',
        osquery: 'Osquery',
        command: 'Command',
        syscheck: 'Syscheck',
        'open-scap': 'OpenSCAP',
        'cis-cat': 'CIS-CAT',
        syscollector: 'Syscollector',
        rootcheck: 'Rootcheck',
        labels: 'Labels',
        sca: 'Security configuration assessment'
      };
      printer.addContent({
        text: `Group ${groupID} configuration`,
        style: 'h1'
      });

      if (components['0']) {
        let configuration = {};

        try {
          const configurationResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${groupID}/configuration`, {}, {
            apiHostID: apiId
          });
          configuration = configurationResponse.data.data;
        } catch (error) {
          (0, _logger.log)('reporting:createReportsGroups', error.message || error, 'debug');
        }

        if (configuration.affected_items.length > 0 && Object.keys(configuration.affected_items[0].config).length) {
          printer.addContent({
            text: 'Configurations',
            style: {
              fontSize: 14,
              color: '#000'
            },
            margin: [0, 10, 0, 15]
          });
          const section = {
            labels: [],
            isGroupConfig: true
          };

          for (let config of configuration.affected_items) {
            let filterTitle = '';
            let index = 0;

            for (let filter of Object.keys(config.filters)) {
              filterTitle = filterTitle.concat(`${filter}: ${config.filters[filter]}`);

              if (index < Object.keys(config.filters).length - 1) {
                filterTitle = filterTitle.concat(' | ');
              }

              index++;
            }

            printer.addContent({
              text: filterTitle,
              style: 'h4',
              margin: [0, 0, 0, 10]
            });
            let idx = 0;
            section.tabs = [];

            for (let _d of Object.keys(config.config)) {
              for (let c of _agentConfiguration.AgentConfiguration.configurations) {
                for (let s of c.sections) {
                  section.opts = s.opts || {};

                  for (let cn of s.config || []) {
                    if (cn.configuration === _d) {
                      section.labels = s.labels || [[]];
                    }
                  }

                  for (let wo of s.wodle || []) {
                    if (wo.name === _d) {
                      section.labels = s.labels || [[]];
                    }
                  }
                }
              }

              section.labels[0]['pack'] = 'Packs';
              section.labels[0]['content'] = 'Evaluations';
              section.labels[0]['7'] = 'Scan listening netwotk ports';
              section.tabs.push(equivalences[_d]);

              if (Array.isArray(config.config[_d])) {
                /* LOG COLLECTOR */
                if (_d === 'localfile') {
                  let groups = [];

                  config.config[_d].forEach(obj => {
                    if (!groups[obj.logformat]) {
                      groups[obj.logformat] = [];
                    }

                    groups[obj.logformat].push(obj);
                  });

                  Object.keys(groups).forEach(group => {
                    let saveidx = 0;
                    groups[group].forEach((x, i) => {
                      if (Object.keys(x).length > Object.keys(groups[group][saveidx]).length) {
                        saveidx = i;
                      }
                    });
                    const columns = Object.keys(groups[group][saveidx]);
                    const rows = groups[group].map(x => {
                      let row = [];
                      columns.forEach(key => {
                        row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
                          return x + '\n';
                        }) : JSON.stringify(x[key]));
                      });
                      return row;
                    });
                    columns.forEach((col, i) => {
                      columns[i] = col[0].toUpperCase() + col.slice(1);
                    });
                    tables.push({
                      title: 'Local files',
                      type: 'table',
                      columns,
                      rows
                    });
                  });
                } else if (_d === 'labels') {
                  const obj = config.config[_d][0].label;
                  const columns = Object.keys(obj[0]);

                  if (!columns.includes('hidden')) {
                    columns.push('hidden');
                  }

                  const rows = obj.map(x => {
                    let row = [];
                    columns.forEach(key => {
                      row.push(x[key]);
                    });
                    return row;
                  });
                  columns.forEach((col, i) => {
                    columns[i] = col[0].toUpperCase() + col.slice(1);
                  });
                  tables.push({
                    title: 'Labels',
                    type: 'table',
                    columns,
                    rows
                  });
                } else {
                  for (let _d2 of config.config[_d]) {
                    tables.push(...this.getConfigTables(_d2, section, idx));
                  }
                }
              } else {
                /*INTEGRITY MONITORING MONITORED DIRECTORIES */
                if (config.config[_d].directories) {
                  const directories = config.config[_d].directories;
                  delete config.config[_d].directories;
                  tables.push(...this.getConfigTables(config.config[_d], section, idx));
                  let diffOpts = [];
                  Object.keys(section.opts).forEach(x => {
                    diffOpts.push(x);
                  });
                  const columns = ['', ...diffOpts.filter(x => x !== 'check_all' && x !== 'check_sum')];
                  let rows = [];
                  directories.forEach(x => {
                    let row = [];
                    row.push(x.path);
                    columns.forEach(y => {
                      if (y !== '') {
                        y = y !== 'check_whodata' ? y : 'whodata';
                        row.push(x[y] ? x[y] : 'no');
                      }
                    });
                    row.push(x.recursion_level);
                    rows.push(row);
                  });
                  columns.forEach((x, idx) => {
                    columns[idx] = section.opts[x];
                  });
                  columns.push('RL');
                  tables.push({
                    title: 'Monitored directories',
                    type: 'table',
                    columns,
                    rows
                  });
                } else {
                  tables.push(...this.getConfigTables(config.config[_d], section, idx));
                }
              }

              for (const table of tables) {
                printer.addConfigTables([table]);
              }

              idx++;
              tables = [];
            }

            tables = [];
          }
        } else {
          printer.addContent({
            text: 'A configuration for this group has not yet been set up.',
            style: {
              fontSize: 12,
              color: '#000'
            },
            margin: [0, 10, 0, 15]
          });
        }
      }

      if (components['1']) {
        let agentsInGroup = [];

        try {
          const agentsInGroupResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${groupID}/agents`, {}, {
            apiHostID: apiId
          });
          agentsInGroup = agentsInGroupResponse.data.data.affected_items;
        } catch (error) {
          (0, _logger.log)('reporting:report', error.message || error, 'debug');
        }

        await this.renderHeader(context, printer, 'groupConfig', groupID, (agentsInGroup || []).map(x => x.id), apiId);
      }

      await printer.print(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID, name));
      return response.ok({
        body: {
          success: true,
          message: `Report ${name} was created`
        }
      });
    } catch (error) {
      (0, _logger.log)('reporting:createReportsGroups', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
    }
  }
  /**
   * Create a report for the agents
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {*} reports list or ErrorResponse
   */


  async createReportsAgents(context, request, response) {
    try {
      (0, _logger.log)('reporting:createReportsAgents', `Report started`, 'info');
      const {
        name,
        components,
        apiId
      } = request.body;
      const {
        agentID
      } = request.params;
      const printer = new _printer.ReportPrinter();
      const {
        username: userID
      } = await context.wazuh.security.getCurrentUser(request, context);
      (0, _filesystem.createDataDirectoryIfNotExists)();
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID));
      let wmodulesResponse = {};
      let tables = [];

      try {
        wmodulesResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents/${agentID}/config/wmodules/wmodules`, {}, {
          apiHostID: apiId
        });
      } catch (error) {
        (0, _logger.log)('reporting:report', error.message || error, 'debug');
      }

      await this.renderHeader(context, printer, 'agentConfig', 'agentConfig', agentID, apiId);
      let idxComponent = 0;

      for (let config of _agentConfiguration.AgentConfiguration.configurations) {
        let titleOfSection = false;
        (0, _logger.log)('reporting:createReportsAgents', `Iterate over ${config.sections.length} configuration sections`, 'debug');

        for (let section of config.sections) {
          let titleOfSubsection = false;

          if (components[idxComponent] && (section.config || section.wodle)) {
            let idx = 0;
            const configs = (section.config || []).concat(section.wodle || []);
            (0, _logger.log)('reporting:createReportsAgents', `Iterate over ${configs.length} configuration blocks`, 'debug');

            for (let conf of configs) {
              let agentConfigResponse = {};

              try {
                if (!conf['name']) {
                  agentConfigResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents/${agentID}/config/${conf.component}/${conf.configuration}`, {}, {
                    apiHostID: apiId
                  });
                } else {
                  for (let wodle of wmodulesResponse.data.data['wmodules']) {
                    if (Object.keys(wodle)[0] === conf['name']) {
                      agentConfigResponse.data = {
                        data: wodle
                      };
                    }
                  }
                }

                const agentConfig = agentConfigResponse && agentConfigResponse.data && agentConfigResponse.data.data;

                if (!titleOfSection) {
                  printer.addContent({
                    text: config.title,
                    style: 'h1',
                    margin: [0, 0, 0, 15]
                  });
                  titleOfSection = true;
                }

                if (!titleOfSubsection) {
                  printer.addContent({
                    text: section.subtitle,
                    style: 'h4'
                  });
                  printer.addContent({
                    text: section.desc,
                    style: {
                      fontSize: 12,
                      color: '#000'
                    },
                    margin: [0, 0, 0, 10]
                  });
                  titleOfSubsection = true;
                }

                if (agentConfig) {
                  for (let agentConfigKey of Object.keys(agentConfig)) {
                    if (Array.isArray(agentConfig[agentConfigKey])) {
                      /* LOG COLLECTOR */
                      if (conf.filterBy) {
                        let groups = [];
                        agentConfig[agentConfigKey].forEach(obj => {
                          if (!groups[obj.logformat]) {
                            groups[obj.logformat] = [];
                          }

                          groups[obj.logformat].push(obj);
                        });
                        Object.keys(groups).forEach(group => {
                          let saveidx = 0;
                          groups[group].forEach((x, i) => {
                            if (Object.keys(x).length > Object.keys(groups[group][saveidx]).length) {
                              saveidx = i;
                            }
                          });
                          const columns = Object.keys(groups[group][saveidx]);
                          const rows = groups[group].map(x => {
                            let row = [];
                            columns.forEach(key => {
                              row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
                                return x + '\n';
                              }) : JSON.stringify(x[key]));
                            });
                            return row;
                          });
                          columns.forEach((col, i) => {
                            columns[i] = col[0].toUpperCase() + col.slice(1);
                          });
                          tables.push({
                            title: section.labels[0][group],
                            type: 'table',
                            columns,
                            rows
                          });
                        });
                      } else if (agentConfigKey.configuration !== 'socket') {
                        tables.push(...this.getConfigTables(agentConfig[agentConfigKey], section, idx));
                      } else {
                        for (let _d2 of agentConfig[agentConfigKey]) {
                          tables.push(...this.getConfigTables(_d2, section, idx));
                        }
                      }
                    } else {
                      /*INTEGRITY MONITORING MONITORED DIRECTORIES */
                      if (conf.matrix) {
                        const {
                          directories,
                          diff,
                          synchronization,
                          file_limit,
                          ...rest
                        } = agentConfig[agentConfigKey];
                        tables.push(...this.getConfigTables(rest, section, idx), ...(diff && diff.disk_quota ? this.getConfigTables(diff.disk_quota, {
                          tabs: ['Disk quota']
                        }, 0) : []), ...(diff && diff.file_size ? this.getConfigTables(diff.file_size, {
                          tabs: ['File size']
                        }, 0) : []), ...(synchronization ? this.getConfigTables(synchronization, {
                          tabs: ['Synchronization']
                        }, 0) : []), ...(file_limit ? this.getConfigTables(file_limit, {
                          tabs: ['File limit']
                        }, 0) : []));
                        let diffOpts = [];
                        Object.keys(section.opts).forEach(x => {
                          diffOpts.push(x);
                        });
                        const columns = ['', ...diffOpts.filter(x => x !== 'check_all' && x !== 'check_sum')];
                        let rows = [];
                        directories.forEach(x => {
                          let row = [];
                          row.push(x.dir);
                          columns.forEach(y => {
                            if (y !== '') {
                              row.push(x.opts.indexOf(y) > -1 ? 'yes' : 'no');
                            }
                          });
                          row.push(x.recursion_level);
                          rows.push(row);
                        });
                        columns.forEach((x, idx) => {
                          columns[idx] = section.opts[x];
                        });
                        columns.push('RL');
                        tables.push({
                          title: 'Monitored directories',
                          type: 'table',
                          columns,
                          rows
                        });
                      } else {
                        tables.push(...this.getConfigTables(agentConfig[agentConfigKey], section, idx));
                      }
                    }
                  }
                } else {
                  // Print no configured module and link to the documentation
                  printer.addContent({
                    text: ['This module is not configured. Please take a look on how to configure it in ', {
                      text: `${section.subtitle.toLowerCase()} configuration.`,
                      link: section.docuLink,
                      style: {
                        fontSize: 12,
                        color: '#1a0dab'
                      }
                    }],
                    margin: [0, 0, 0, 20]
                  });
                }
              } catch (error) {
                (0, _logger.log)('reporting:report', error.message || error, 'debug');
              }

              idx++;
            }

            for (const table of tables) {
              printer.addConfigTables([table]);
            }
          }

          idxComponent++;
          tables = [];
        }
      }

      await printer.print(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID, name));
      return response.ok({
        body: {
          success: true,
          message: `Report ${name} was created`
        }
      });
    } catch (error) {
      (0, _logger.log)('reporting:createReportsAgents', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
    }
  }
  /**
   * Create a report for the agents
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {*} reports list or ErrorResponse
   */


  async createReportsAgentsInventory(context, request, response) {
    try {
      (0, _logger.log)('reporting:createReportsAgentsInventory', `Report started`, 'info');
      const {
        searchBar,
        filters,
        time,
        name,
        indexPatternTitle,
        apiId
      } = request.body;
      const {
        agentID
      } = request.params;
      const {
        from,
        to
      } = time || {}; // Init

      const printer = new _printer.ReportPrinter();
      const {
        username: userID
      } = await context.wazuh.security.getCurrentUser(request, context);
      (0, _filesystem.createDataDirectoryIfNotExists)();
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID));
      (0, _logger.log)('reporting:createReportsAgentsInventory', `Syscollector report`, 'debug');
      const sanitizedFilters = filters ? this.sanitizeKibanaFilters(filters, searchBar) : false; // Get the agent OS

      let agentOs = '';

      try {
        const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', '/agents', {
          params: {
            q: `id=${agentID}`
          }
        }, {
          apiHostID: apiId
        });
        agentOs = agentResponse.data.data.affected_items[0].os.platform;
      } catch (error) {
        (0, _logger.log)('reporting:createReportsAgentsInventory', error.message || error, 'debug');
      } // Add title


      printer.addContentWithNewLine({
        text: 'Inventory data report',
        style: 'h1'
      }); // Add table with the agent info

      await this.buildAgentsTable(context, printer, [agentID], apiId); // Get syscollector packages and processes

      const agentRequestsInventory = [{
        endpoint: `/syscollector/${agentID}/packages`,
        loggerMessage: `Fetching packages for agent ${agentID}`,
        table: {
          title: 'Packages',
          columns: agentOs === 'windows' ? [{
            id: 'name',
            label: 'Name'
          }, {
            id: 'architecture',
            label: 'Architecture'
          }, {
            id: 'version',
            label: 'Version'
          }, {
            id: 'vendor',
            label: 'Vendor'
          }] : [{
            id: 'name',
            label: 'Name'
          }, {
            id: 'architecture',
            label: 'Architecture'
          }, {
            id: 'version',
            label: 'Version'
          }, {
            id: 'vendor',
            label: 'Vendor'
          }, {
            id: 'description',
            label: 'Description'
          }]
        }
      }, {
        endpoint: `/syscollector/${agentID}/processes`,
        loggerMessage: `Fetching processes for agent ${agentID}`,
        table: {
          title: 'Processes',
          columns: agentOs === 'windows' ? [{
            id: 'name',
            label: 'Name'
          }, {
            id: 'cmd',
            label: 'CMD'
          }, {
            id: 'priority',
            label: 'Priority'
          }, {
            id: 'nlwp',
            label: 'NLWP'
          }] : [{
            id: 'name',
            label: 'Name'
          }, {
            id: 'euser',
            label: 'Effective user'
          }, {
            id: 'nice',
            label: 'Priority'
          }, {
            id: 'state',
            label: 'State'
          }]
        },
        mapResponseItems: item => agentOs === 'windows' ? item : { ...item,
          state: _processStateEquivalence.default[item.state]
        }
      }, {
        endpoint: `/syscollector/${agentID}/ports`,
        loggerMessage: `Fetching ports for agent ${agentID}`,
        table: {
          title: 'Network ports',
          columns: agentOs === 'windows' ? [{
            id: 'local_ip',
            label: 'Local IP'
          }, {
            id: 'local_port',
            label: 'Local port'
          }, {
            id: 'process',
            label: 'Process'
          }, {
            id: 'state',
            label: 'State'
          }, {
            id: 'protocol',
            label: 'Protocol'
          }] : [{
            id: 'local_ip',
            label: 'Local IP'
          }, {
            id: 'local_port',
            label: 'Local port'
          }, {
            id: 'state',
            label: 'State'
          }, {
            id: 'protocol',
            label: 'Protocol'
          }]
        },
        mapResponseItems: item => ({ ...item,
          local_ip: item.local.ip,
          local_port: item.local.port
        })
      }, {
        endpoint: `/syscollector/${agentID}/netiface`,
        loggerMessage: `Fetching netiface for agent ${agentID}`,
        table: {
          title: 'Network interfaces',
          columns: [{
            id: 'name',
            label: 'Name'
          }, {
            id: 'mac',
            label: 'Mac'
          }, {
            id: 'state',
            label: 'State'
          }, {
            id: 'mtu',
            label: 'MTU'
          }, {
            id: 'type',
            label: 'Type'
          }]
        }
      }, {
        endpoint: `/syscollector/${agentID}/netaddr`,
        loggerMessage: `Fetching netaddr for agent ${agentID}`,
        table: {
          title: 'Network settings',
          columns: [{
            id: 'iface',
            label: 'Interface'
          }, {
            id: 'address',
            label: 'address'
          }, {
            id: 'netmask',
            label: 'Netmask'
          }, {
            id: 'proto',
            label: 'Protocol'
          }, {
            id: 'broadcast',
            label: 'Broadcast'
          }]
        }
      }];
      agentOs === 'windows' && agentRequestsInventory.push({
        endpoint: `/syscollector/${agentID}/hotfixes`,
        loggerMessage: `Fetching hotfixes for agent ${agentID}`,
        table: {
          title: 'Windows updates',
          columns: [{
            id: 'hotfix',
            label: 'Update code'
          }]
        }
      });

      const requestInventory = async agentRequestInventory => {
        try {
          (0, _logger.log)('reporting:createReportsAgentsInventory', agentRequestInventory.loggerMessage, 'debug');
          const inventoryResponse = await context.wazuh.api.client.asCurrentUser.request('GET', agentRequestInventory.endpoint, {}, {
            apiHostID: apiId
          });
          const inventory = inventoryResponse && inventoryResponse.data && inventoryResponse.data.data && inventoryResponse.data.data.affected_items;

          if (inventory) {
            return { ...agentRequestInventory.table,
              items: agentRequestInventory.mapResponseItems ? inventory.map(agentRequestInventory.mapResponseItems) : inventory
            };
          }
        } catch (error) {
          (0, _logger.log)('reporting:createReportsAgentsInventory', error.message || error, 'debug');
        }
      };

      if (time) {
        await this.extendedInformation(context, printer, 'agents', 'syscollector', apiId, from, to, sanitizedFilters + ' AND rule.groups: "vulnerability-detector"', indexPatternTitle, agentID);
      } // Add inventory tables


      (await Promise.all(agentRequestsInventory.map(requestInventory))).filter(table => table).forEach(table => printer.addSimpleTable(table)); // Print the document

      await printer.print(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID, name));
      return response.ok({
        body: {
          success: true,
          message: `Report ${name} was created`
        }
      });
    } catch (error) {
      (0, _logger.log)('reporting:createReportsAgents', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
    }
  }
  /**
   * Fetch the reports list
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Array<Object>} reports list or ErrorResponse
   */


  async getReports(context, request, response) {
    try {
      (0, _logger.log)('reporting:getReports', `Fetching created reports`, 'info');
      const {
        username: userID
      } = await context.wazuh.security.getCurrentUser(request, context);
      (0, _filesystem.createDataDirectoryIfNotExists)();
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);

      const userReportsDirectory = _path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID);

      (0, _filesystem.createDirectoryIfNotExists)(userReportsDirectory);
      (0, _logger.log)('reporting:getReports', `Directory: ${userReportsDirectory}`, 'debug');

      const sortReportsByDate = (a, b) => a.date < b.date ? 1 : a.date > b.date ? -1 : 0;

      const reports = _fs.default.readdirSync(userReportsDirectory).map(file => {
        const stats = _fs.default.statSync(userReportsDirectory + '/' + file); // Get the file creation time (bithtime). It returns the first value that is a truthy value of next file stats: birthtime, mtime, ctime and atime.
        // This solves some OSs can have the bithtimeMs equal to 0 and returns the date like 1970-01-01


        const birthTimeField = ['birthtime', 'mtime', 'ctime', 'atime'].find(time => stats[`${time}Ms`]);
        return {
          name: file,
          size: stats.size,
          date: stats[birthTimeField]
        };
      });

      (0, _logger.log)('reporting:getReports', `Using TimSort for sorting ${reports.length} items`, 'debug');
      TimSort.sort(reports, sortReportsByDate);
      (0, _logger.log)('reporting:getReports', `Total reports: ${reports.length}`, 'debug');
      return response.ok({
        body: {
          reports
        }
      });
    } catch (error) {
      (0, _logger.log)('reporting:getReports', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5031, 500, response);
    }
  }
  /**
   * Fetch specific report
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} report or ErrorResponse
   */


  async getReportByName(context, request, response) {
    try {
      (0, _logger.log)('reporting:getReportByName', `Getting ${request.params.name} report`, 'debug');
      const {
        username: userID
      } = await context.wazuh.security.getCurrentUser(request, context);

      const reportFileBuffer = _fs.default.readFileSync(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID, request.params.name));

      return response.ok({
        headers: {
          'Content-Type': 'application/pdf'
        },
        body: reportFileBuffer
      });
    } catch (error) {
      (0, _logger.log)('reporting:getReportByName', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5030, 500, response);
    }
  }
  /**
   * Delete specific report
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} status obj or ErrorResponse
   */


  async deleteReportByName(context, request, response) {
    try {
      (0, _logger.log)('reporting:deleteReportByName', `Deleting ${request.params.name} report`, 'debug');
      const {
        username: userID
      } = await context.wazuh.security.getCurrentUser(request, context);

      _fs.default.unlinkSync(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, userID, request.params.name));

      (0, _logger.log)('reporting:deleteReportByName', `${request.params.name} report was deleted`, 'info');
      return response.ok({
        body: {
          error: 0
        }
      });
    } catch (error) {
      (0, _logger.log)('reporting:deleteReportByName', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5032, 500, response);
    }
  }

}

exports.WazuhReportingCtrl = WazuhReportingCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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