"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ErrorResponse = ErrorResponse;

/*
 * Wazuh app - Generic error response constructor
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */

/**
 * Error codes:
 * wazuh-api-elastic 20XX
 * wazuh-api         30XX
 * wazuh-elastic     40XX
 * wazuh-reporting   50XX
 * unknown           1000
 */

/**
 * Returns a suitable error message
 * @param {String} message Error message
 * @param {Number} code Error code
 * @param {Number} statusCode Error status code
 * @returns {Object} Error response object
 */
function ErrorResponse(message = null, code = null, statusCode = null, response) {
  message.includes('password: ') ? message = message.split('password: ')[0] + ' password: ***' : false;
  let filteredMessage = '';

  if (code) {
    const isString = typeof message === 'string';

    if (isString && message === 'socket hang up' && code === 3005) {
      filteredMessage = 'Wrong protocol being used to connect to the Wazuh API';
    } else if (isString && (message.includes('ENOTFOUND') || message.includes('EHOSTUNREACH') || message.includes('EINVAL') || message.includes('EAI_AGAIN')) && code === 3005) {
      filteredMessage = 'Wazuh API is not reachable. Please check your url and port.';
    } else if (isString && message.includes('ECONNREFUSED') && code === 3005) {
      filteredMessage = 'Wazuh API is not reachable. Please check your url and port.';
    } else if (isString && message.toLowerCase().includes('not found') && code === 3002) {
      filteredMessage = 'It seems the selected API was deleted.';
    } else if (isString && message.includes('ENOENT') && message.toLowerCase().includes('no such file or directory') && message.toLowerCase().includes('data') && code === 5029) {
      filteredMessage = 'Reporting was aborted';
    } else if (isString && code === 5029) {
      filteredMessage = `Reporting was aborted (${message})`;
    }
  }

  const statusCodeResponse = statusCode || 500;
  return response.custom({
    statusCode: statusCodeResponse,
    body: {
      message: filteredMessage ? `${code || 1000} - ${filteredMessage}` : typeof message === 'string' ? `${code || 1000} - ${message}` : `${code || 1000} - Unexpected error`,
      code: code || 1000,
      statusCode: statusCodeResponse
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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