"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SaveDocument = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _getConfiguration = require("../../lib/get-configuration");

var _logger = require("../../lib/logger");

var _indexDate = require("../../lib/index-date");

var _constants = require("../../../common/constants");

var _tryCatchForIndexPermissionError = require("../tryCatchForIndexPermissionError");

class SaveDocument {
  constructor(context) {
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "esClientInternalUser", void 0);
    (0, _defineProperty2.default)(this, "logPath", 'cron-scheduler|SaveDocument');
    this.context = context;
    this.esClientInternalUser = context.core.elasticsearch.client.asInternalUser;
  }

  async save(doc, indexConfig) {
    const {
      name,
      creation,
      mapping,
      shards,
      replicas
    } = indexConfig;
    const index = this.addIndexPrefix(name);
    const indexCreation = `${index}-${(0, _indexDate.indexDate)(creation)}`;

    try {
      await this.checkIndexAndCreateIfNotExists(indexCreation, shards, replicas);
      const createDocumentObject = this.createDocument(doc, indexCreation, mapping);
      const response = await this.esClientInternalUser.bulk(createDocumentObject);
      (0, _logger.log)(this.logPath, `Response of create new document ${JSON.stringify(response)}`, 'debug'); // await this.checkIndexPatternAndCreateIfNotExists(index);
    } catch (error) {
      if (error.status === 403) throw {
        error: 403,
        message: `Authorization Exception in the index "${index}"`
      };
      if (error.status === 409) throw {
        error: 409,
        message: `Duplicate index-pattern: ${index}`
      };
      throw error;
    }
  }

  async checkIndexAndCreateIfNotExists(index, shards, replicas) {
    try {
      await (0, _tryCatchForIndexPermissionError.tryCatchForIndexPermissionError)(index)(async () => {
        const exists = await this.esClientInternalUser.indices.exists({
          index
        });
        (0, _logger.log)(this.logPath, `Index '${index}' exists? ${exists.body}`, 'debug');

        if (!exists.body) {
          const response = await this.esClientInternalUser.indices.create({
            index,
            body: {
              settings: {
                index: {
                  number_of_shards: shards !== null && shards !== void 0 ? shards : _constants.WAZUH_STATISTICS_DEFAULT_INDICES_SHARDS,
                  number_of_replicas: replicas !== null && replicas !== void 0 ? replicas : _constants.WAZUH_STATISTICS_DEFAULT_INDICES_REPLICAS
                }
              }
            }
          });
          (0, _logger.log)(this.logPath, `Status of create a new index: ${JSON.stringify(response)}`, 'debug');
        }
      })();
    } catch (error) {
      (0, _logger.log)(this.logPath, error.message || error);
      this.checkDuplicateIndexError(error);
    }
  }

  checkDuplicateIndexError(error) {
    const {
      type
    } = ((error || {}).body || {}).error || {};
    if (!['resource_already_exists_exception'].includes(type)) throw error;
  }

  createDocument(doc, index, mapping) {
    const createDocumentObject = {
      index,
      type: '_doc',
      body: doc.flatMap(item => [{
        index: {
          _index: index
        }
      }, { ...this.buildData(item, mapping),
        timestamp: new Date(Date.now()).toISOString()
      }])
    };
    (0, _logger.log)(this.logPath, `Document object: ${JSON.stringify(createDocumentObject)}`, 'debug');
    return createDocumentObject;
  }

  buildData(item, mapping) {
    const getItemArray = (array, index) => {
      return JSON.stringify(array[index || 0]);
    };

    const getValue = (key, item) => {
      const keys = key.split('.');

      if (keys.length === 1) {
        if (key.match(/\[.*\]/)) {
          return getItemArray(item[key.replace(/\[.*\]/, '')], key.match(/\[(.*)\]/)[1]);
        }

        return JSON.stringify(item[key]);
      }

      return getValue(keys.slice(1).join('.'), item[keys[0]]);
    };

    if (mapping) {
      let data;
      data = mapping.replace(/\${([a-z|A-Z|0-9|\.\-\_\[.*\]]+)}/gi, (...key) => getValue(key[1], item));
      return JSON.parse(data);
    }

    if (typeof item.data === 'object') {
      return item.data;
    }

    return {
      data: item.data
    };
  }

  addIndexPrefix(index) {
    const configFile = (0, _getConfiguration.getConfiguration)();
    const prefix = configFile['cron.prefix'] || 'wazuh';
    return `${prefix}-${index}`;
  }

}

exports.SaveDocument = SaveDocument;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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