"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhHostsCtrl = void 0;

var _manageHosts = require("../lib/manage-hosts");

var _updateRegistry = require("../lib/update-registry");

var _logger = require("../lib/logger");

var _errorResponse = require("../lib/error-response");

var _cacheApiUserHasRunAs = require("../lib/cache-api-user-has-run-as");

var _constants = require("../../common/constants");

/*
 * Wazuh app - Class for Wazuh-API functions
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
class WazuhHostsCtrl {
  constructor() {
    this.manageHosts = new _manageHosts.ManageHosts();
    this.updateRegistry = new _updateRegistry.UpdateRegistry();
  }
  /**
   * This get all hosts entries in the wazuh.yml and the related info in the wazuh-registry.json
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * API entries or ErrorResponse
   */


  async getHostsEntries(context, request, response) {
    try {
      const removePassword = true;
      const hosts = await this.manageHosts.getHosts(removePassword);
      const registry = await this.updateRegistry.getHosts();
      const result = await this.joinHostRegistry(hosts, registry, removePassword);
      return response.ok({
        body: result
      });
    } catch (error) {
      if (error && error.message && ['ENOENT: no such file or directory', _constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH].every(text => error.message.includes(text))) {
        return response.badRequest({
          body: {
            message: `Error getting the hosts entries: The \'${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH}\' directory could not exist in your ${_constants.PLUGIN_PLATFORM_NAME} installation.
            If this doesn't exist, create it and give the permissions 'sudo mkdir ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH};sudo chown -R ${_constants.PLUGIN_PLATFORM_INSTALLATION_USER}:${_constants.PLUGIN_PLATFORM_INSTALLATION_USER_GROUP} ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH}'. After, restart the ${_constants.PLUGIN_PLATFORM_NAME} service.`
          }
        });
      }

      (0, _logger.log)('wazuh-hosts:getHostsEntries', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 2001, 500, response);
    }
  }
  /**
   * Joins the hosts with the related information in the registry
   * @param {Object} hosts
   * @param {Object} registry
   * @param {Boolean} removePassword
   */


  async joinHostRegistry(hosts, registry, removePassword = true) {
    try {
      if (!Array.isArray(hosts)) {
        throw new Error('Hosts configuration error in wazuh.yml');
      }

      return await Promise.all(hosts.map(async h => {
        const id = Object.keys(h)[0];
        const api = Object.assign(h[id], {
          id: id
        });
        const host = Object.assign(api, registry[id]); // Add to run_as from API user. Use the cached value or get it doing a request

        host.allow_run_as = await _cacheApiUserHasRunAs.APIUserAllowRunAs.check(id);

        if (removePassword) {
          delete host.password;
          delete host.token;
        }

        ;
        return host;
      }));
    } catch (error) {
      throw new Error(error);
    }
  }
  /**
   * This update an API hostname
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * Status response or ErrorResponse
   */


  async updateClusterInfo(context, request, response) {
    try {
      const {
        id
      } = request.params;
      const {
        cluster_info
      } = request.body;
      await this.updateRegistry.updateClusterInfo(id, cluster_info);
      (0, _logger.log)('wazuh-hosts:updateClusterInfo', `API entry ${id} hostname updated`, 'debug');
      return response.ok({
        body: {
          statusCode: 200,
          message: 'ok'
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-hosts:updateClusterInfo', error.message || error);
      return (0, _errorResponse.ErrorResponse)(`Could not update data in wazuh-registry.json due to ${error.message || error}`, 2012, 500, response);
    }
  }
  /**
   * Remove the orphan host entries in the registry
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   */


  async removeOrphanEntries(context, request, response) {
    try {
      const {
        entries
      } = request.body;
      (0, _logger.log)('wazuh-hosts:cleanRegistry', 'Cleaning registry', 'debug');
      await this.updateRegistry.removeOrphanEntries(entries);
      return response.ok({
        body: {
          statusCode: 200,
          message: 'ok'
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-hosts:cleanRegistry', error.message || error);
      return (0, _errorResponse.ErrorResponse)(`Could not clean entries in the wazuh-registry.json due to ${error.message || error}`, 2013, 500, response);
    }
  }

}

exports.WazuhHostsCtrl = WazuhHostsCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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