"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhUtilsCtrl = void 0;

var _errorResponse = require("../../lib/error-response");

var _getConfiguration = require("../../lib/get-configuration");

var _readLastLines = require("read-last-lines");

var _updateConfiguration = require("../../lib/update-configuration");

var _jwtDecode = _interopRequireDefault(require("jwt-decode"));

var _constants = require("../../../common/constants");

var _manageHosts = require("../../lib/manage-hosts");

var _cookie = require("../../lib/cookie");

/*
 * Wazuh app - Class for Wazuh-API functions
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
// Require some libraries
const updateConfigurationFile = new _updateConfiguration.UpdateConfigurationFile();

class WazuhUtilsCtrl {
  /**
   * Constructor
   * @param {*} server
   */
  constructor() {
    this.manageHosts = new _manageHosts.ManageHosts();
  }
  /**
   * Returns the wazuh.yml file parsed
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} Configuration File or ErrorResponse
   */


  getConfigurationFile(context, request, response) {
    try {
      const configFile = (0, _getConfiguration.getConfiguration)();
      return response.ok({
        body: {
          statusCode: 200,
          error: 0,
          data: configFile || {}
        }
      });
    } catch (error) {
      return (0, _errorResponse.ErrorResponse)(error.message || error, 3019, 500, response);
    }
  }
  /**
   * Returns the wazuh.yml file in raw
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} Configuration File or ErrorResponse
   */


  async updateConfigurationFile(context, request, response) {
    try {
      // Check if user has administrator role in token
      const token = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-token');

      if (!token) {
        return (0, _errorResponse.ErrorResponse)('No token provided', 401, 401, response);
      }

      ;
      const decodedToken = (0, _jwtDecode.default)(token);

      if (!decodedToken) {
        return (0, _errorResponse.ErrorResponse)('No permissions in token', 401, 401, response);
      }

      ;

      if (!decodedToken.rbac_roles || !decodedToken.rbac_roles.includes(_constants.WAZUH_ROLE_ADMINISTRATOR_ID)) {
        return (0, _errorResponse.ErrorResponse)('No administrator role', 401, 401, response);
      }

      ;
      response; // Check the provided token is valid

      const apiHostID = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-api');

      if (!apiHostID) {
        return (0, _errorResponse.ErrorResponse)('No API id provided', 401, 401, response);
      }

      ;
      const responseTokenIsWorking = await context.wazuh.api.client.asCurrentUser.request('GET', '/', {}, {
        apiHostID
      });

      if (responseTokenIsWorking.status !== 200) {
        return (0, _errorResponse.ErrorResponse)('Token is not valid', 401, 401, response);
      }

      ;
      const result = await updateConfigurationFile.updateConfiguration(request);
      return response.ok({
        body: {
          statusCode: 200,
          error: 0,
          data: result
        }
      });
    } catch (error) {
      return (0, _errorResponse.ErrorResponse)(error.message || error, 3021, 500, response);
    }
  }
  /**
   * Returns Wazuh app logs
   * @param {Object} context 
   * @param {Object} request
   * @param {Object} response
   * @returns {Array<String>} app logs or ErrorResponse
   */


  async getAppLogs(context, request, response) {
    try {
      const lastLogs = await (0, _readLastLines.read)(_constants.WAZUH_DATA_LOGS_RAW_PATH, 50);
      const spliterLog = lastLogs.split('\n');
      return spliterLog && Array.isArray(spliterLog) ? response.ok({
        body: {
          error: 0,
          lastLogs: spliterLog.filter(item => typeof item === 'string' && item.length)
        }
      }) : response.ok({
        error: 0,
        lastLogs: []
      });
    } catch (error) {
      return (0, _errorResponse.ErrorResponse)(error.message || error, 3036, 500, response);
    }
  }

}

exports.WazuhUtilsCtrl = WazuhUtilsCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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