"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.AgentConfiguration = void 0;

/*
 * Wazuh app - Agent configuration request objet for exporting it
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
const AgentConfiguration = {
  configurations: [{
    title: 'Main configurations',
    sections: [{
      subtitle: 'Global configuration',
      desc: 'Logging settings that apply to the agent',
      config: [{
        component: 'com',
        configuration: 'logging'
      }],
      labels: [{
        plain: 'Write internal logs in plain text',
        json: 'Write internal logs in JSON format',
        server: 'List of managers to connect'
      }]
    }, {
      subtitle: 'Communication',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/client.html',
      desc: 'Settings related to the connection with the manager',
      config: [{
        component: 'agent',
        configuration: 'client'
      }],
      labels: [{
        crypto_method: 'Method used to encrypt communications',
        auto_restart: 'Auto-restart the agent when receiving valid configuration from manager',
        notify_time: 'Time (in seconds) between agent checkings to the manager',
        'time-reconnect': 'Time (in seconds) before attempting to reconnect',
        server: 'List of managers to connect',
        'config-profile': 'Configuration profiles',
        remote_conf: 'Remote configuration is enabled'
      }]
    }, {
      subtitle: 'Anti-flooding settings',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/capabilities/antiflooding.html',
      desc: 'Agent bucket parameters to avoid event flooding',
      config: [{
        component: 'agent',
        configuration: 'buffer'
      }],
      labels: [{
        disabled: 'Buffer disabled',
        queue_size: 'Queue size',
        events_per_second: 'Events per second'
      }]
    }, {
      subtitle: 'Labels',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/labels.html',
      desc: 'User-defined information about the agent included in alerts',
      config: [{
        component: 'agent',
        configuration: 'labels'
      }]
    }]
  }, {
    title: 'Auditing and policy monitoring',
    sections: [{
      subtitle: 'Policy monitoring',
      docuLink: 'https://documentation.wazuh.com/current/pci-dss/policy-monitoring.html',
      desc: 'Configuration to ensure compliance with security policies, standards and hardening guides',
      config: [{
        component: 'syscheck',
        configuration: 'rootcheck'
      }],
      wodle: [{
        name: 'sca'
      }],
      labels: [{
        disabled: 'Policy monitoring service disabled',
        base_directory: 'Base directory',
        rootkit_files: 'Rootkit files database path',
        rootkit_trojans: 'Rootkit trojans database path',
        scanall: 'Scan the entire system',
        skip_nfs: 'Skip scan on CIFS/NFS mounts',
        frequency: 'Frequency (in seconds) to run the scan',
        check_dev: 'Check /dev path',
        check_files: 'Check files',
        check_if: 'Check network interfaces',
        check_pids: 'Check processes IDs',
        check_ports: 'Check network ports',
        check_sys: 'Check anomalous system objects',
        check_trojans: 'Check trojans',
        check_unixaudit: 'Check UNIX audit',
        system_audit: 'UNIX audit files paths',
        enabled: 'Security configuration assessment enabled',
        scan_on_start: 'Scan on start',
        interval: 'Interval',
        policies: 'Policies'
      }],
      tabs: ['General', 'Security configuration assessment']
    }, {
      subtitle: 'OpenSCAP',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/wodle-openscap.html',
      desc: 'Configuration assessment and automation of compliance monitoring using SCAP checks',
      wodle: [{
        name: 'open-scap'
      }],
      labels: [{
        content: 'Evaluations',
        disabled: 'OpenSCAP integration disabled',
        'scan-on-start': 'Scan on start',
        interval: 'Interval between scan executions',
        timeout: 'Timeout (in seconds) for scan executions'
      }]
    }, {
      subtitle: 'CIS-CAT',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/wodle-ciscat.html',
      desc: 'Configuration assessment using CIS scanner and SCAP checks',
      wodle: [{
        name: 'cis-cat'
      }],
      labels: [{
        disabled: 'CIS-CAT integration disabled',
        'scan-on-start': 'Scan on start',
        interval: 'Interval between scan executions',
        java_path: 'Path to Java executable directory',
        ciscat_path: 'Path to CIS-CAT executable directory',
        timeout: 'Timeout (in seconds) for scan executions',
        content: 'Benchmarks'
      }]
    }]
  }, {
    title: 'System threats and incident response',
    sections: [{
      subtitle: 'Osquery',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/wodle-osquery.html',
      desc: 'Expose an operating system as a high-performance relational database',
      wodle: [{
        name: 'osquery'
      }],
      labels: [{
        disabled: 'Osquery integration disabled',
        run_daemon: 'Auto-run the Osquery daemon',
        add_labels: 'Use defined labels as decorators',
        log_path: 'Path to the Osquery results log file',
        config_path: 'Path to the Osquery configuration file'
      }]
    }, {
      subtitle: 'Inventory data',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/wodle-syscollector.html',
      desc: 'Gather relevant information about system OS, hardware, networking and packages',
      wodle: [{
        name: 'syscollector'
      }],
      labels: [{
        disabled: 'Syscollector integration disabled',
        'scan-on-start': 'Scan on start',
        interval: 'Interval between system scans',
        network: 'Scan network interfaces',
        os: 'Scan operating system info',
        hardware: 'Scan hardware info',
        packages: 'Scan installed packages',
        ports: 'Scan listening network ports',
        ports_all: 'Scan all network ports',
        processes: 'Scan current processes'
      }]
    }, {
      subtitle: 'Active response',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/active-response.html',
      desc: 'Active threat addressing by immediate response',
      config: [{
        component: 'com',
        configuration: 'active-response'
      }],
      labels: [{
        disabled: 'Active response disabled',
        ca_store: 'Use the following list of root CA certificates',
        ca_verification: 'Validate WPKs using root CA certificate'
      }]
    }, {
      subtitle: 'Commands',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/wodle-command.html',
      desc: 'Configuration options of the Command wodle',
      wodle: [{
        name: 'command'
      }],
      labels: [{
        disabled: 'Command disabled',
        run_on_start: 'Run on start',
        ignore_output: 'Ignore command output',
        skip_verification: 'Ignore checksum verification',
        interval: 'Interval between executions',
        tag: 'Command name',
        command: 'Command to execute',
        verify_md5: 'Verify MD5 sum',
        verify_sha1: 'Verify SHA1 sum',
        verify_sha256: 'Verify SHA256 sum'
      }]
    }, {
      subtitle: 'Docker listener',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/wodle-docker.html',
      desc: 'Monitor and collect the activity from Docker containers such as creation, running, starting, stopping or pausing events',
      wodle: [{
        name: 'docker-listener'
      }],
      labels: [{
        disabled: 'Docker listener disabled',
        run_on_start: 'Run the listener immediately when service is started',
        interval: 'Waiting time to rerun the listener in case it fails',
        attempts: 'Number of attempts to execute the listener'
      }]
    }]
  }, {
    title: 'Log data analysis',
    sections: [{
      subtitle: 'Log collection',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/capabilities/log-data-collection/index.html',
      desc: 'Log analysis from text files, Windows events or syslog outputs',
      config: [{
        component: 'logcollector',
        configuration: 'localfile',
        filterBy: 'logformat'
      }, {
        component: 'logcollector',
        configuration: 'socket'
      }],
      labels: [{
        logformat: 'Log format',
        log_format: 'Log format',
        alias: 'Command alias',
        ignore_binaries: 'Ignore binaries',
        target: 'Redirect output to this socket',
        frequency: 'Interval between command executions',
        file: 'Log location',
        location: 'Log location',
        socket: 'Output sockets',
        syslog: 'Syslog',
        command: 'Command',
        full_command: 'Full command',
        audit: 'Audit'
      }],
      options: {
        hideHeader: true
      }
    }, {
      subtitle: 'Integrity monitoring',
      docuLink: 'https://documentation.wazuh.com/current/user-manual/reference/ossec-conf/syscheck.html',
      desc: 'Identify changes in content, permissions, ownership, and attributes of files',
      config: [{
        component: 'syscheck',
        configuration: 'syscheck',
        matrix: true
      }],
      tabs: ['General', 'Who data'],
      labels: [{
        disabled: 'Integrity monitoring disabled',
        frequency: 'Interval (in seconds) to run the integrity scan',
        skip_nfs: 'Skip scan on CIFS/NFS mounts',
        scan_on_start: 'Scan on start',
        directories: 'Monitored directories',
        nodiff: 'No diff directories',
        ignore: 'Ignored files and directories',
        restart_audit: 'Restart audit',
        startup_healthcheck: 'Startup healthcheck'
      }],
      opts: {
        realtime: 'RT',
        check_whodata: 'WD',
        report_changes: 'Changes',
        check_md5sum: 'MD5',
        check_sha1sum: 'SHA1',
        check_perm: 'Per.',
        check_size: 'Size',
        check_owner: 'Owner',
        check_group: 'Group',
        check_mtime: 'MT',
        check_inode: 'Inode',
        check_sha256sum: 'SHA256',
        follow_symbolic_link: 'SL'
      }
    }]
  }]
};
exports.AgentConfiguration = AgentConfiguration;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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