"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobMonitoringRun = jobMonitoringRun;

var _nodeCron = _interopRequireDefault(require("node-cron"));

var _logger = require("../../lib/logger");

var _monitoringTemplate = require("../../integration-files/monitoring-template");

var _getConfiguration = require("../../lib/get-configuration");

var _parseCron = require("../../lib/parse-cron");

var _indexDate = require("../../lib/index-date");

var _buildIndexSettings = require("../../lib/build-index-settings");

var _wazuhHosts = require("../../controllers/wazuh-hosts");

var _constants = require("../../../common/constants");

var _tryCatchForIndexPermissionError = require("../tryCatchForIndexPermissionError");

var _utils = require("../../../common/utils");

/*
 * Wazuh app - Module for agent info fetching functions
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
const blueWazuh = '\u001b[34mwazuh\u001b[39m';
const monitoringErrorLogColors = [blueWazuh, 'monitoring', 'error'];
const wazuhHostController = new _wazuhHosts.WazuhHostsCtrl();
let MONITORING_ENABLED, MONITORING_FREQUENCY, MONITORING_CRON_FREQ, MONITORING_CREATION, MONITORING_INDEX_PATTERN, MONITORING_INDEX_PREFIX; // Utils functions

/**
 * Get the setting value from the configuration
 * @param setting
 * @param configuration
 * @param defaultValue
 */

function getAppConfigurationSetting(setting, configuration, defaultValue) {
  return typeof configuration[setting] !== 'undefined' ? configuration[setting] : defaultValue;
}

;
/**
 * Set the monitoring variables
 * @param context
 */

function initMonitoringConfiguration(context) {
  try {
    const appConfig = (0, _getConfiguration.getConfiguration)();
    MONITORING_ENABLED = appConfig && typeof appConfig['wazuh.monitoring.enabled'] !== 'undefined' ? appConfig['wazuh.monitoring.enabled'] && appConfig['wazuh.monitoring.enabled'] !== 'worker' : _constants.WAZUH_MONITORING_DEFAULT_ENABLED;
    MONITORING_FREQUENCY = getAppConfigurationSetting('wazuh.monitoring.frequency', appConfig, _constants.WAZUH_MONITORING_DEFAULT_FREQUENCY);
    MONITORING_CRON_FREQ = (0, _parseCron.parseCron)(MONITORING_FREQUENCY);
    MONITORING_CREATION = getAppConfigurationSetting('wazuh.monitoring.creation', appConfig, _constants.WAZUH_MONITORING_DEFAULT_CREATION);
    MONITORING_INDEX_PATTERN = getAppConfigurationSetting('wazuh.monitoring.pattern', appConfig, _constants.WAZUH_MONITORING_PATTERN);
    const lastCharIndexPattern = MONITORING_INDEX_PATTERN[MONITORING_INDEX_PATTERN.length - 1];

    if (lastCharIndexPattern !== '*') {
      MONITORING_INDEX_PATTERN += '*';
    }

    ;
    MONITORING_INDEX_PREFIX = MONITORING_INDEX_PATTERN.slice(0, MONITORING_INDEX_PATTERN.length - 1);
    (0, _logger.log)('monitoring:initMonitoringConfiguration', `wazuh.monitoring.enabled: ${MONITORING_ENABLED}`, 'debug');
    (0, _logger.log)('monitoring:initMonitoringConfiguration', `wazuh.monitoring.frequency: ${MONITORING_FREQUENCY} (${MONITORING_CRON_FREQ})`, 'debug');
    (0, _logger.log)('monitoring:initMonitoringConfiguration', `wazuh.monitoring.pattern: ${MONITORING_INDEX_PATTERN} (index prefix: ${MONITORING_INDEX_PREFIX})`, 'debug');
  } catch (error) {
    const errorMessage = error.message || error;
    (0, _logger.log)('monitoring:initMonitoringConfiguration', errorMessage);
    context.wazuh.logger.error(errorMessage);
  }
}

;
/**
 * Main. First execution when installing / loading App.
 * @param context
 */

async function init(context) {
  try {
    if (MONITORING_ENABLED) {
      await checkTemplate(context);
    }

    ;
  } catch (error) {
    const errorMessage = error.message || error;
    (0, _logger.log)('monitoring:init', error.message || error);
    context.wazuh.logger.error(errorMessage);
  }
}
/**
 * Verify wazuh-agent template
 */


async function checkTemplate(context) {
  try {
    (0, _logger.log)('monitoring:checkTemplate', 'Updating the monitoring template', 'debug');

    try {
      // Check if the template already exists
      const currentTemplate = await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
        name: _constants.WAZUH_MONITORING_TEMPLATE_NAME
      }); // Copy already created index patterns

      _monitoringTemplate.monitoringTemplate.index_patterns = currentTemplate.body[_constants.WAZUH_MONITORING_TEMPLATE_NAME].index_patterns;
    } catch (error) {
      // Init with the default index pattern
      _monitoringTemplate.monitoringTemplate.index_patterns = [_constants.WAZUH_MONITORING_PATTERN];
    } // Check if the user is using a custom pattern and add it to the template if it does


    if (!_monitoringTemplate.monitoringTemplate.index_patterns.includes(MONITORING_INDEX_PATTERN)) {
      _monitoringTemplate.monitoringTemplate.index_patterns.push(MONITORING_INDEX_PATTERN);
    }

    ; // Update the monitoring template

    await context.core.elasticsearch.client.asInternalUser.indices.putTemplate({
      name: _constants.WAZUH_MONITORING_TEMPLATE_NAME,
      body: _monitoringTemplate.monitoringTemplate
    });
    (0, _logger.log)('monitoring:checkTemplate', 'Updated the monitoring template', 'debug');
  } catch (error) {
    const errorMessage = `Something went wrong updating the monitoring template ${error.message || error}`;
    (0, _logger.log)('monitoring:checkTemplate', errorMessage);
    context.wazuh.logger.error(monitoringErrorLogColors, errorMessage);
    throw error;
  }
}
/**
 * Save agent status into elasticsearch, create index and/or insert document
 * @param {*} context
 * @param {*} data
 */


async function insertMonitoringDataElasticsearch(context, data) {
  const monitoringIndexName = MONITORING_INDEX_PREFIX + (0, _indexDate.indexDate)(MONITORING_CREATION);

  if (!MONITORING_ENABLED) {
    return;
  }

  ;

  try {
    await (0, _tryCatchForIndexPermissionError.tryCatchForIndexPermissionError)(monitoringIndexName)(async () => {
      const exists = await context.core.elasticsearch.client.asInternalUser.indices.exists({
        index: monitoringIndexName
      });

      if (!exists.body) {
        await createIndex(context, monitoringIndexName);
      }

      ; // Update the index configuration

      const appConfig = (0, _getConfiguration.getConfiguration)();
      const indexConfiguration = (0, _buildIndexSettings.buildIndexSettings)(appConfig, 'wazuh.monitoring', _constants.WAZUH_MONITORING_DEFAULT_INDICES_SHARDS); // To update the index settings with this client is required close the index, update the settings and open it
      // Number of shards is not dynamic so delete that setting if it's given

      delete indexConfiguration.settings.index.number_of_shards;
      await context.core.elasticsearch.client.asInternalUser.indices.putSettings({
        index: monitoringIndexName,
        body: indexConfiguration
      }); // Insert data to the monitoring index

      await insertDataToIndex(context, monitoringIndexName, data);
    })();
  } catch (error) {
    (0, _logger.log)('monitoring:insertMonitoringDataElasticsearch', error.message || error);
    context.wazuh.logger.error(error.message);
  }
}
/**
 * Inserting one document per agent into Elastic. Bulk.
 * @param {*} context Endpoint
 * @param {String} indexName The name for the index (e.g. daily: wazuh-monitoring-YYYY.MM.DD)
 * @param {*} data
 */


async function insertDataToIndex(context, indexName, data) {
  const {
    agents,
    apiHost
  } = data;

  try {
    if (agents.length > 0) {
      (0, _logger.log)('monitoring:insertDataToIndex', `Bulk data to index ${indexName} for ${agents.length} agents`, 'debug');
      const bodyBulk = agents.map(agent => {
        const agentInfo = { ...agent
        };
        agentInfo['timestamp'] = new Date(Date.now()).toISOString();
        agentInfo.host = agent.manager;
        agentInfo.cluster = {
          name: apiHost.clusterName ? apiHost.clusterName : 'disabled'
        };
        return `{ "index":  { "_index": "${indexName}" } }\n${JSON.stringify(agentInfo)}\n`;
      }).join('');
      await context.core.elasticsearch.client.asInternalUser.bulk({
        index: indexName,
        body: bodyBulk
      });
      (0, _logger.log)('monitoring:insertDataToIndex', `Bulk data to index ${indexName} for ${agents.length} agents completed`, 'debug');
    }
  } catch (error) {
    (0, _logger.log)('monitoring:insertDataToIndex', `Error inserting agent data into elasticsearch. Bulk request failed due to ${error.message || error}`);
  }
}
/**
 * Create the wazuh-monitoring index
 * @param {*} context context
 * @param {String} indexName The name for the index (e.g. daily: wazuh-monitoring-YYYY.MM.DD)
 */


async function createIndex(context, indexName) {
  try {
    if (!MONITORING_ENABLED) return;
    const appConfig = (0, _getConfiguration.getConfiguration)();
    const IndexConfiguration = {
      settings: {
        index: {
          number_of_shards: getAppConfigurationSetting('wazuh.monitoring.shards', appConfig, _constants.WAZUH_MONITORING_DEFAULT_INDICES_SHARDS),
          number_of_replicas: getAppConfigurationSetting('wazuh.monitoring.replicas', appConfig, _constants.WAZUH_MONITORING_DEFAULT_INDICES_REPLICAS)
        }
      }
    };
    await context.core.elasticsearch.client.asInternalUser.indices.create({
      index: indexName,
      body: IndexConfiguration
    });
    (0, _logger.log)('monitoring:createIndex', `Successfully created new index: ${indexName}`, 'debug');
  } catch (error) {
    const errorMessage = `Could not create ${indexName} index on elasticsearch due to ${error.message || error}`;
    (0, _logger.log)('monitoring:createIndex', errorMessage);
    context.wazuh.logger.error(errorMessage);
  }
}
/**
* Wait until Kibana server is ready
*/


async function checkPluginPlatformStatus(context) {
  try {
    (0, _logger.log)('monitoring:checkPluginPlatformStatus', 'Waiting for Kibana and Elasticsearch servers to be ready...', 'debug');
    await checkElasticsearchServer(context);
    await init(context);
    return;
  } catch (error) {
    (0, _logger.log)('monitoring:checkPluginPlatformStatus', error.mesage || error);

    try {
      await (0, _utils.delayAsPromise)(3000);
      await checkPluginPlatformStatus(context);
    } catch (error) {}

    ;
  }
}
/**
 * Check Elasticsearch Server status and Kibana index presence
 */


async function checkElasticsearchServer(context) {
  try {
    const data = await context.core.elasticsearch.client.asInternalUser.indices.exists({
      index: context.server.config.kibana.index
    });
    return data.body; // TODO: check if Elasticsearch can receive requests
    // if (data) {
    //   const pluginsData = await this.server.plugins.elasticsearch.waitUntilReady();
    //   return pluginsData;
    // }

    return Promise.reject(data);
  } catch (error) {
    (0, _logger.log)('monitoring:checkElasticsearchServer', error.message || error);
    return Promise.reject(error);
  }
}

const fakeResponseEndpoint = {
  ok: body => body,
  custom: body => body
};
/**
 * Get API configuration from elastic and callback to loadCredentials
 */

async function getHostsConfiguration() {
  try {
    const hosts = await wazuhHostController.getHostsEntries(false, false, fakeResponseEndpoint);

    if (hosts.body.length) {
      return hosts.body;
    }

    ;
    (0, _logger.log)('monitoring:getConfig', 'There are no Wazuh API entries yet', 'debug');
    return Promise.reject({
      error: 'no credentials',
      error_code: 1
    });
  } catch (error) {
    (0, _logger.log)('monitoring:getHostsConfiguration', error.message || error);
    return Promise.reject({
      error: 'no wazuh hosts',
      error_code: 2
    });
  }
}
/**
   * Task used by the cron job.
   */


async function cronTask(context) {
  try {
    const templateMonitoring = await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
      name: _constants.WAZUH_MONITORING_TEMPLATE_NAME
    });
    const apiHosts = await getHostsConfiguration();
    const apiHostsUnique = (apiHosts || []).filter((apiHost, index, self) => index === self.findIndex(t => t.user === apiHost.user && t.password === apiHost.password && t.url === apiHost.url && t.port === apiHost.port));

    for (let apiHost of apiHostsUnique) {
      try {
        const {
          agents,
          apiHost: host
        } = await getApiInfo(context, apiHost);
        await insertMonitoringDataElasticsearch(context, {
          agents,
          apiHost: host
        });
      } catch (error) {}

      ;
    }
  } catch (error) {
    // Retry to call itself again if Kibana index is not ready yet
    // try {
    //   if (
    //     this.wzWrapper.buildingKibanaIndex ||
    //     ((error || {}).status === 404 &&
    //       (error || {}).displayName === 'NotFound')
    //   ) {
    //     await delayAsPromise(1000);
    //     return cronTask(context);
    //   }
    // } catch (error) {} //eslint-disable-line
    (0, _logger.log)('monitoring:cronTask', error.message || error);
    context.wazuh.logger.error(error.message || error);
  }
}
/**
 * Get API and agents info
 * @param context
 * @param apiHost
 */


async function getApiInfo(context, apiHost) {
  try {
    (0, _logger.log)('monitoring:getApiInfo', `Getting API info for ${apiHost.id}`, 'debug');
    const responseIsCluster = await context.wazuh.api.client.asInternalUser.request('GET', '/cluster/status', {}, {
      apiHostID: apiHost.id
    });
    const isCluster = (((responseIsCluster || {}).data || {}).data || {}).enabled === 'yes';

    if (isCluster) {
      const responseClusterInfo = await context.wazuh.api.client.asInternalUser.request('GET', `/cluster/local/info`, {}, {
        apiHostID: apiHost.id
      });
      apiHost.clusterName = responseClusterInfo.data.data.affected_items[0].cluster;
    }

    ;
    const agents = await fetchAllAgentsFromApiHost(context, apiHost);
    return {
      agents,
      apiHost
    };
  } catch (error) {
    (0, _logger.log)('monitoring:getApiInfo', error.message || error);
    throw error;
  }
}

;
/**
 * Fetch all agents for the API provided
 * @param context
 * @param apiHost
 */

async function fetchAllAgentsFromApiHost(context, apiHost) {
  let agents = [];

  try {
    (0, _logger.log)('monitoring:fetchAllAgentsFromApiHost', `Getting all agents from ApiID: ${apiHost.id}`, 'debug');
    const responseAgentsCount = await context.wazuh.api.client.asInternalUser.request('GET', '/agents', {
      params: {
        offset: 0,
        limit: 1,
        q: 'id!=000'
      }
    }, {
      apiHostID: apiHost.id
    });
    const agentsCount = responseAgentsCount.data.data.total_affected_items;
    (0, _logger.log)('monitoring:fetchAllAgentsFromApiHost', `ApiID: ${apiHost.id}, Agent count: ${agentsCount}`, 'debug');
    let payload = {
      offset: 0,
      limit: 500,
      q: 'id!=000'
    };

    while (agents.length < agentsCount && payload.offset < agentsCount) {
      try {
        /* 
        TODO: Improve the performance of request with:
          - Reduce the number of requests to the Wazuh API
          - Reduce (if possible) the quantity of data to index by document
         Requirements:
          - Research about the neccesary data to index.
         How to do:
          - Wazuh API request:
            - select the required data to retrieve depending on is required to index (using the `select` query param)
            - increase the limit of results to retrieve (currently, the requests use the recommended value: 500).
              See the allowed values. This depends on the selected data because the response could fail if contains a lot of data
        */
        const responseAgents = await context.wazuh.api.client.asInternalUser.request('GET', `/agents`, {
          params: payload
        }, {
          apiHostID: apiHost.id
        });
        agents = [...agents, ...responseAgents.data.data.affected_items];
        payload.offset += payload.limit;
      } catch (error) {
        (0, _logger.log)('monitoring:fetchAllAgentsFromApiHost', `ApiID: ${apiHost.id}, Error request with offset/limit ${payload.offset}/${payload.limit}: ${error.message || error}`);
      }
    }

    return agents;
  } catch (error) {
    (0, _logger.log)('monitoring:fetchAllAgentsFromApiHost', `ApiID: ${apiHost.id}. Error: ${error.message || error}`);
    throw error;
  }
}

;
/**
 * Start the cron job
 */

async function jobMonitoringRun(context) {
  // Init the monitoring variables
  initMonitoringConfiguration(context); // Check Kibana index and if it is prepared, start the initialization of Wazuh App.

  await checkPluginPlatformStatus(context); // // Run the cron job only it it's enabled

  if (MONITORING_ENABLED) {
    cronTask(context);

    _nodeCron.default.schedule(MONITORING_CRON_FREQ, () => cronTask(context));
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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