"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTopFailedSyscalls = exports.getTop3AgentsSudoNonSuccessful = exports.getTop3AgentsFailedSyscalls = void 0;

var _baseQuery = require("./base-query");

var _auditMap = _interopRequireDefault(require("./audit-map"));

var _constants = require("../../../common/constants");

/*
 * Wazuh app - Specific methods to fetch Wazuh Audit data from Elasticsearch
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */

/**
   * Returns top 3 agents that execute sudo commands without success
   * @param {*} context Endpoint context
   * @param {*} gte
   * @param {*} lte
   * @param {*} filters
   * @param {*} pattern
   */
const getTop3AgentsSudoNonSuccessful = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '3': {
        terms: {
          field: 'agent.id',
          size: 3,
          order: {
            _count: 'desc'
          }
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'data.audit.uid': {
          query: '0'
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'data.audit.success': {
          query: 'no'
        }
      }
    });
    base.query.bool.must_not.push({
      match_phrase: {
        'data.audit.auid': {
          query: '0'
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['3'];
    return buckets.map(item => item.key);
  } catch (error) {
    return Promise.reject(error);
  }
};
/**
 * Returns the most failed syscall in the top 3 agents with failed system calls
 * @param {*} context Endpoint context
 * @param {*} gte
 * @param {*} lte
 * @param {*} filters
 * @param {*} pattern
 */


exports.getTop3AgentsSudoNonSuccessful = getTop3AgentsSudoNonSuccessful;

const getTop3AgentsFailedSyscalls = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '3': {
        terms: {
          field: 'agent.id',
          size: 3,
          order: {
            _count: 'desc'
          }
        },
        aggs: {
          '4': {
            terms: {
              field: 'data.audit.syscall',
              size: 1,
              order: {
                _count: 'desc'
              }
            }
          }
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'data.audit.success': {
          query: 'no'
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['3'];
    return buckets.map(bucket => {
      try {
        const agent = bucket.key;
        const syscall = {
          id: bucket['4'].buckets[0].key,
          syscall: _auditMap.default[bucket['4'].buckets[0].key] || 'Warning: Unknown system call'
        };
        return {
          agent,
          syscall
        };
      } catch (error) {
        return undefined;
      }
    }).filter(bucket => bucket);
  } catch (error) {
    return Promise.reject(error);
  }
};
/**
 * Returns the top failed syscalls
 * @param {*} context Endpoint context
 * @param {*} gte
 * @param {*} lte
 * @param {*} filters
 * @param {*} pattern
 */


exports.getTop3AgentsFailedSyscalls = getTop3AgentsFailedSyscalls;

const getTopFailedSyscalls = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'data.audit.syscall',
          size: 10,
          order: {
            _count: 'desc'
          }
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'data.audit.success': {
          query: 'no'
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => ({
      id: item.key,
      syscall: _auditMap.default[item.key]
    }));
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.getTopFailedSyscalls = getTopFailedSyscalls;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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