"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhReportingRoutes = WazuhReportingRoutes;

var _controllers = require("../controllers");

var _configSchema = require("@kbn/config-schema");

/*
 * Wazuh app - Module for Wazuh reporting routes
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
function WazuhReportingRoutes(router) {
  const ctrl = new _controllers.WazuhReportingCtrl();

  const agentIDValidation = _configSchema.schema.string({
    minLength: 3,
    validate: agentID => /^\d{3,}$/.test(agentID) ? undefined : 'must be 0-9 are allowed'
  });

  const groupIDValidation = _configSchema.schema.string({
    minLength: 1,
    validate: agentID => /^(?!^(\.{1,2}|all)$)[\w\.\-]+$/.test(agentID) ? undefined : 'must be A-z, 0-9, _, . are allowed. It must not be ., .. or all.'
  });

  const ReportFilenameValidation = _configSchema.schema.string({
    validate: agentID => /^[\w\-\.]+\.pdf$/.test(agentID) ? undefined : 'must be A-z, 0-9, _, ., and - are allowed. It must end with .pdf.'
  });

  const moduleIDValidation = _configSchema.schema.oneOf([_configSchema.schema.literal('general'), _configSchema.schema.literal('fim'), _configSchema.schema.literal('aws'), _configSchema.schema.literal('gcp'), _configSchema.schema.literal('pm'), _configSchema.schema.literal('audit'), _configSchema.schema.literal('sca'), _configSchema.schema.literal('office'), _configSchema.schema.literal('github'), _configSchema.schema.literal('ciscat'), _configSchema.schema.literal('vuls'), _configSchema.schema.literal('mitre'), _configSchema.schema.literal('virustotal'), _configSchema.schema.literal('docker'), _configSchema.schema.literal('osquery'), _configSchema.schema.literal('oscap'), _configSchema.schema.literal('pci'), _configSchema.schema.literal('hipaa'), _configSchema.schema.literal('nist'), _configSchema.schema.literal('gdpr'), _configSchema.schema.literal('tsc')]);

  router.post({
    path: '/reports/modules/{moduleID}',
    validate: {
      body: _configSchema.schema.object({
        array: _configSchema.schema.any(),
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        agents: _configSchema.schema.maybe(_configSchema.schema.oneOf([agentIDValidation, _configSchema.schema.boolean()])),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        searchBar: _configSchema.schema.maybe(_configSchema.schema.string()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        tab: _configSchema.schema.string(),
        tables: _configSchema.schema.maybe(_configSchema.schema.any()),
        time: _configSchema.schema.oneOf([_configSchema.schema.object({
          from: _configSchema.schema.string(),
          to: _configSchema.schema.string()
        }), _configSchema.schema.string()]),
        indexPatternTitle: _configSchema.schema.string(),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        moduleID: moduleIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsModules(context, request, response));
  router.post({
    path: '/reports/groups/{groupID}',
    validate: {
      body: _configSchema.schema.object({
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        groupID: groupIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsGroups(context, request, response));
  router.post({
    path: '/reports/agents/{agentID}',
    validate: {
      body: _configSchema.schema.object({
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.any(),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        agentID: agentIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsAgentsConfiguration(context, request, response));
  router.post({
    path: '/reports/agents/{agentID}/inventory',
    validate: {
      body: _configSchema.schema.object({
        array: _configSchema.schema.any(),
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        agents: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.boolean()])),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        searchBar: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.boolean()])),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        tab: _configSchema.schema.string(),
        tables: _configSchema.schema.maybe(_configSchema.schema.any()),
        time: _configSchema.schema.oneOf([_configSchema.schema.object({
          from: _configSchema.schema.string(),
          to: _configSchema.schema.string()
        }), _configSchema.schema.string()]),
        indexPatternTitle: _configSchema.schema.string(),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        agentID: agentIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsAgentsInventory(context, request, response)); // Fetch specific report

  router.get({
    path: '/reports/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: ReportFilenameValidation
      })
    }
  }, (context, request, response) => ctrl.getReportByName(context, request, response)); // Delete specific report

  router.delete({
    path: '/reports/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: ReportFilenameValidation
      })
    }
  }, (context, request, response) => ctrl.deleteReportByName(context, request, response)); // Fetch the reports list

  router.get({
    path: '/reports',
    validate: false
  }, (context, request, response) => ctrl.getReports(context, request, response));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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