"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SaveDocument = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _getConfiguration = require("../../lib/get-configuration");

var _logger = require("../../lib/logger");

var _indexDate = require("../../lib/index-date");

var _constants = require("../../../common/constants");

var _tryCatchForIndexPermissionError = require("../tryCatchForIndexPermissionError");

class SaveDocument {
  constructor(context) {
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "esClientInternalUser", void 0);
    (0, _defineProperty2.default)(this, "logPath", 'cron-scheduler|SaveDocument');
    this.context = context;
    this.esClientInternalUser = context.core.elasticsearch.client.asInternalUser;
  }

  async save(doc, indexConfig) {
    const {
      name,
      creation,
      mapping,
      shards,
      replicas
    } = indexConfig;
    const index = this.addIndexPrefix(name);
    const indexCreation = `${index}-${(0, _indexDate.indexDate)(creation)}`;

    try {
      await this.checkIndexAndCreateIfNotExists(indexCreation, shards, replicas);
      const createDocumentObject = this.createDocument(doc, indexCreation, mapping);
      const response = await this.esClientInternalUser.bulk(createDocumentObject);
      (0, _logger.log)(this.logPath, `Response of create new document ${JSON.stringify(response)}`, 'debug'); // await this.checkIndexPatternAndCreateIfNotExists(index);
    } catch (error) {
      if (error.status === 403) throw {
        error: 403,
        message: `Authorization Exception in the index "${index}"`
      };
      if (error.status === 409) throw {
        error: 409,
        message: `Duplicate index-pattern: ${index}`
      };
      throw error;
    }
  }

  async checkIndexAndCreateIfNotExists(index, shards, replicas) {
    try {
      await (0, _tryCatchForIndexPermissionError.tryCatchForIndexPermissionError)(index)(async () => {
        const exists = await this.esClientInternalUser.indices.exists({
          index
        });
        (0, _logger.log)(this.logPath, `Index '${index}' exists? ${exists.body}`, 'debug');

        if (!exists.body) {
          const response = await this.esClientInternalUser.indices.create({
            index,
            body: {
              settings: {
                index: {
                  number_of_shards: shards !== null && shards !== void 0 ? shards : _constants.WAZUH_STATISTICS_DEFAULT_INDICES_SHARDS,
                  number_of_replicas: replicas !== null && replicas !== void 0 ? replicas : _constants.WAZUH_STATISTICS_DEFAULT_INDICES_REPLICAS
                }
              }
            }
          });
          (0, _logger.log)(this.logPath, `Status of create a new index: ${JSON.stringify(response)}`, 'debug');
        }
      })();
    } catch (error) {
      this.checkDuplicateIndexError(error);
    }
  }

  checkDuplicateIndexError(error) {
    const {
      type
    } = ((error || {}).body || {}).error || {};
    if (!['resource_already_exists_exception'].includes(type)) throw error;
  }

  createDocument(doc, index, mapping) {
    const createDocumentObject = {
      index,
      type: '_doc',
      body: doc.flatMap(item => [{
        index: {
          _index: index
        }
      }, { ...this.buildData(item, mapping),
        timestamp: new Date(Date.now()).toISOString()
      }])
    };
    (0, _logger.log)(this.logPath, `Document object: ${JSON.stringify(createDocumentObject)}`, 'debug');
    return createDocumentObject;
  }

  buildData(item, mapping) {
    const getItemArray = (array, index) => {
      return JSON.stringify(array[index || 0]);
    };

    const getValue = (key, item) => {
      const keys = key.split('.');

      if (keys.length === 1) {
        if (key.match(/\[.*\]/)) {
          return getItemArray(item[key.replace(/\[.*\]/, '')], key.match(/\[(.*)\]/)[1]);
        }

        return JSON.stringify(item[key]);
      }

      return getValue(keys.slice(1).join('.'), item[keys[0]]);
    };

    if (mapping) {
      let data;
      data = mapping.replace(/\${([a-z|A-Z|0-9|\.\-\_\[.*\]]+)}/gi, (...key) => getValue(key[1], item));
      return JSON.parse(data);
    }

    if (typeof item.data === 'object') {
      return item.data;
    }

    return {
      data: item.data
    };
  }

  addIndexPrefix(index) {
    const configFile = (0, _getConfiguration.getConfiguration)();
    const prefix = configFile['cron.prefix'] || 'wazuh';
    return `${prefix}-${index}`;
  }

}

exports.SaveDocument = SaveDocument;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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