"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ReportPrinter = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _fs = _interopRequireDefault(require("fs"));

var _path = _interopRequireDefault(require("path"));

var _printer = _interopRequireDefault(require("pdfmake/src/printer"));

var _clockIconRaw = _interopRequireDefault(require("./clock-icon-raw"));

var _filterIconRaw = _interopRequireDefault(require("./filter-icon-raw"));

var _visualizations = require("../../integration-files/visualizations");

var _logger = require("../logger");

var TimSort = _interopRequireWildcard(require("timsort"));

var _getConfiguration = require("../get-configuration");

var _constants = require("../../../common/constants");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

const COLORS = {
  PRIMARY: _constants.REPORTS_PRIMARY_COLOR
};

const pageConfiguration = nameLogo => ({
  styles: {
    h1: {
      fontSize: 22,
      monslight: true,
      color: COLORS.PRIMARY
    },
    h2: {
      fontSize: 18,
      monslight: true,
      color: COLORS.PRIMARY
    },
    h3: {
      fontSize: 16,
      monslight: true,
      color: COLORS.PRIMARY
    },
    h4: {
      fontSize: 14,
      monslight: true,
      color: COLORS.PRIMARY
    },
    standard: {
      color: '#333'
    },
    whiteColorFilters: {
      color: '#FFF',
      fontSize: 14
    },
    whiteColor: {
      color: '#FFF'
    }
  },
  pageMargins: [40, 80, 40, 80],
  header: {
    margin: [40, 20, 0, 0],
    columns: [{
      image: _path.default.join(__dirname, `../../../public/assets/${nameLogo}`),
      width: 190
    }, {
      text: _constants.REPORTS_PAGE_HEADER_TEXT,
      alignment: 'right',
      margin: [0, 0, 40, 0],
      color: COLORS.PRIMARY
    }]
  },
  content: [],

  footer(currentPage, pageCount) {
    return {
      columns: [{
        text: _constants.REPORTS_PAGE_FOOTER_TEXT,
        color: COLORS.PRIMARY,
        margin: [40, 40, 0, 0]
      }, {
        text: 'Page ' + currentPage.toString() + ' of ' + pageCount,
        alignment: 'right',
        margin: [0, 40, 40, 0],
        color: COLORS.PRIMARY
      }]
    };
  },

  pageBreakBefore(currentNode, followingNodesOnPage) {
    if (currentNode.id && currentNode.id.includes('splitvis')) {
      return followingNodesOnPage.length === 6 || followingNodesOnPage.length === 7;
    }

    if (currentNode.id && currentNode.id.includes('splitsinglevis') || currentNode.id && currentNode.id.includes('singlevis')) {
      return followingNodesOnPage.length === 6;
    }

    return false;
  }

});

const fonts = {
  Roboto: {
    normal: _path.default.join(__dirname, '../../../public/assets/fonts/opensans/OpenSans-Light.ttf'),
    bold: _path.default.join(__dirname, '../../../public/assets/fonts/opensans/OpenSans-Bold.ttf'),
    italics: _path.default.join(__dirname, '../../../public/assets/fonts/opensans/OpenSans-Italic.ttf'),
    bolditalics: _path.default.join(__dirname, '../../../public/assets/fonts/opensans/OpenSans-BoldItalic.ttf'),
    monslight: _path.default.join(__dirname, '../../../public/assets/fonts/opensans/Montserrat-Light.ttf')
  }
};

class ReportPrinter {
  constructor() {
    (0, _defineProperty2.default)(this, "_content", void 0);
    (0, _defineProperty2.default)(this, "_printer", void 0);
    this._printer = new _printer.default(fonts);
    this._content = [];
  }

  addContent(...content) {
    this._content.push(...content);

    return this;
  }

  addConfigTables(tables) {
    (0, _logger.log)('reporting:renderConfigTables', 'Started to render configuration tables', 'info');
    (0, _logger.log)('reporting:renderConfigTables', `tables: ${tables.length}`, 'debug');

    for (const table of tables) {
      let rowsparsed = table.rows;

      if (Array.isArray(rowsparsed) && rowsparsed.length) {
        const rows = rowsparsed.length > 100 ? rowsparsed.slice(0, 99) : rowsparsed;
        this.addContent({
          text: table.title,
          style: {
            fontSize: 11,
            color: '#000'
          },
          margin: table.title && table.type === 'table' ? [0, 0, 0, 5] : ''
        });

        if (table.title === 'Monitored directories') {
          this.addContent({
            text: 'RT: Real time | WD: Who-data | Per.: Permission | MT: Modification time | SL: Symbolic link | RL: Recursion level',
            style: {
              fontSize: 8,
              color: COLORS.PRIMARY
            },
            margin: [0, 0, 0, 5]
          });
        }

        const full_body = [];
        const modifiedRows = rows.map(row => row.map(cell => ({
          text: cell || '-',
          style: 'standard'
        }))); // for (const row of rows) {
        //   modifiedRows.push(
        //     row.map(cell => ({ text: cell || '-', style: 'standard' }))
        //   );
        // }

        let widths = [];
        widths = Array(table.columns.length - 1).fill('auto');
        widths.push('*');

        if (table.type === 'config') {
          full_body.push(table.columns.map(col => ({
            text: col || '-',
            border: [0, 0, 0, 20],
            fontSize: 0,
            colSpan: 2
          })), ...modifiedRows);
          this.addContent({
            fontSize: 8,
            table: {
              headerRows: 0,
              widths,
              body: full_body,
              dontBreakRows: true
            },
            layout: {
              fillColor: i => i === 0 ? '#fff' : null,
              hLineColor: () => '#D3DAE6',
              hLineWidth: () => 1,
              vLineWidth: () => 0
            }
          });
        } else if (table.type === 'table') {
          full_body.push(table.columns.map(col => ({
            text: col || '-',
            style: 'whiteColor',
            border: [0, 0, 0, 0]
          })), ...modifiedRows);
          this.addContent({
            fontSize: 8,
            table: {
              headerRows: 1,
              widths,
              body: full_body
            },
            layout: {
              fillColor: i => i === 0 ? COLORS.PRIMARY : null,
              hLineColor: () => COLORS.PRIMARY,
              hLineWidth: () => 1,
              vLineWidth: () => 0
            }
          });
        }

        this.addNewLine();
      }

      (0, _logger.log)('reporting:renderConfigTables', `Table rendered`, 'debug');
    }
  }

  addTables(tables) {
    (0, _logger.log)('reporting:renderTables', 'Started to render tables', 'info');
    (0, _logger.log)('reporting:renderTables', `tables: ${tables.length}`, 'debug');

    for (const table of tables) {
      let rowsparsed = [];
      rowsparsed = table.rows;

      if (Array.isArray(rowsparsed) && rowsparsed.length) {
        const rows = rowsparsed.length > 100 ? rowsparsed.slice(0, 99) : rowsparsed;
        this.addContent({
          text: table.title,
          style: 'h3',
          pageBreak: 'before',
          pageOrientation: table.columns.length >= 9 ? 'landscape' : 'portrait'
        });
        this.addNewLine();
        const full_body = [];

        const sortTableRows = (a, b) => parseInt(a[a.length - 1]) < parseInt(b[b.length - 1]) ? 1 : parseInt(a[a.length - 1]) > parseInt(b[b.length - 1]) ? -1 : 0;

        TimSort.sort(rows, sortTableRows);
        const modifiedRows = rows.map(row => row.map(cell => ({
          text: cell || '-',
          style: 'standard'
        }))); // the width of the columns is assigned

        const widths = Array(table.columns.length - 1).fill('auto');
        widths.push('*');
        full_body.push(table.columns.map(col => ({
          text: col || '-',
          style: 'whiteColor',
          border: [0, 0, 0, 0]
        })), ...modifiedRows);
        this.addContent({
          fontSize: 8,
          table: {
            headerRows: 1,
            widths,
            body: full_body
          },
          layout: {
            fillColor: i => i === 0 ? COLORS.PRIMARY : null,
            hLineColor: () => COLORS.PRIMARY,
            hLineWidth: () => 1,
            vLineWidth: () => 0
          }
        });
        this.addNewLine();
        (0, _logger.log)('reporting:renderTables', `Table rendered`, 'debug');
      }
    }
  }

  addTimeRangeAndFilters(from, to, filters, timeZone) {
    (0, _logger.log)('reporting:renderTimeRangeAndFilters', `Started to render the time range and the filters`, 'info');
    (0, _logger.log)('reporting:renderTimeRangeAndFilters', `from: ${from}, to: ${to}, filters: ${filters}, timeZone: ${timeZone}`, 'debug');
    const fromDate = new Date(new Date(from).toLocaleString('en-US', {
      timeZone
    }));
    const toDate = new Date(new Date(to).toLocaleString('en-US', {
      timeZone
    }));
    const str = `${this.formatDate(fromDate)} to ${this.formatDate(toDate)}`;
    this.addContent({
      fontSize: 8,
      table: {
        widths: ['*'],
        body: [[{
          columns: [{
            svg: _clockIconRaw.default,
            width: 10,
            height: 10,
            margin: [40, 5, 0, 0]
          }, {
            text: str || '-',
            margin: [43, 0, 0, 0],
            style: 'whiteColorFilters'
          }]
        }], [{
          columns: [{
            svg: _filterIconRaw.default,
            width: 10,
            height: 10,
            margin: [40, 6, 0, 0]
          }, {
            text: filters || '-',
            margin: [43, 0, 0, 0],
            style: 'whiteColorFilters'
          }]
        }]]
      },
      margin: [-40, 0, -40, 0],
      layout: {
        fillColor: () => COLORS.PRIMARY,
        hLineWidth: () => 0,
        vLineWidth: () => 0
      }
    });
    this.addContent({
      text: '\n'
    });
    (0, _logger.log)('reporting:renderTimeRangeAndFilters', 'Time range and filters rendered', 'debug');
  }

  addVisualizations(visualizations, isAgents, tab) {
    (0, _logger.log)('reporting:renderVisualizations', `${visualizations.length} visualizations for tab ${tab}`, 'info');
    const single_vis = visualizations.filter(item => item.width >= 600);
    const double_vis = visualizations.filter(item => item.width < 600);
    single_vis.forEach(visualization => {
      const title = this.checkTitle(visualization, isAgents, tab);
      this.addContent({
        id: 'singlevis' + title[0]._source.title,
        text: title[0]._source.title,
        style: 'h3'
      });
      this.addContent({
        columns: [{
          image: visualization.element,
          width: 500
        }]
      });
      this.addNewLine();
    });
    let pair = [];

    for (const item of double_vis) {
      pair.push(item);

      if (pair.length === 2) {
        const title_1 = this.checkTitle(pair[0], isAgents, tab);
        const title_2 = this.checkTitle(pair[1], isAgents, tab);
        this.addContent({
          columns: [{
            id: 'splitvis' + title_1[0]._source.title,
            text: title_1[0]._source.title,
            style: 'h3',
            width: 280
          }, {
            id: 'splitvis' + title_2[0]._source.title,
            text: title_2[0]._source.title,
            style: 'h3',
            width: 280
          }]
        });
        this.addContent({
          columns: [{
            image: pair[0].element,
            width: 270
          }, {
            image: pair[1].element,
            width: 270
          }]
        });
        this.addNewLine();
        pair = [];
      }
    }

    if (double_vis.length % 2 !== 0) {
      const item = double_vis[double_vis.length - 1];
      const title = this.checkTitle(item, isAgents, tab);
      this.addContent({
        columns: [{
          id: 'splitsinglevis' + title[0]._source.title,
          text: title[0]._source.title,
          style: 'h3',
          width: 280
        }]
      });
      this.addContent({
        columns: [{
          image: item.element,
          width: 280
        }]
      });
      this.addNewLine();
    }
  }

  formatDate(date) {
    (0, _logger.log)('reporting:formatDate', `Format date ${date}`, 'info');
    const year = date.getFullYear();
    const month = date.getMonth() + 1;
    const day = date.getDate();
    const hours = date.getHours();
    const minutes = date.getMinutes();
    const seconds = date.getSeconds();
    const str = `${year}-${month < 10 ? '0' + month : month}-${day < 10 ? '0' + day : day}T${hours < 10 ? '0' + hours : hours}:${minutes < 10 ? '0' + minutes : minutes}:${seconds < 10 ? '0' + seconds : seconds}`;
    (0, _logger.log)('reporting:formatDate', `str: ${str}`, 'debug');
    return str;
  }

  checkTitle(item, isAgents, tab) {
    (0, _logger.log)('reporting:checkTitle', `Item ID ${item.id}, from ${isAgents ? 'agents' : 'overview'} and tab ${tab}`, 'info');
    const title = isAgents ? _visualizations.AgentsVisualizations[tab].filter(v => v._id === item.id) : _visualizations.OverviewVisualizations[tab].filter(v => v._id === item.id);
    return title;
  }

  addSimpleTable({
    columns,
    items,
    title
  }) {
    if (title) {
      this.addContent(typeof title === 'string' ? {
        text: title,
        style: 'h4'
      } : title).addNewLine();
    }

    if (!items || !items.length) {
      this.addContent({
        text: 'No results match your search criteria',
        style: 'standard'
      });
      return this;
    }

    const tableHeader = columns.map(column => {
      return {
        text: column.label,
        style: 'whiteColor',
        border: [0, 0, 0, 0]
      };
    });
    const tableRows = items.map((item, index) => {
      return columns.map(column => {
        const cellValue = item[column.id];
        return {
          text: typeof cellValue !== 'undefined' ? cellValue : '-',
          style: 'standard'
        };
      });
    }); // 385 is the max initial width per column

    let totalLength = columns.length - 1;
    const widthColumn = 385 / totalLength;
    let totalWidth = totalLength * widthColumn;
    const widths = [];

    for (let step = 0; step < columns.length - 1; step++) {
      let columnLength = this.getColumnWidth(columns[step], tableRows, step);

      if (columnLength <= Math.round(totalWidth / totalLength)) {
        widths.push(columnLength);
        totalWidth -= columnLength;
      } else {
        widths.push(Math.round(totalWidth / totalLength));
        totalWidth -= Math.round(totalWidth / totalLength);
      }

      totalLength--;
    }

    widths.push('*');
    this.addContent({
      fontSize: 8,
      table: {
        headerRows: 1,
        widths,
        body: [tableHeader, ...tableRows]
      },
      layout: {
        fillColor: i => i === 0 ? COLORS.PRIMARY : null,
        hLineColor: () => COLORS.PRIMARY,
        hLineWidth: () => 1,
        vLineWidth: () => 0
      }
    }).addNewLine();
    return this;
  }

  addList({
    title,
    list
  }) {
    return this.addContentWithNewLine(typeof title === 'string' ? {
      text: title,
      style: 'h2'
    } : title).addContent({
      ul: list.filter(element => element)
    }).addNewLine();
  }

  addNewLine() {
    return this.addContent({
      text: '\n'
    });
  }

  addContentWithNewLine(title) {
    return this.addContent(title).addNewLine();
  }

  addAgentsFilters(agents) {
    (0, _logger.log)('reporting:addAgentsFilters', `Started to render the authorized agents filters`, 'info');
    (0, _logger.log)('reporting:addAgentsFilters', `agents: ${agents}`, 'debug');
    this.addNewLine();
    this.addContent({
      text: 'NOTE: This report only includes the authorized agents of the user who generated the report',
      style: {
        fontSize: 10,
        color: COLORS.PRIMARY
      },
      margin: [0, 0, 0, 5]
    });
    /*TODO: This will be enabled by a config*/

    /* this.addContent({
      fontSize: 8,
      table: {
        widths: ['*'],
        body: [
          [
            {
              columns: [
                {
                  svg: filterIconRaw,
                  width: 10,
                  height: 10,
                  margin: [40, 6, 0, 0]
                },
                {
                  text: `Agent IDs: ${agents}` || '-',
                  margin: [43, 0, 0, 0],
                  style: { fontSize: 8, color: '#333' }
                }
              ]
            }
          ]
        ]
      },
      margin: [-40, 0, -40, 0],
      layout: {
        fillColor: () => null,
        hLineWidth: () => 0,
        vLineWidth: () => 0
      }
    }); */

    this.addContent({
      text: '\n'
    });
    (0, _logger.log)('reporting:addAgentsFilters', 'Time range and filters rendered', 'debug');
  }

  async print(reportPath) {
    const nameLogo = (await (0, _getConfiguration.getConfiguration)())['customization.logo.reports'] || _constants.REPORTS_LOGO_IMAGE_ASSETS_RELATIVE_PATH;

    const document = this._printer.createPdfKitDocument({ ...pageConfiguration(nameLogo),
      content: this._content
    });

    await document.pipe(_fs.default.createWriteStream(reportPath));
    document.end();
  }
  /**
   * Returns the width of a given column
   * 
   * @param column 
   * @param tableRows 
   * @param step 
   * @returns {number}
   */


  getColumnWidth(column, tableRows, index) {
    const widthCharacter = 5; //min width per character
    //Get the longest row value

    const maxRowLength = tableRows.reduce((maxLength, row) => {
      return row[index].text.length > maxLength ? row[index].text.length : maxLength;
    }, 0); //Get column name length

    const headerLength = column.label.length; //Use the longest to get the column width

    const maxLength = maxRowLength > headerLength ? maxRowLength : headerLength;
    return maxLength * widthCharacter;
  }

}

exports.ReportPrinter = ReportPrinter;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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