"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobInitializeRun = jobInitializeRun;

var _logger = require("../../lib/logger");

var _package = _interopRequireDefault(require("../../../package.json"));

var _kibanaTemplate = require("../../integration-files/kibana-template");

var _getConfiguration = require("../../lib/get-configuration");

var _os = require("os");

var _fs = _interopRequireDefault(require("fs"));

var _constants = require("../../../common/constants");

var _filesystem = require("../../lib/filesystem");

var _lodash = _interopRequireDefault(require("lodash"));

/*
 * Wazuh app - Module for app initialization
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
function jobInitializeRun(context) {
  const PLUGIN_PLATFORM_INDEX = context.server.config.kibana.index;
  (0, _logger.log)('initialize', `${_constants.PLUGIN_PLATFORM_NAME} index: ${PLUGIN_PLATFORM_INDEX}`, 'info');
  (0, _logger.log)('initialize', `App revision: ${_package.default.revision}`, 'info');
  let configurationFile = {};
  let pattern = null; // Read config from package.json and wazuh.yml

  try {
    configurationFile = (0, _getConfiguration.getConfiguration)();
    pattern = configurationFile && typeof configurationFile.pattern !== 'undefined' ? configurationFile.pattern : _constants.WAZUH_ALERTS_PATTERN;
  } catch (error) {
    (0, _logger.log)('initialize', error.message || error);
    context.wazuh.logger.error('Something went wrong while reading the configuration.' + (error.message || error));
  }

  try {
    // RAM in MB
    const ram = Math.ceil((0, _os.totalmem)() / 1024 / 1024);
    (0, _logger.log)('initialize', `Total RAM: ${ram}MB`, 'info');
  } catch (error) {
    (0, _logger.log)('initialize', `Could not check total RAM due to: ${error.message || error}`);
  } // Save Wazuh App setup


  const saveConfiguration = async (hosts = {}) => {
    try {
      const commonDate = new Date().toISOString();
      const configuration = {
        name: 'Wazuh App',
        'app-version': _package.default.version,
        revision: _package.default.revision,
        installationDate: commonDate,
        lastRestart: commonDate,
        hosts
      };

      try {
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDataDirectoryIfNotExists)('config');
        (0, _logger.log)('initialize:saveConfiguration', `Saving configuration in registry file: ${JSON.stringify(configuration)}`, 'debug');
        await _fs.default.writeFileSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH, JSON.stringify(configuration), 'utf8');
        (0, _logger.log)('initialize:saveConfiguration', 'Wazuh configuration registry saved.', 'debug');
      } catch (error) {
        (0, _logger.log)('initialize:saveConfiguration', error.message || error);
        context.wazuh.logger.error('Could not create Wazuh configuration registry');
      }
    } catch (error) {
      (0, _logger.log)('initialize:saveConfiguration', error.message || error);
      context.wazuh.logger.error('Error creating wazuh-registry.json file.');
    }
  };
  /**
   * Checks if the .wazuh-registry.json file exists:
   * - yes: check the plugin version and revision match the values stored in the registry file.
   *  If not, then it migrates the data rebuilding the registry file.
   * - no: create the file with empty hosts
   */


  const checkWazuhRegistry = async () => {
    (0, _logger.log)('initialize:checkwazuhRegistry', 'Checking wazuh-registry.json file.', 'debug');

    if (!_fs.default.existsSync(_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH)) {
      throw new Error(`The data directory is missing in the ${_constants.PLUGIN_PLATFORM_NAME} root instalation. Create the directory in ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH} and give it the required permissions (sudo mkdir ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH};sudo chown -R ${_constants.PLUGIN_PLATFORM_INSTALLATION_USER}:${_constants.PLUGIN_PLATFORM_INSTALLATION_USER_GROUP} ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH}). After restart the ${_constants.PLUGIN_PLATFORM_NAME} service.`);
    }

    ;

    if (!_fs.default.existsSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH)) {
      (0, _logger.log)('initialize:checkwazuhRegistry', 'wazuh-registry.json file does not exist. Initializing configuration.', 'debug'); // Create the app registry file for the very first time

      await saveConfiguration();
    } else {
      // If this function fails, it throws an exception
      const source = JSON.parse(_fs.default.readFileSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH, 'utf8')); // Check if the stored revision differs from the package.json revision

      const isUpgradedApp = _package.default.revision !== source.revision || _package.default.version !== source['app-version']; // Rebuild the registry file if revision or version fields are differents

      if (isUpgradedApp) {
        (0, _logger.log)('initialize:checkwazuhRegistry', 'Wazuh app revision or version changed, regenerating wazuh-registry.json.', 'info'); // Rebuild the registry file `wazuh-registry.json`
        // Get the supported extensions for the installed plugin

        const supportedDefaultExtensionsConfiguration = Object.entries(_constants.WAZUH_DEFAULT_APP_CONFIG).filter(([setting]) => setting.startsWith('extensions.')).map(([setting, settingValue]) => {
          return [setting.split('.')[1], settingValue];
        }); // Get the supported extensions by ID

        const supportedDefaultExtensionsNames = supportedDefaultExtensionsConfiguration.map(([setting]) => setting); // Generate the hosts data, migrating the extensions.
        // Keep the supported and existent extensions for the installed plugin with the configurated value
        // Add the extensions with default values that didn't exist in the previous configuration
        // Remove the unsupported extensions for the installed plugin

        const registryHostsData = Object.entries(source.hosts).reduce((accum, [hostID, hostData]) => {
          accum[hostID] = hostData;

          if (accum[hostID].extensions) {
            // Migrate extensions to those supported by the installed plugin
            const defaultHostExtentionsConfiguration = Object.fromEntries(supportedDefaultExtensionsConfiguration); // Select of current configuration the extension IDs that are supported in the installed plugin

            const currentHostConfiguration = _lodash.default.pick(accum[hostID].extensions, supportedDefaultExtensionsNames); // Merge the default extensions configuration with the configuration stored in the registry file


            accum[hostID].extensions = _lodash.default.merge(defaultHostExtentionsConfiguration, currentHostConfiguration);
          }

          return accum;
        }, {}); // Rebuild the registry file with the migrated host data (extensions are migrated to these supported by the installed plugin).

        await saveConfiguration(registryHostsData);
        (0, _logger.log)('initialize:checkwazuhRegistry', 'Migrated the registry file.', 'info');
      }
    }
  }; // Init function. Check for wazuh-registry.json file exists.


  const init = async () => {
    await checkWazuhRegistry();
  };

  const createKibanaTemplate = () => {
    (0, _logger.log)('initialize:createKibanaTemplate', `Creating template for ${PLUGIN_PLATFORM_INDEX}`, 'debug');

    try {
      _kibanaTemplate.pluginPlatformTemplate.template = PLUGIN_PLATFORM_INDEX + '*';
    } catch (error) {
      (0, _logger.log)('initialize:createKibanaTemplate', error.message || error);
      context.wazuh.logger.error('Exception: ' + error.message || error);
    }

    return context.core.elasticsearch.client.asInternalUser.indices.putTemplate({
      name: _constants.WAZUH_PLUGIN_PLATFORM_TEMPLATE_NAME,
      order: 0,
      create: true,
      body: _kibanaTemplate.pluginPlatformTemplate
    });
  };

  const createEmptyKibanaIndex = async () => {
    try {
      (0, _logger.log)('initialize:createEmptyKibanaIndex', `Creating ${PLUGIN_PLATFORM_INDEX} index.`, 'info');
      await context.core.elasticsearch.client.asInternalUser.indices.create({
        index: PLUGIN_PLATFORM_INDEX
      });
      (0, _logger.log)('initialize:createEmptyKibanaIndex', `Successfully created ${PLUGIN_PLATFORM_INDEX} index.`, 'debug');
      await init();
    } catch (error) {
      return Promise.reject(new Error(`Error creating ${PLUGIN_PLATFORM_INDEX} index due to ${error.message || error}`));
    }
  };

  const fixKibanaTemplate = async () => {
    try {
      await createKibanaTemplate();
      (0, _logger.log)('initialize:fixKibanaTemplate', `Successfully created ${PLUGIN_PLATFORM_INDEX} template.`, 'debug');
      await createEmptyKibanaIndex();
    } catch (error) {
      return Promise.reject(new Error(`Error creating template for ${PLUGIN_PLATFORM_INDEX} due to ${error.message || error}`));
    }
  };

  const getTemplateByName = async () => {
    try {
      await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
        name: _constants.WAZUH_PLUGIN_PLATFORM_TEMPLATE_NAME
      });
      (0, _logger.log)('initialize:getTemplateByName', `No need to create the ${PLUGIN_PLATFORM_INDEX} template, already exists.`, 'debug');
      await createEmptyKibanaIndex();
    } catch (error) {
      (0, _logger.log)('initialize:getTemplateByName', error.message || error);
      return fixKibanaTemplate();
    }
  }; // Does Kibana index exist?


  const checkKibanaStatus = async () => {
    try {
      const response = await context.core.elasticsearch.client.asInternalUser.indices.exists({
        index: PLUGIN_PLATFORM_INDEX
      });

      if (response.body) {
        // It exists, initialize!
        await init();
      } else {
        // No Kibana index created...
        (0, _logger.log)('initialize:checkKibanaStatus', `Not found ${PLUGIN_PLATFORM_INDEX} index`, 'info');
        await getTemplateByName();
      }
    } catch (error) {
      (0, _logger.log)('initialize:checkKibanaStatus', error.message || error);
      context.wazuh.logger.error(error.message || error);
    }
  }; // Wait until Elasticsearch js is ready


  const checkStatus = async () => {
    try {
      // TODO: wait until elasticsearch is ready?
      // await server.plugins.elasticsearch.waitUntilReady();
      return await checkKibanaStatus();
    } catch (error) {
      (0, _logger.log)('initialize:checkStatus', 'Waiting for elasticsearch plugin to be ready...', 'debug');
      setTimeout(() => checkStatus(), 3000);
    }
  }; // Check Kibana index and if it is prepared, start the initialization of Wazuh App.


  return checkStatus();
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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