"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobInitializeRun = jobInitializeRun;

var _logger = require("../../lib/logger");

var _package = _interopRequireDefault(require("../../../package.json"));

var _kibanaTemplate = require("../../integration-files/kibana-template");

var _getConfiguration = require("../../lib/get-configuration");

var _os = require("os");

var _fs = _interopRequireDefault(require("fs"));

var _manageHosts = require("../../lib/manage-hosts");

var _constants = require("../../../common/constants");

var _filesystem = require("../../lib/filesystem");

/*
 * Wazuh app - Module for app initialization
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
const manageHosts = new _manageHosts.ManageHosts();

function jobInitializeRun(context) {
  const PLUGIN_PLATFORM_INDEX = context.server.config.kibana.index;
  (0, _logger.log)('initialize', `${_constants.PLUGIN_PLATFORM_NAME} index: ${PLUGIN_PLATFORM_INDEX}`, 'info');
  (0, _logger.log)('initialize', `App revision: ${_package.default.revision}`, 'info');
  let configurationFile = {};
  let pattern = null; // Read config from package.json and wazuh.yml

  try {
    configurationFile = (0, _getConfiguration.getConfiguration)();
    pattern = configurationFile && typeof configurationFile.pattern !== 'undefined' ? configurationFile.pattern : _constants.WAZUH_ALERTS_PATTERN; // global.XPACK_RBAC_ENABLED =
    //   configurationFile &&
    //     typeof configurationFile['xpack.rbac.enabled'] !== 'undefined'
    //     ? configurationFile['xpack.rbac.enabled']
    //     : true;
  } catch (error) {
    (0, _logger.log)('initialize', error.message || error);
    context.wazuh.logger.error('Something went wrong while reading the configuration.' + (error.message || error));
  }

  try {
    // RAM in MB
    const ram = Math.ceil((0, _os.totalmem)() / 1024 / 1024);
    (0, _logger.log)('initialize', `Total RAM: ${ram}MB`, 'info');
  } catch (error) {
    (0, _logger.log)('initialize', `Could not check total RAM due to: ${error.message || error}`);
  } // Save Wazuh App setup


  const saveConfiguration = async () => {
    try {
      const commonDate = new Date().toISOString();
      const configuration = {
        name: 'Wazuh App',
        'app-version': _package.default.version,
        revision: _package.default.revision,
        installationDate: commonDate,
        lastRestart: commonDate,
        hosts: {}
      };

      try {
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDataDirectoryIfNotExists)('config');
        await _fs.default.writeFileSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH, JSON.stringify(configuration), 'utf8');
        (0, _logger.log)('initialize:saveConfiguration', 'Wazuh configuration registry inserted', 'debug');
      } catch (error) {
        (0, _logger.log)('initialize:saveConfiguration', error.message || error);
        context.wazuh.logger.error('Could not create Wazuh configuration registry');
      }
    } catch (error) {
      (0, _logger.log)('initialize:saveConfiguration', error.message || error);
      context.wazuh.logger.error('Error creating wazuh-version registry');
    }
  };
  /**
   * Checks if the .wazuh-version exists, in this case it will be deleted and the wazuh-registry.json will be created
   */


  const checkWazuhRegistry = async () => {
    try {
      (0, _logger.log)('initialize:checkwazuhRegistry', 'Checking wazuh-version registry.', 'debug');

      if (!_fs.default.existsSync(_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH)) {
        throw new Error(`The data directory is missing in the ${_constants.PLUGIN_PLATFORM_NAME} root instalation. Create the directory in ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH} and give it the required permissions (sudo mkdir ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH};sudo chown -R ${_constants.PLUGIN_PLATFORM_INSTALLATION_USER}:${_constants.PLUGIN_PLATFORM_INSTALLATION_USER_GROUP} ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH}). After restart the ${_constants.PLUGIN_PLATFORM_NAME} service.`);
      }

      ;

      if (!_fs.default.existsSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH)) {
        (0, _logger.log)('initialize:checkwazuhRegistry', 'wazuh-version registry does not exist. Initializing configuration.', 'debug'); // Create the app registry file for the very first time

        await saveConfiguration();
      } else {
        // If this function fails, it throws an exception
        const source = JSON.parse(_fs.default.readFileSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH, 'utf8')); // Check if the stored revision differs from the package.json revision

        const isUpgradedApp = _package.default.revision !== source.revision || _package.default.version !== source['app-version']; // Rebuild the registry file if revision or version fields are differents

        if (isUpgradedApp) {
          (0, _logger.log)('initialize:checkwazuhRegistry', 'Wazuh app revision or version changed, regenerating wazuh-version registry', 'info'); // Rebuild registry file in blank

          await saveConfiguration();
        }
      }
    } catch (error) {
      return Promise.reject(error);
    }
  }; // Init function. Check for "wazuh-version" document existance.


  const init = async () => {
    await checkWazuhRegistry();
  };

  const createKibanaTemplate = () => {
    (0, _logger.log)('initialize:createKibanaTemplate', `Creating template for ${PLUGIN_PLATFORM_INDEX}`, 'debug');

    try {
      _kibanaTemplate.pluginPlatformTemplate.template = PLUGIN_PLATFORM_INDEX + '*';
    } catch (error) {
      (0, _logger.log)('initialize:createKibanaTemplate', error.message || error);
      context.wazuh.logger.error('Exception: ' + error.message || error);
    }

    return context.core.elasticsearch.client.asInternalUser.indices.putTemplate({
      name: _constants.WAZUH_PLUGIN_PLATFORM_TEMPLATE_NAME,
      order: 0,
      create: true,
      body: _kibanaTemplate.pluginPlatformTemplate
    });
  };

  const createEmptyKibanaIndex = async () => {
    try {
      (0, _logger.log)('initialize:createEmptyKibanaIndex', `Creating ${PLUGIN_PLATFORM_INDEX} index.`, 'info');
      await context.core.elasticsearch.client.asInternalUser.indices.create({
        index: PLUGIN_PLATFORM_INDEX
      });
      (0, _logger.log)('initialize:createEmptyKibanaIndex', `Successfully created ${PLUGIN_PLATFORM_INDEX} index.`, 'debug');
      await init();
      return;
    } catch (error) {
      return Promise.reject(new Error(`Error creating ${PLUGIN_PLATFORM_INDEX} index due to ${error.message || error}`));
    }
  };

  const fixKibanaTemplate = async () => {
    try {
      await createKibanaTemplate();
      (0, _logger.log)('initialize:fixKibanaTemplate', `Successfully created ${PLUGIN_PLATFORM_INDEX} template.`, 'debug');
      await createEmptyKibanaIndex();
      return;
    } catch (error) {
      return Promise.reject(new Error(`Error creating template for ${PLUGIN_PLATFORM_INDEX} due to ${error.message || error}`));
    }
  };

  const getTemplateByName = async () => {
    try {
      await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
        name: _constants.WAZUH_PLUGIN_PLATFORM_TEMPLATE_NAME
      });
      (0, _logger.log)('initialize:getTemplateByName', `No need to create the ${PLUGIN_PLATFORM_INDEX} template, already exists.`, 'debug');
      await createEmptyKibanaIndex();
      return;
    } catch (error) {
      (0, _logger.log)('initialize:getTemplateByName', error.message || error);
      return fixKibanaTemplate();
    }
  }; // Does Kibana index exist?


  const checkKibanaStatus = async () => {
    try {
      const response = await context.core.elasticsearch.client.asInternalUser.indices.exists({
        index: PLUGIN_PLATFORM_INDEX
      });

      if (response.body) {
        // It exists, initialize!
        await init();
      } else {
        // No Kibana index created...
        (0, _logger.log)('initialize:checkKibanaStatus', `Not found ${PLUGIN_PLATFORM_INDEX} index`, 'info');
        await getTemplateByName();
      }
    } catch (error) {
      (0, _logger.log)('initialize:checkKibanaStatus', error.message || error);
      context.wazuh.logger.error(error.message || error);
    }
  }; // Wait until Elasticsearch js is ready


  const checkStatus = async () => {
    try {
      // TODO: wait until elasticsearch is ready?
      // await server.plugins.elasticsearch.waitUntilReady();
      return await checkKibanaStatus();
    } catch (error) {
      (0, _logger.log)('initialize:checkStatus', 'Waiting for elasticsearch plugin to be ready...', 'debug');
      setTimeout(() => checkStatus(), 3000);
    }
  }; // Check Kibana index and if it is prepared, start the initialization of Wazuh App.


  return checkStatus();
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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