"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhReportingCtrl = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _path = _interopRequireDefault(require("path"));

var _fs = _interopRequireDefault(require("fs"));

var _wazuhModules = require("../../common/wazuh-modules");

var TimSort = _interopRequireWildcard(require("timsort"));

var _errorResponse = require("../lib/error-response");

var VulnerabilityRequest = _interopRequireWildcard(require("../lib/reporting/vulnerability-request"));

var OverviewRequest = _interopRequireWildcard(require("../lib/reporting/overview-request"));

var RootcheckRequest = _interopRequireWildcard(require("../lib/reporting/rootcheck-request"));

var PCIRequest = _interopRequireWildcard(require("../lib/reporting/pci-request"));

var GDPRRequest = _interopRequireWildcard(require("../lib/reporting/gdpr-request"));

var TSCRequest = _interopRequireWildcard(require("../lib/reporting/tsc-request"));

var AuditRequest = _interopRequireWildcard(require("../lib/reporting/audit-request"));

var SyscheckRequest = _interopRequireWildcard(require("../lib/reporting/syscheck-request"));

var _pciRequirementsPdfmake = _interopRequireDefault(require("../integration-files/pci-requirements-pdfmake"));

var _gdprRequirementsPdfmake = _interopRequireDefault(require("../integration-files/gdpr-requirements-pdfmake"));

var _tscRequirementsPdfmake = _interopRequireDefault(require("../integration-files/tsc-requirements-pdfmake"));

var _processStateEquivalence = _interopRequireDefault(require("../lib/process-state-equivalence"));

var _csvKeyEquivalence = require("../../common/csv-key-equivalence");

var _agentConfiguration = require("../lib/reporting/agent-configuration");

var _printer = require("../lib/reporting/printer");

var _logger = require("../lib/logger");

var _constants = require("../../common/constants");

var _filesystem = require("../lib/filesystem");

var _moment = _interopRequireDefault(require("moment"));

var _wz_agent_status = require("../../common/services/wz_agent_status");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Wazuh app - Class for Wazuh reporting controller
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
class WazuhReportingCtrl {
  constructor() {
    (0, _defineProperty2.default)(this, "createReportsModules", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsModules', `Report started`, 'info');
        const {
          array,
          agents,
          browserTimezone,
          searchBar,
          filters,
          time,
          tables,
          section,
          indexPatternTitle,
          apiId
        } = request.body;
        const {
          moduleID
        } = request.params;
        const {
          from,
          to
        } = time || {}; // Init

        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        await this.renderHeader(context, printer, section, moduleID, agents, apiId);
        const [sanitizedFilters, agentsFilter] = filters ? this.sanitizeKibanaFilters(filters, searchBar) : [false, false];

        if (time && sanitizedFilters) {
          printer.addTimeRangeAndFilters(from, to, sanitizedFilters, browserTimezone);
        }

        if (time) {
          await this.extendedInformation(context, printer, section, moduleID, apiId, new Date(from).getTime(), new Date(to).getTime(), sanitizedFilters, indexPatternTitle, agents);
        }

        printer.addVisualizations(array, agents, moduleID);

        if (tables) {
          printer.addTables(tables);
        } //add authorized agents


        if (agentsFilter) {
          printer.addAgentsFilters(agentsFilter);
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      body: {
        agents
      },
      params: {
        moduleID
      }
    }) => `wazuh-module-${agents ? `agents-${agents}` : 'overview'}-${moduleID}-${this.generateReportTimestamp()}.pdf`));
    (0, _defineProperty2.default)(this, "createReportsGroups", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsGroups', `Report started`, 'info');
        const {
          components,
          apiId
        } = request.body;
        const {
          groupID
        } = request.params; // Init

        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        let tables = [];
        const equivalences = {
          localfile: 'Local files',
          osquery: 'Osquery',
          command: 'Command',
          syscheck: 'Syscheck',
          'open-scap': 'OpenSCAP',
          'cis-cat': 'CIS-CAT',
          syscollector: 'Syscollector',
          rootcheck: 'Rootcheck',
          labels: 'Labels',
          sca: 'Security configuration assessment'
        };
        printer.addContent({
          text: `Group ${groupID} configuration`,
          style: 'h1'
        });

        if (components['0']) {
          let configuration = {};

          try {
            const configurationResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${groupID}/configuration`, {}, {
              apiHostID: apiId
            });
            configuration = configurationResponse.data.data;
          } catch (error) {
            (0, _logger.log)('reporting:createReportsGroups', error.message || error, 'debug');
          }

          if (configuration.affected_items.length > 0 && Object.keys(configuration.affected_items[0].config).length) {
            printer.addContent({
              text: 'Configurations',
              style: {
                fontSize: 14,
                color: '#000'
              },
              margin: [0, 10, 0, 15]
            });
            const section = {
              labels: [],
              isGroupConfig: true
            };

            for (let config of configuration.affected_items) {
              let filterTitle = '';
              let index = 0;

              for (let filter of Object.keys(config.filters)) {
                filterTitle = filterTitle.concat(`${filter}: ${config.filters[filter]}`);

                if (index < Object.keys(config.filters).length - 1) {
                  filterTitle = filterTitle.concat(' | ');
                }

                index++;
              }

              printer.addContent({
                text: filterTitle,
                style: 'h4',
                margin: [0, 0, 0, 10]
              });
              let idx = 0;
              section.tabs = [];

              for (let _d of Object.keys(config.config)) {
                for (let c of _agentConfiguration.AgentConfiguration.configurations) {
                  for (let s of c.sections) {
                    section.opts = s.opts || {};

                    for (let cn of s.config || []) {
                      if (cn.configuration === _d) {
                        section.labels = s.labels || [[]];
                      }
                    }

                    for (let wo of s.wodle || []) {
                      if (wo.name === _d) {
                        section.labels = s.labels || [[]];
                      }
                    }
                  }
                }

                section.labels[0]['pack'] = 'Packs';
                section.labels[0]['content'] = 'Evaluations';
                section.labels[0]['7'] = 'Scan listening netwotk ports';
                section.tabs.push(equivalences[_d]);

                if (Array.isArray(config.config[_d])) {
                  /* LOG COLLECTOR */
                  if (_d === 'localfile') {
                    let groups = [];

                    config.config[_d].forEach(obj => {
                      if (!groups[obj.logformat]) {
                        groups[obj.logformat] = [];
                      }

                      groups[obj.logformat].push(obj);
                    });

                    Object.keys(groups).forEach(group => {
                      let saveidx = 0;
                      groups[group].forEach((x, i) => {
                        if (Object.keys(x).length > Object.keys(groups[group][saveidx]).length) {
                          saveidx = i;
                        }
                      });
                      const columns = Object.keys(groups[group][saveidx]);
                      const rows = groups[group].map(x => {
                        let row = [];
                        columns.forEach(key => {
                          row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
                            return x + '\n';
                          }) : JSON.stringify(x[key]));
                        });
                        return row;
                      });
                      columns.forEach((col, i) => {
                        columns[i] = col[0].toUpperCase() + col.slice(1);
                      });
                      tables.push({
                        title: 'Local files',
                        type: 'table',
                        columns,
                        rows
                      });
                    });
                  } else if (_d === 'labels') {
                    const obj = config.config[_d][0].label;
                    const columns = Object.keys(obj[0]);

                    if (!columns.includes('hidden')) {
                      columns.push('hidden');
                    }

                    const rows = obj.map(x => {
                      let row = [];
                      columns.forEach(key => {
                        row.push(x[key]);
                      });
                      return row;
                    });
                    columns.forEach((col, i) => {
                      columns[i] = col[0].toUpperCase() + col.slice(1);
                    });
                    tables.push({
                      title: 'Labels',
                      type: 'table',
                      columns,
                      rows
                    });
                  } else {
                    for (let _d2 of config.config[_d]) {
                      tables.push(...this.getConfigTables(_d2, section, idx));
                    }
                  }
                } else {
                  /*INTEGRITY MONITORING MONITORED DIRECTORIES */
                  if (config.config[_d].directories) {
                    const directories = config.config[_d].directories;
                    delete config.config[_d].directories;
                    tables.push(...this.getConfigTables(config.config[_d], section, idx));
                    let diffOpts = [];
                    Object.keys(section.opts).forEach(x => {
                      diffOpts.push(x);
                    });
                    const columns = ['', ...diffOpts.filter(x => x !== 'check_all' && x !== 'check_sum')];
                    let rows = [];
                    directories.forEach(x => {
                      let row = [];
                      row.push(x.path);
                      columns.forEach(y => {
                        if (y !== '') {
                          y = y !== 'check_whodata' ? y : 'whodata';
                          row.push(x[y] ? x[y] : 'no');
                        }
                      });
                      row.push(x.recursion_level);
                      rows.push(row);
                    });
                    columns.forEach((x, idx) => {
                      columns[idx] = section.opts[x];
                    });
                    columns.push('RL');
                    tables.push({
                      title: 'Monitored directories',
                      type: 'table',
                      columns,
                      rows
                    });
                  } else {
                    tables.push(...this.getConfigTables(config.config[_d], section, idx));
                  }
                }

                for (const table of tables) {
                  printer.addConfigTables([table]);
                }

                idx++;
                tables = [];
              }

              tables = [];
            }
          } else {
            printer.addContent({
              text: 'A configuration for this group has not yet been set up.',
              style: {
                fontSize: 12,
                color: '#000'
              },
              margin: [0, 10, 0, 15]
            });
          }
        }

        if (components['1']) {
          let agentsInGroup = [];

          try {
            const agentsInGroupResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${groupID}/agents`, {}, {
              apiHostID: apiId
            });
            agentsInGroup = agentsInGroupResponse.data.data.affected_items;
          } catch (error) {
            (0, _logger.log)('reporting:report', error.message || error, 'debug');
          }

          await this.renderHeader(context, printer, 'groupConfig', groupID, (agentsInGroup || []).map(x => x.id), apiId);
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsGroups', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        groupID
      }
    }) => `wazuh-group-configuration-${groupID}-${this.generateReportTimestamp()}.pdf`));
    (0, _defineProperty2.default)(this, "createReportsAgentsConfiguration", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Report started`, 'info');
        const {
          components,
          apiId
        } = request.body;
        const {
          agentID
        } = request.params;
        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        let wmodulesResponse = {};
        let tables = [];

        try {
          wmodulesResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents/${agentID}/config/wmodules/wmodules`, {}, {
            apiHostID: apiId
          });
        } catch (error) {
          (0, _logger.log)('reporting:report', error.message || error, 'debug');
        }

        await this.renderHeader(context, printer, 'agentConfig', 'agentConfig', agentID, apiId);
        let idxComponent = 0;

        for (let config of _agentConfiguration.AgentConfiguration.configurations) {
          let titleOfSection = false;
          (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Iterate over ${config.sections.length} configuration sections`, 'debug');

          for (let section of config.sections) {
            let titleOfSubsection = false;

            if (components[idxComponent] && (section.config || section.wodle)) {
              let idx = 0;
              const configs = (section.config || []).concat(section.wodle || []);
              (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Iterate over ${configs.length} configuration blocks`, 'debug');

              for (let conf of configs) {
                let agentConfigResponse = {};

                try {
                  if (!conf['name']) {
                    agentConfigResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents/${agentID}/config/${conf.component}/${conf.configuration}`, {}, {
                      apiHostID: apiId
                    });
                  } else {
                    for (let wodle of wmodulesResponse.data.data['wmodules']) {
                      if (Object.keys(wodle)[0] === conf['name']) {
                        agentConfigResponse.data = {
                          data: wodle
                        };
                      }
                    }
                  }

                  const agentConfig = agentConfigResponse && agentConfigResponse.data && agentConfigResponse.data.data;

                  if (!titleOfSection) {
                    printer.addContent({
                      text: config.title,
                      style: 'h1',
                      margin: [0, 0, 0, 15]
                    });
                    titleOfSection = true;
                  }

                  if (!titleOfSubsection) {
                    printer.addContent({
                      text: section.subtitle,
                      style: 'h4'
                    });
                    printer.addContent({
                      text: section.desc,
                      style: {
                        fontSize: 12,
                        color: '#000'
                      },
                      margin: [0, 0, 0, 10]
                    });
                    titleOfSubsection = true;
                  }

                  if (agentConfig) {
                    for (let agentConfigKey of Object.keys(agentConfig)) {
                      if (Array.isArray(agentConfig[agentConfigKey])) {
                        /* LOG COLLECTOR */
                        if (conf.filterBy) {
                          let groups = [];
                          agentConfig[agentConfigKey].forEach(obj => {
                            if (!groups[obj.logformat]) {
                              groups[obj.logformat] = [];
                            }

                            groups[obj.logformat].push(obj);
                          });
                          Object.keys(groups).forEach(group => {
                            let saveidx = 0;
                            groups[group].forEach((x, i) => {
                              if (Object.keys(x).length > Object.keys(groups[group][saveidx]).length) {
                                saveidx = i;
                              }
                            });
                            const columns = Object.keys(groups[group][saveidx]);
                            const rows = groups[group].map(x => {
                              let row = [];
                              columns.forEach(key => {
                                row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
                                  return x + '\n';
                                }) : JSON.stringify(x[key]));
                              });
                              return row;
                            });
                            columns.forEach((col, i) => {
                              columns[i] = col[0].toUpperCase() + col.slice(1);
                            });
                            tables.push({
                              title: section.labels[0][group],
                              type: 'table',
                              columns,
                              rows
                            });
                          });
                        } else if (agentConfigKey.configuration !== 'socket') {
                          tables.push(...this.getConfigTables(agentConfig[agentConfigKey], section, idx));
                        } else {
                          for (let _d2 of agentConfig[agentConfigKey]) {
                            tables.push(...this.getConfigTables(_d2, section, idx));
                          }
                        }
                      } else {
                        /*INTEGRITY MONITORING MONITORED DIRECTORIES */
                        if (conf.matrix) {
                          const {
                            directories,
                            diff,
                            synchronization,
                            file_limit,
                            ...rest
                          } = agentConfig[agentConfigKey];
                          tables.push(...this.getConfigTables(rest, section, idx), ...(diff && diff.disk_quota ? this.getConfigTables(diff.disk_quota, {
                            tabs: ['Disk quota']
                          }, 0) : []), ...(diff && diff.file_size ? this.getConfigTables(diff.file_size, {
                            tabs: ['File size']
                          }, 0) : []), ...(synchronization ? this.getConfigTables(synchronization, {
                            tabs: ['Synchronization']
                          }, 0) : []), ...(file_limit ? this.getConfigTables(file_limit, {
                            tabs: ['File limit']
                          }, 0) : []));
                          let diffOpts = [];
                          Object.keys(section.opts).forEach(x => {
                            diffOpts.push(x);
                          });
                          const columns = ['', ...diffOpts.filter(x => x !== 'check_all' && x !== 'check_sum')];
                          let rows = [];
                          directories.forEach(x => {
                            let row = [];
                            row.push(x.dir);
                            columns.forEach(y => {
                              if (y !== '') {
                                row.push(x.opts.indexOf(y) > -1 ? 'yes' : 'no');
                              }
                            });
                            row.push(x.recursion_level);
                            rows.push(row);
                          });
                          columns.forEach((x, idx) => {
                            columns[idx] = section.opts[x];
                          });
                          columns.push('RL');
                          tables.push({
                            title: 'Monitored directories',
                            type: 'table',
                            columns,
                            rows
                          });
                        } else {
                          tables.push(...this.getConfigTables(agentConfig[agentConfigKey], section, idx));
                        }
                      }
                    }
                  } else {
                    // Print no configured module and link to the documentation
                    printer.addContent({
                      text: ['This module is not configured. Please take a look on how to configure it in ', {
                        text: `${section.subtitle.toLowerCase()} configuration.`,
                        link: section.docuLink,
                        style: {
                          fontSize: 12,
                          color: '#1a0dab'
                        }
                      }],
                      margin: [0, 0, 0, 20]
                    });
                  }
                } catch (error) {
                  (0, _logger.log)('reporting:report', error.message || error, 'debug');
                }

                idx++;
              }

              for (const table of tables) {
                printer.addConfigTables([table]);
              }
            }

            idxComponent++;
            tables = [];
          }
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsAgentsConfiguration', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        agentID
      }
    }) => `wazuh-agent-configuration-${agentID}-${this.generateReportTimestamp()}.pdf`));
    (0, _defineProperty2.default)(this, "createReportsAgentsInventory", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsAgentsInventory', `Report started`, 'info');
        const {
          searchBar,
          filters,
          time,
          indexPatternTitle,
          apiId
        } = request.body;
        const {
          agentID
        } = request.params;
        const {
          from,
          to
        } = time || {}; // Init

        const printer = new _printer.ReportPrinter();
        const {
          hashUsername
        } = await context.wazuh.security.getCurrentUser(request, context);
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername));
        (0, _logger.log)('reporting:createReportsAgentsInventory', `Syscollector report`, 'debug');
        const sanitizedFilters = filters ? this.sanitizeKibanaFilters(filters, searchBar) : false; // Get the agent OS

        let agentOs = '';

        try {
          const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', '/agents', {
            params: {
              q: `id=${agentID}`
            }
          }, {
            apiHostID: apiId
          });
          agentOs = agentResponse.data.data.affected_items[0].os.platform;
        } catch (error) {
          (0, _logger.log)('reporting:createReportsAgentsInventory', error.message || error, 'debug');
        } // Add title


        printer.addContentWithNewLine({
          text: 'Inventory data report',
          style: 'h1'
        }); // Add table with the agent info

        await this.buildAgentsTable(context, printer, [agentID], apiId); // Get syscollector packages and processes

        const agentRequestsInventory = [{
          endpoint: `/syscollector/${agentID}/packages`,
          loggerMessage: `Fetching packages for agent ${agentID}`,
          table: {
            title: 'Packages',
            columns: agentOs === 'windows' ? [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'architecture',
              label: 'Architecture'
            }, {
              id: 'version',
              label: 'Version'
            }, {
              id: 'vendor',
              label: 'Vendor'
            }] : [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'architecture',
              label: 'Architecture'
            }, {
              id: 'version',
              label: 'Version'
            }, {
              id: 'vendor',
              label: 'Vendor'
            }, {
              id: 'description',
              label: 'Description'
            }]
          }
        }, {
          endpoint: `/syscollector/${agentID}/processes`,
          loggerMessage: `Fetching processes for agent ${agentID}`,
          table: {
            title: 'Processes',
            columns: agentOs === 'windows' ? [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'cmd',
              label: 'CMD'
            }, {
              id: 'priority',
              label: 'Priority'
            }, {
              id: 'nlwp',
              label: 'NLWP'
            }] : [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'euser',
              label: 'Effective user'
            }, {
              id: 'nice',
              label: 'Priority'
            }, {
              id: 'state',
              label: 'State'
            }]
          },
          mapResponseItems: item => agentOs === 'windows' ? item : { ...item,
            state: _processStateEquivalence.default[item.state]
          }
        }, {
          endpoint: `/syscollector/${agentID}/ports`,
          loggerMessage: `Fetching ports for agent ${agentID}`,
          table: {
            title: 'Network ports',
            columns: agentOs === 'windows' ? [{
              id: 'local_ip',
              label: 'Local IP'
            }, {
              id: 'local_port',
              label: 'Local port'
            }, {
              id: 'process',
              label: 'Process'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'protocol',
              label: 'Protocol'
            }] : [{
              id: 'local_ip',
              label: 'Local IP'
            }, {
              id: 'local_port',
              label: 'Local port'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'protocol',
              label: 'Protocol'
            }]
          },
          mapResponseItems: item => ({ ...item,
            local_ip: item.local.ip,
            local_port: item.local.port
          })
        }, {
          endpoint: `/syscollector/${agentID}/netiface`,
          loggerMessage: `Fetching netiface for agent ${agentID}`,
          table: {
            title: 'Network interfaces',
            columns: [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'mac',
              label: 'Mac'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'mtu',
              label: 'MTU'
            }, {
              id: 'type',
              label: 'Type'
            }]
          }
        }, {
          endpoint: `/syscollector/${agentID}/netaddr`,
          loggerMessage: `Fetching netaddr for agent ${agentID}`,
          table: {
            title: 'Network settings',
            columns: [{
              id: 'iface',
              label: 'Interface'
            }, {
              id: 'address',
              label: 'address'
            }, {
              id: 'netmask',
              label: 'Netmask'
            }, {
              id: 'proto',
              label: 'Protocol'
            }, {
              id: 'broadcast',
              label: 'Broadcast'
            }]
          }
        }];
        agentOs === 'windows' && agentRequestsInventory.push({
          endpoint: `/syscollector/${agentID}/hotfixes`,
          loggerMessage: `Fetching hotfixes for agent ${agentID}`,
          table: {
            title: 'Windows updates',
            columns: [{
              id: 'hotfix',
              label: 'Update code'
            }]
          }
        });

        const requestInventory = async agentRequestInventory => {
          try {
            (0, _logger.log)('reporting:createReportsAgentsInventory', agentRequestInventory.loggerMessage, 'debug');
            const inventoryResponse = await context.wazuh.api.client.asCurrentUser.request('GET', agentRequestInventory.endpoint, {}, {
              apiHostID: apiId
            });
            const inventory = inventoryResponse && inventoryResponse.data && inventoryResponse.data.data && inventoryResponse.data.data.affected_items;

            if (inventory) {
              return { ...agentRequestInventory.table,
                items: agentRequestInventory.mapResponseItems ? inventory.map(agentRequestInventory.mapResponseItems) : inventory
              };
            }
          } catch (error) {
            (0, _logger.log)('reporting:createReportsAgentsInventory', error.message || error, 'debug');
          }
        };

        if (time) {
          await this.extendedInformation(context, printer, 'agents', 'syscollector', apiId, from, to, sanitizedFilters + ' AND rule.groups: "vulnerability-detector"', indexPatternTitle, agentID);
        } // Add inventory tables


        (await Promise.all(agentRequestsInventory.map(requestInventory))).filter(table => table).forEach(table => printer.addSimpleTable(table)); // Print the document

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsAgents', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        agentID
      }
    }) => `wazuh-agent-inventory-${agentID}-${this.generateReportTimestamp()}.pdf`));
    (0, _defineProperty2.default)(this, "getReportByName", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:getReportByName', `Getting ${context.wazuhEndpointParams.pathFilename} report`, 'debug');

        const reportFileBuffer = _fs.default.readFileSync(context.wazuhEndpointParams.pathFilename);

        return response.ok({
          headers: {
            'Content-Type': 'application/pdf'
          },
          body: reportFileBuffer
        });
      } catch (error) {
        (0, _logger.log)('reporting:getReportByName', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5030, 500, response);
      }
    }, request => request.params.name));
    (0, _defineProperty2.default)(this, "deleteReportByName", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:deleteReportByName', `Deleting ${context.wazuhEndpointParams.pathFilename} report`, 'debug');

        _fs.default.unlinkSync(context.wazuhEndpointParams.pathFilename);

        (0, _logger.log)('reporting:deleteReportByName', `${context.wazuhEndpointParams.pathFilename} report was deleted`, 'info');
        return response.ok({
          body: {
            error: 0
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:deleteReportByName', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5032, 500, response);
      }
    }, request => request.params.name));
  }
  /**
   * This do format to filters
   * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
   * @param {String} searchBar search term
   */


  sanitizeKibanaFilters(filters, searchBar) {
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `Started to sanitize filters`, 'info');
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `filters: ${filters.length}, searchBar: ${searchBar}`, 'debug');
    let str = '';
    const agentsFilter = []; //separate agents filter

    filters = filters.filter(filter => {
      if (filter.meta.controlledBy === _constants.AUTHORIZED_AGENTS) {
        agentsFilter.push(filter);
        return false;
      }

      return filter;
    });
    const len = filters.length;

    for (let i = 0; i < len; i++) {
      const {
        negate,
        key,
        value,
        params,
        type
      } = filters[i].meta;
      str += `${negate ? 'NOT ' : ''}`;
      str += `${key}: `;
      str += `${type === 'range' ? `${params.gte}-${params.lt}` : type === 'phrases' ? '(' + params.join(" OR ") + ')' : type === 'exists' ? '*' : !!value ? value : (params || {}).query}`;
      str += `${i === len - 1 ? '' : ' AND '}`;
    }

    if (searchBar) {
      str += ` AND (${searchBar})`;
    }

    const agentsFilterStr = agentsFilter.map(filter => filter.meta.value).join(',');
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `str: ${str}, agentsFilterStr: ${agentsFilterStr}`, 'debug');
    return [str, agentsFilterStr];
  }
  /**
   * This performs the rendering of given header
   * @param {String} printer section target
   * @param {String} section section target
   * @param {Object} tab tab target
   * @param {Boolean} isAgents is agents section
   * @param {String} apiId ID of API
   */


  async renderHeader(context, printer, section, tab, isAgents, apiId) {
    try {
      (0, _logger.log)('reporting:renderHeader', `section: ${section}, tab: ${tab}, isAgents: ${isAgents}, apiId: ${apiId}`, 'debug');

      if (section && typeof section === 'string') {
        if (!['agentConfig', 'groupConfig'].includes(section)) {
          printer.addContent({
            text: _wazuhModules.WAZUH_MODULES[tab].title + ' report',
            style: 'h1'
          });
        } else if (section === 'agentConfig') {
          printer.addContent({
            text: `Agent ${isAgents} configuration`,
            style: 'h1'
          });
        } else if (section === 'groupConfig') {
          printer.addContent({
            text: 'Agents in group',
            style: {
              fontSize: 14,
              color: '#000'
            },
            margin: [0, 20, 0, 0]
          });

          if (section === 'groupConfig' && !Object.keys(isAgents).length) {
            printer.addContent({
              text: 'There are still no agents in this group.',
              style: {
                fontSize: 12,
                color: '#000'
              },
              margin: [0, 10, 0, 0]
            });
          }
        }

        printer.addNewLine();
      }

      if (isAgents && typeof isAgents === 'object') {
        await this.buildAgentsTable(context, printer, isAgents, apiId, section === 'groupConfig' ? tab : false);
      }

      if (isAgents && typeof isAgents === 'string') {
        const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents`, {
          params: {
            agents_list: isAgents
          }
        }, {
          apiHostID: apiId
        });
        const agentData = agentResponse.data.data.affected_items[0];

        if (agentData && agentData.status !== _constants.API_NAME_AGENT_STATUS.ACTIVE) {
          printer.addContentWithNewLine({
            text: `Warning. Agent is ${(0, _wz_agent_status.agentStatusLabelByAgentStatus)(agentData.status).toLowerCase()}`,
            style: 'standard'
          });
        }

        await this.buildAgentsTable(context, printer, [isAgents], apiId);

        if (agentData && agentData.group) {
          const agentGroups = agentData.group.join(', ');
          printer.addContentWithNewLine({
            text: `Group${agentData.group.length > 1 ? 's' : ''}: ${agentGroups}`,
            style: 'standard'
          });
        }
      }

      if (_wazuhModules.WAZUH_MODULES[tab] && _wazuhModules.WAZUH_MODULES[tab].description) {
        printer.addContentWithNewLine({
          text: _wazuhModules.WAZUH_MODULES[tab].description,
          style: 'standard'
        });
      }

      return;
    } catch (error) {
      (0, _logger.log)('reporting:renderHeader', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This build the agents table
   * @param {Array<Strings>} ids ids of agents
   * @param {String} apiId API id
   */


  async buildAgentsTable(context, printer, agentIDs, apiId, multi = false) {
    const dateFormat = await context.core.uiSettings.client.get('dateFormat');
    if (!agentIDs || !agentIDs.length) return;
    (0, _logger.log)('reporting:buildAgentsTable', `${agentIDs.length} agents for API ${apiId}`, 'info');

    try {
      let agentRows = [];

      if (multi) {
        try {
          const agentsResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${multi}/agents`, {}, {
            apiHostID: apiId
          });
          const agentsData = agentsResponse && agentsResponse.data && agentsResponse.data.data && agentsResponse.data.data.affected_items;
          agentRows = (agentsData || []).map(agent => ({ ...agent,
            manager: agent.manager || agent.manager_host,
            os: agent.os && agent.os.name && agent.os.version ? `${agent.os.name} ${agent.os.version}` : ''
          }));
        } catch (error) {
          (0, _logger.log)('reporting:buildAgentsTable', `Skip agent due to: ${error.message || error}`, 'debug');
        }
      } else {
        for (const agentID of agentIDs) {
          try {
            const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents`, {
              params: {
                q: `id=${agentID}`
              }
            }, {
              apiHostID: apiId
            });
            const [agent] = agentResponse.data.data.affected_items;
            agentRows.push({ ...agent,
              manager: agent.manager || agent.manager_host,
              os: agent.os && agent.os.name && agent.os.version ? `${agent.os.name} ${agent.os.version}` : '',
              lastKeepAlive: (0, _moment.default)(agent.lastKeepAlive).format(dateFormat),
              dateAdd: (0, _moment.default)(agent.dateAdd).format(dateFormat)
            });
          } catch (error) {
            (0, _logger.log)('reporting:buildAgentsTable', `Skip agent due to: ${error.message || error}`, 'debug');
          }
        }
      }

      printer.addSimpleTable({
        columns: [{
          id: 'id',
          label: 'ID'
        }, {
          id: 'name',
          label: 'Name'
        }, {
          id: 'ip',
          label: 'IP'
        }, {
          id: 'version',
          label: 'Version'
        }, {
          id: 'manager',
          label: 'Manager'
        }, {
          id: 'os',
          label: 'OS'
        }, {
          id: 'dateAdd',
          label: 'Registration date'
        }, {
          id: 'lastKeepAlive',
          label: 'Last keep alive'
        }],
        items: agentRows
      });
    } catch (error) {
      (0, _logger.log)('reporting:buildAgentsTable', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This load more information
   * @param {*} context Endpoint context
   * @param {*} printer printer instance
   * @param {String} section section target
   * @param {Object} tab tab target
   * @param {String} apiId ID of API
   * @param {Number} from Timestamp (ms) from
   * @param {Number} to Timestamp (ms) to
   * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
   * @param {String} pattern
   * @param {Object} agent agent target
   * @returns {Object} Extended information
   */


  async extendedInformation(context, printer, section, tab, apiId, from, to, filters, pattern = _constants.WAZUH_ALERTS_PATTERN, agent = null) {
    try {
      (0, _logger.log)('reporting:extendedInformation', `Section ${section} and tab ${tab}, API is ${apiId}. From ${from} to ${to}. Filters ${filters}. Index pattern ${pattern}`, 'info');

      if (section === 'agents' && !agent) {
        throw new Error('Reporting for specific agent needs an agent ID in order to work properly');
      }

      const agents = await context.wazuh.api.client.asCurrentUser.request('GET', '/agents', {
        params: {
          limit: 1
        }
      }, {
        apiHostID: apiId
      });
      const totalAgents = agents.data.data.total_affected_items;

      if (section === 'overview' && tab === 'vuls') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector metrics', 'debug');
        const vulnerabilitiesLevels = ['Low', 'Medium', 'High', 'Critical'];
        const vulnerabilitiesResponsesCount = (await Promise.all(vulnerabilitiesLevels.map(async vulnerabilitiesLevel => {
          try {
            const count = await VulnerabilityRequest.uniqueSeverityCount(context, from, to, vulnerabilitiesLevel, filters, pattern);
            return count ? `${count} of ${totalAgents} agents have ${vulnerabilitiesLevel.toLocaleLowerCase()} vulnerabilities.` : undefined;
          } catch (error) {}
        }))).filter(vulnerabilitiesResponse => vulnerabilitiesResponse);
        printer.addList({
          title: {
            text: 'Summary',
            style: 'h2'
          },
          list: vulnerabilitiesResponsesCount
        });
        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector top 3 agents by category', 'debug');
        const lowRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Low', filters, pattern);
        const mediumRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Medium', filters, pattern);
        const highRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'High', filters, pattern);
        const criticalRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Critical', filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding overview vulnerability detector top 3 agents by category', 'debug');

        if (criticalRank && criticalRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with critical severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, criticalRank, apiId);
          printer.addNewLine();
        }

        if (highRank && highRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with high severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, highRank, apiId);
          printer.addNewLine();
        }

        if (mediumRank && mediumRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with medium severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, mediumRank, apiId);
          printer.addNewLine();
        }

        if (lowRank && lowRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with low severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, lowRank, apiId);
          printer.addNewLine();
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector top 3 CVEs', 'debug');
        const cveRank = await VulnerabilityRequest.topCVECount(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding overview vulnerability detector top 3 CVEs', 'debug');

        if (cveRank && cveRank.length) {
          printer.addSimpleTable({
            title: {
              text: 'Top 3 CVE',
              style: 'h2'
            },
            columns: [{
              id: 'top',
              label: 'Top'
            }, {
              id: 'cve',
              label: 'CVE'
            }],
            items: cveRank.map(item => ({
              top: cveRank.indexOf(item) + 1,
              cve: item
            }))
          });
        }
      }

      if (section === 'overview' && tab === 'general') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 agents with level 15 alerts', 'debug');
        const level15Rank = await OverviewRequest.topLevel15(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding top 3 agents with level 15 alerts', 'debug');

        if (level15Rank.length) {
          printer.addContent({
            text: 'Top 3 agents with level 15 alerts',
            style: 'h2'
          });
          await this.buildAgentsTable(context, printer, level15Rank, apiId);
        }
      }

      if (section === 'overview' && tab === 'pm') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching most common rootkits', 'debug');
        const top5RootkitsRank = await RootcheckRequest.top5RootkitsDetected(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding most common rootkits', 'debug');

        if (top5RootkitsRank && top5RootkitsRank.length) {
          printer.addContentWithNewLine({
            text: 'Most common rootkits found among your agents',
            style: 'h2'
          }).addContentWithNewLine({
            text: 'Rootkits are a set of software tools that enable an unauthorized user to gain control of a computer system without being detected.',
            style: 'standard'
          }).addSimpleTable({
            items: top5RootkitsRank.map(item => {
              return {
                top: top5RootkitsRank.indexOf(item) + 1,
                name: item
              };
            }),
            columns: [{
              id: 'top',
              label: 'Top'
            }, {
              id: 'name',
              label: 'Rootkit'
            }]
          });
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching hidden pids', 'debug');
        const hiddenPids = await RootcheckRequest.agentsWithHiddenPids(context, from, to, filters, pattern);
        hiddenPids && printer.addContent({
          text: `${hiddenPids} of ${totalAgents} agents have hidden processes`,
          style: 'h3'
        });
        !hiddenPids && printer.addContentWithNewLine({
          text: `No agents have hidden processes`,
          style: 'h3'
        });
        const hiddenPorts = await RootcheckRequest.agentsWithHiddenPorts(context, from, to, filters, pattern);
        hiddenPorts && printer.addContent({
          text: `${hiddenPorts} of ${totalAgents} agents have hidden ports`,
          style: 'h3'
        });
        !hiddenPorts && printer.addContent({
          text: `No agents have hidden ports`,
          style: 'h3'
        });
        printer.addNewLine();
      }

      if (['overview', 'agents'].includes(section) && tab === 'pci') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top PCI DSS requirements', 'debug');
        const topPciRequirements = await PCIRequest.topPCIRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common PCI DSS requirements alerts found',
          style: 'h2'
        });

        for (const item of topPciRequirements) {
          const rules = await PCIRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_pciRequirementsPdfmake.default[item]) {
            const content = typeof _pciRequirementsPdfmake.default[item] === 'string' ? {
              text: _pciRequirementsPdfmake.default[item],
              style: 'standard'
            } : _pciRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }
      }

      if (['overview', 'agents'].includes(section) && tab === 'tsc') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top TSC requirements', 'debug');
        const topTSCRequirements = await TSCRequest.topTSCRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common TSC requirements alerts found',
          style: 'h2'
        });

        for (const item of topTSCRequirements) {
          const rules = await TSCRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_tscRequirementsPdfmake.default[item]) {
            const content = typeof _tscRequirementsPdfmake.default[item] === 'string' ? {
              text: _tscRequirementsPdfmake.default[item],
              style: 'standard'
            } : _tscRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }
      }

      if (['overview', 'agents'].includes(section) && tab === 'gdpr') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top GDPR requirements', 'debug');
        const topGdprRequirements = await GDPRRequest.topGDPRRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common GDPR requirements alerts found',
          style: 'h2'
        });

        for (const item of topGdprRequirements) {
          const rules = await GDPRRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_gdprRequirementsPdfmake.default && _gdprRequirementsPdfmake.default[item]) {
            const content = typeof _gdprRequirementsPdfmake.default[item] === 'string' ? {
              text: _gdprRequirementsPdfmake.default[item],
              style: 'standard'
            } : _gdprRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }

        printer.addNewLine();
      }

      if (section === 'overview' && tab === 'audit') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching agents with high number of failed sudo commands', 'debug');
        const auditAgentsNonSuccess = await AuditRequest.getTop3AgentsSudoNonSuccessful(context, from, to, filters, pattern);

        if (auditAgentsNonSuccess && auditAgentsNonSuccess.length) {
          printer.addContent({
            text: 'Agents with high number of failed sudo commands',
            style: 'h2'
          });
          await this.buildAgentsTable(context, printer, auditAgentsNonSuccess, apiId);
        }

        const auditAgentsFailedSyscall = await AuditRequest.getTop3AgentsFailedSyscalls(context, from, to, filters, pattern);

        if (auditAgentsFailedSyscall && auditAgentsFailedSyscall.length) {
          printer.addSimpleTable({
            columns: [{
              id: 'agent',
              label: 'Agent ID'
            }, {
              id: 'syscall_id',
              label: 'Syscall ID'
            }, {
              id: 'syscall_syscall',
              label: 'Syscall'
            }],
            items: auditAgentsFailedSyscall.map(item => ({
              agent: item.agent,
              syscall_id: item.syscall.id,
              syscall_syscall: item.syscall.syscall
            })),
            title: {
              text: 'Most common failing syscalls',
              style: 'h2'
            }
          });
        }
      }

      if (section === 'overview' && tab === 'fim') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 rules for FIM', 'debug');
        const rules = await SyscheckRequest.top3Rules(context, from, to, filters, pattern);

        if (rules && rules.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 FIM rules',
            style: 'h2'
          }).addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: {
              text: 'Top 3 rules that are generating most alerts.',
              style: 'standard'
            }
          });
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 agents for FIM', 'debug');
        const agents = await SyscheckRequest.top3agents(context, from, to, filters, pattern);

        if (agents && agents.length) {
          printer.addContentWithNewLine({
            text: 'Agents with suspicious FIM activity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'Top 3 agents that have most FIM alerts from level 7 to level 15. Take care about them.',
            style: 'standard'
          });
          await this.buildAgentsTable(context, printer, agents, apiId);
        }
      }

      if (section === 'agents' && tab === 'audit') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching most common failed syscalls`, 'debug');
        const auditFailedSyscall = await AuditRequest.getTopFailedSyscalls(context, from, to, filters, pattern);
        auditFailedSyscall && auditFailedSyscall.length && printer.addSimpleTable({
          columns: [{
            id: 'id',
            label: 'id'
          }, {
            id: 'syscall',
            label: 'Syscall'
          }],
          items: auditFailedSyscall,
          title: 'Most common failing syscalls'
        });
      }

      if (section === 'agents' && tab === 'fim') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching syscheck database for agent ${agent}`, 'debug');
        const lastScanResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/syscheck/${agent}/last_scan`, {}, {
          apiHostID: apiId
        });

        if (lastScanResponse && lastScanResponse.data) {
          const lastScanData = lastScanResponse.data.data.affected_items[0];

          if (lastScanData.start && lastScanData.end) {
            printer.addContent({
              text: `Last file integrity monitoring scan was executed from ${lastScanData.start} to ${lastScanData.end}.`
            });
          } else if (lastScanData.start) {
            printer.addContent({
              text: `File integrity monitoring scan is currently in progress for this agent (started on ${lastScanData.start}).`
            });
          } else {
            printer.addContent({
              text: `File integrity monitoring scan is currently in progress for this agent.`
            });
          }

          printer.addNewLine();
        }

        (0, _logger.log)('reporting:extendedInformation', `Fetching last 10 deleted files for FIM`, 'debug');
        const lastTenDeleted = await SyscheckRequest.lastTenDeletedFiles(context, from, to, filters, pattern);
        lastTenDeleted && lastTenDeleted.length && printer.addSimpleTable({
          columns: [{
            id: 'path',
            label: 'Path'
          }, {
            id: 'date',
            label: 'Date'
          }],
          items: lastTenDeleted,
          title: 'Last 10 deleted files'
        });
        (0, _logger.log)('reporting:extendedInformation', `Fetching last 10 modified files`, 'debug');
        const lastTenModified = await SyscheckRequest.lastTenModifiedFiles(context, from, to, filters, pattern);
        lastTenModified && lastTenModified.length && printer.addSimpleTable({
          columns: [{
            id: 'path',
            label: 'Path'
          }, {
            id: 'date',
            label: 'Date'
          }],
          items: lastTenModified,
          title: 'Last 10 modified files'
        });
      }

      if (section === 'agents' && tab === 'syscollector') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching hardware information for agent ${agent}`, 'debug');
        const requestsSyscollectorLists = [{
          endpoint: `/syscollector/${agent}/hardware`,
          loggerMessage: `Fetching Hardware information for agent ${agent}`,
          list: {
            title: {
              text: 'Hardware information',
              style: 'h2'
            }
          },
          mapResponse: hardware => [hardware.cpu && hardware.cpu.cores && `${hardware.cpu.cores} cores`, hardware.cpu && hardware.cpu.name, hardware.ram && hardware.ram.total && `${Number(hardware.ram.total / 1024 / 1024).toFixed(2)}GB RAM`]
        }, {
          endpoint: `/syscollector/${agent}/os`,
          loggerMessage: `Fetching OS information for agent ${agent}`,
          list: {
            title: {
              text: 'OS information',
              style: 'h2'
            }
          },
          mapResponse: osData => [osData.sysname, osData.version, osData.architecture, osData.release, osData.os && osData.os.name && osData.os.version && `${osData.os.name} ${osData.os.version}`]
        }];
        const syscollectorLists = await Promise.all(requestsSyscollectorLists.map(async requestSyscollector => {
          try {
            (0, _logger.log)('reporting:extendedInformation', requestSyscollector.loggerMessage, 'debug');
            const responseSyscollector = await context.wazuh.api.client.asCurrentUser.request('GET', requestSyscollector.endpoint, {}, {
              apiHostID: apiId
            });
            const [data] = responseSyscollector && responseSyscollector.data && responseSyscollector.data.data && responseSyscollector.data.data.affected_items || [];

            if (data) {
              return { ...requestSyscollector.list,
                list: requestSyscollector.mapResponse(data)
              };
            }
          } catch (error) {
            (0, _logger.log)('reporting:extendedInformation', error.message || error);
          }
        }));

        if (syscollectorLists) {
          syscollectorLists.filter(syscollectorList => syscollectorList).forEach(syscollectorList => printer.addList(syscollectorList));
        }

        const vulnerabilitiesRequests = ['Critical', 'High'];
        const vulnerabilitiesResponsesItems = (await Promise.all(vulnerabilitiesRequests.map(async vulnerabilitiesLevel => {
          try {
            (0, _logger.log)('reporting:extendedInformation', `Fetching top ${vulnerabilitiesLevel} packages`, 'debug');
            return await VulnerabilityRequest.topPackages(context, from, to, vulnerabilitiesLevel, filters, pattern);
          } catch (error) {
            (0, _logger.log)('reporting:extendedInformation', error.message || error);
          }
        }))).filter(vulnerabilitiesResponse => vulnerabilitiesResponse).flat();

        if (vulnerabilitiesResponsesItems && vulnerabilitiesResponsesItems.length) {
          printer.addSimpleTable({
            title: {
              text: 'Vulnerable packages found (last 24 hours)',
              style: 'h2'
            },
            columns: [{
              id: 'package',
              label: 'Package'
            }, {
              id: 'severity',
              label: 'Severity'
            }],
            items: vulnerabilitiesResponsesItems
          });
        }
      }

      if (section === 'agents' && tab === 'vuls') {
        const topCriticalPackages = await VulnerabilityRequest.topPackagesWithCVE(context, from, to, 'Critical', filters, pattern);

        if (topCriticalPackages && topCriticalPackages.length) {
          printer.addContentWithNewLine({
            text: 'Critical severity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'These vulnerabilties are critical, please review your agent. Click on each link to read more about each found vulnerability.',
            style: 'standard'
          });
          const customul = [];

          for (const critical of topCriticalPackages) {
            customul.push({
              text: critical.package,
              style: 'standard'
            });
            customul.push({
              ul: critical.references.map(item => ({
                text: item.substring(0, 80) + '...',
                link: item,
                color: '#1EA5C8'
              }))
            });
          }

          printer.addContentWithNewLine({
            ul: customul
          });
        }

        const topHighPackages = await VulnerabilityRequest.topPackagesWithCVE(context, from, to, 'High', filters, pattern);

        if (topHighPackages && topHighPackages.length) {
          printer.addContentWithNewLine({
            text: 'High severity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'Click on each link to read more about each found vulnerability.',
            style: 'standard'
          });
          const customul = [];

          for (const critical of topHighPackages) {
            customul.push({
              text: critical.package,
              style: 'standard'
            });
            customul.push({
              ul: critical.references.map(item => ({
                text: item,
                color: '#1EA5C8'
              }))
            });
          }

          customul && customul.length && printer.addContent({
            ul: customul
          });
          printer.addNewLine();
        }
      }

      return false;
    } catch (error) {
      (0, _logger.log)('reporting:extendedInformation', error.message || error);
      return Promise.reject(error);
    }
  }

  getConfigRows(data, labels) {
    (0, _logger.log)('reporting:getConfigRows', `Building configuration rows`, 'info');
    const result = [];

    for (let prop in data || []) {
      if (Array.isArray(data[prop])) {
        data[prop].forEach((x, idx) => {
          if (typeof x === 'object') data[prop][idx] = JSON.stringify(x);
        });
      }

      result.push([(labels || {})[prop] || _csvKeyEquivalence.KeyEquivalence[prop] || prop, data[prop] || '-']);
    }

    return result;
  }

  getConfigTables(data, section, tab, array = []) {
    (0, _logger.log)('reporting:getConfigTables', `Building configuration tables`, 'info');
    let plainData = {};
    const nestedData = [];
    const tableData = [];

    if (data.length === 1 && Array.isArray(data)) {
      tableData[section.config[tab].configuration] = data;
    } else {
      for (let key in data) {
        if (typeof data[key] !== 'object' && !Array.isArray(data[key]) || Array.isArray(data[key]) && typeof data[key][0] !== 'object') {
          plainData[key] = Array.isArray(data[key]) && typeof data[key][0] !== 'object' ? data[key].map(x => {
            return typeof x === 'object' ? JSON.stringify(x) : x + '\n';
          }) : data[key];
        } else if (Array.isArray(data[key]) && typeof data[key][0] === 'object') {
          tableData[key] = data[key];
        } else {
          if (section.isGroupConfig && ['pack', 'content'].includes(key)) {
            tableData[key] = [data[key]];
          } else {
            nestedData.push(data[key]);
          }
        }
      }
    }

    array.push({
      title: (section.options || {}).hideHeader ? '' : (section.tabs || [])[tab] || (section.isGroupConfig ? ((section.labels || [])[0] || [])[tab] : ''),
      columns: ['', ''],
      type: 'config',
      rows: this.getConfigRows(plainData, (section.labels || [])[0])
    });

    for (let key in tableData) {
      const columns = Object.keys(tableData[key][0]);
      columns.forEach((col, i) => {
        columns[i] = col[0].toUpperCase() + col.slice(1);
      });
      const rows = tableData[key].map(x => {
        let row = [];

        for (let key in x) {
          row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
            return x + '\n';
          }) : JSON.stringify(x[key]));
        }

        while (row.length < columns.length) {
          row.push('-');
        }

        return row;
      });
      array.push({
        title: ((section.labels || [])[0] || [])[key] || '',
        type: 'table',
        columns,
        rows
      });
    }

    nestedData.forEach(nest => {
      this.getConfigTables(nest, section, tab + 1, array);
    });
    return array;
  }
  /**
   * Create a report for the modules
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {*} reports list or ErrorResponse
   */


  /**
   * Fetch the reports list
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Array<Object>} reports list or ErrorResponse
   */
  async getReports(context, request, response) {
    try {
      (0, _logger.log)('reporting:getReports', `Fetching created reports`, 'info');
      const {
        hashUsername
      } = await context.wazuh.security.getCurrentUser(request, context);
      (0, _filesystem.createDataDirectoryIfNotExists)();
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);

      const userReportsDirectoryPath = _path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername);

      (0, _filesystem.createDirectoryIfNotExists)(userReportsDirectoryPath);
      (0, _logger.log)('reporting:getReports', `Directory: ${userReportsDirectoryPath}`, 'debug');

      const sortReportsByDate = (a, b) => a.date < b.date ? 1 : a.date > b.date ? -1 : 0;

      const reports = _fs.default.readdirSync(userReportsDirectoryPath).map(file => {
        const stats = _fs.default.statSync(userReportsDirectoryPath + '/' + file); // Get the file creation time (bithtime). It returns the first value that is a truthy value of next file stats: birthtime, mtime, ctime and atime.
        // This solves some OSs can have the bithtimeMs equal to 0 and returns the date like 1970-01-01


        const birthTimeField = ['birthtime', 'mtime', 'ctime', 'atime'].find(time => stats[`${time}Ms`]);
        return {
          name: file,
          size: stats.size,
          date: stats[birthTimeField]
        };
      });

      (0, _logger.log)('reporting:getReports', `Using TimSort for sorting ${reports.length} items`, 'debug');
      TimSort.sort(reports, sortReportsByDate);
      (0, _logger.log)('reporting:getReports', `Total reports: ${reports.length}`, 'debug');
      return response.ok({
        body: {
          reports
        }
      });
    } catch (error) {
      (0, _logger.log)('reporting:getReports', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5031, 500, response);
    }
  }
  /**
   * Fetch specific report
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} report or ErrorResponse
   */


  checkReportsUserDirectoryIsValidRouteDecorator(routeHandler, reportFileNameAccessor) {
    return async (context, request, response) => {
      try {
        const {
          username,
          hashUsername
        } = await context.wazuh.security.getCurrentUser(request, context);

        const userReportsDirectoryPath = _path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername);

        const filename = reportFileNameAccessor(request);

        const pathFilename = _path.default.join(userReportsDirectoryPath, filename);

        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', `Checking the user ${username}(${hashUsername}) can do actions in the reports file: ${pathFilename}`, 'debug');

        if (!pathFilename.startsWith(userReportsDirectoryPath) || pathFilename.includes('../')) {
          (0, _logger.log)('security:reporting:checkReportsUserDirectoryIsValidRouteDecorator', `User ${username}(${hashUsername}) tried to access to a non user report file: ${pathFilename}`, 'warn');
          return response.badRequest({
            body: {
              message: '5040 - You shall not pass!'
            }
          });
        }

        ;
        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', 'Checking the user can do actions in the reports file', 'debug');
        return await routeHandler.bind(this)({ ...context,
          wazuhEndpointParams: {
            hashUsername,
            filename,
            pathFilename
          }
        }, request, response);
      } catch (error) {
        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5040, 500, response);
      }
    };
  }

  generateReportTimestamp() {
    return `${Date.now() / 1000 | 0}`;
  }

}

exports.WazuhReportingCtrl = WazuhReportingCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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