"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobSchedulerRun = jobSchedulerRun;

var _index = require("./index");

var _configuredJobs = require("./configured-jobs");

var _logger = require("../../lib/logger");

var _getConfiguration = require("../../lib/get-configuration");

var _nodeCron = _interopRequireDefault(require("node-cron"));

var _constants = require("../../../common/constants");

var _statisticsTemplate = require("../../integration-files/statistics-template");

var _utils = require("../../../common/utils");

const blueWazuh = '\u001b[34mwazuh\u001b[39m';
const schedulerErrorLogColors = [blueWazuh, 'scheduler', 'error'];
const schedulerJobs = [];
/**
* Wait until Kibana server is ready
*/

const checkPluginPlatformStatus = async function (context) {
  try {
    (0, _logger.log)('scheduler-handler:checkPluginPlatformStatus', 'Waiting for Kibana and Elasticsearch servers to be ready...', 'debug');
    await checkElasticsearchServer(context);
    await checkTemplate(context);
    return;
  } catch (error) {
    (0, _logger.log)('scheduler-handler:checkPluginPlatformStatus', error.mesage || error);

    try {
      await (0, _utils.delayAsPromise)(3000);
      await checkPluginPlatformStatus(context);
    } catch (error) {}

    ;
  }
};
/**
 * Check Elasticsearch Server status and Kibana index presence
 */


const checkElasticsearchServer = async function (context) {
  try {
    const data = await context.core.elasticsearch.client.asInternalUser.indices.exists({
      index: context.server.config.kibana.index
    });
    return data.body;
  } catch (error) {
    (0, _logger.log)('scheduler-handler:checkElasticsearchServer', error.message || error);
    return Promise.reject(error);
  }
};
/**
* Verify wazuh-statistics template
*/


const checkTemplate = async function (context) {
  try {
    (0, _logger.log)('scheduler-handler:checkTemplate', 'Updating the statistics template', 'debug');
    const appConfig = await (0, _getConfiguration.getConfiguration)();
    const prefixTemplateName = appConfig['cron.prefix'] || _constants.WAZUH_STATISTICS_DEFAULT_PREFIX;
    const statisticsIndicesTemplateName = appConfig['cron.statistics.index.name'] || _constants.WAZUH_STATISTICS_DEFAULT_NAME;
    const pattern = `${prefixTemplateName}-${statisticsIndicesTemplateName}-*`;

    try {
      // Check if the template already exists
      const currentTemplate = await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
        name: _constants.WAZUH_STATISTICS_TEMPLATE_NAME
      }); // Copy already created index patterns

      _statisticsTemplate.statisticsTemplate.index_patterns = currentTemplate.body[_constants.WAZUH_STATISTICS_TEMPLATE_NAME].index_patterns;
    } catch (error) {
      // Init with the default index pattern
      _statisticsTemplate.statisticsTemplate.index_patterns = [pattern];
    } // Check if the user is using a custom pattern and add it to the template if it does


    if (!_statisticsTemplate.statisticsTemplate.index_patterns.includes(pattern)) {
      _statisticsTemplate.statisticsTemplate.index_patterns.push(pattern);
    }

    ; // Update the statistics template

    await context.core.elasticsearch.client.asInternalUser.indices.putTemplate({
      name: _constants.WAZUH_STATISTICS_TEMPLATE_NAME,
      body: _statisticsTemplate.statisticsTemplate
    });
    (0, _logger.log)('scheduler-handler:checkTemplate', 'Updated the statistics template', 'debug');
  } catch (error) {
    const errorMessage = `Something went wrong updating the statistics template ${error.message || error}`;
    (0, _logger.log)('scheduler-handler:checkTemplate', errorMessage);
    context.wazuh.logger.error(schedulerErrorLogColors, errorMessage);
    throw error;
  }
};

async function jobSchedulerRun(context) {
  // Check Kibana index and if it is prepared, start the initialization of Wazuh App.
  await checkPluginPlatformStatus(context);

  for (const job in (0, _configuredJobs.configuredJobs)({})) {
    const schedulerJob = new _index.SchedulerJob(job, context);
    schedulerJobs.push(schedulerJob);

    const task = _nodeCron.default.schedule(_index.jobs[job].interval, () => schedulerJob.run());
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInNjaGVkdWxlci1oYW5kbGVyLnRzIl0sIm5hbWVzIjpbImJsdWVXYXp1aCIsInNjaGVkdWxlckVycm9yTG9nQ29sb3JzIiwic2NoZWR1bGVySm9icyIsImNoZWNrUGx1Z2luUGxhdGZvcm1TdGF0dXMiLCJjb250ZXh0IiwiY2hlY2tFbGFzdGljc2VhcmNoU2VydmVyIiwiY2hlY2tUZW1wbGF0ZSIsImVycm9yIiwibWVzYWdlIiwiZGF0YSIsImNvcmUiLCJlbGFzdGljc2VhcmNoIiwiY2xpZW50IiwiYXNJbnRlcm5hbFVzZXIiLCJpbmRpY2VzIiwiZXhpc3RzIiwiaW5kZXgiLCJzZXJ2ZXIiLCJjb25maWciLCJraWJhbmEiLCJib2R5IiwibWVzc2FnZSIsIlByb21pc2UiLCJyZWplY3QiLCJhcHBDb25maWciLCJwcmVmaXhUZW1wbGF0ZU5hbWUiLCJXQVpVSF9TVEFUSVNUSUNTX0RFRkFVTFRfUFJFRklYIiwic3RhdGlzdGljc0luZGljZXNUZW1wbGF0ZU5hbWUiLCJXQVpVSF9TVEFUSVNUSUNTX0RFRkFVTFRfTkFNRSIsInBhdHRlcm4iLCJjdXJyZW50VGVtcGxhdGUiLCJnZXRUZW1wbGF0ZSIsIm5hbWUiLCJXQVpVSF9TVEFUSVNUSUNTX1RFTVBMQVRFX05BTUUiLCJzdGF0aXN0aWNzVGVtcGxhdGUiLCJpbmRleF9wYXR0ZXJucyIsImluY2x1ZGVzIiwicHVzaCIsInB1dFRlbXBsYXRlIiwiZXJyb3JNZXNzYWdlIiwid2F6dWgiLCJsb2dnZXIiLCJqb2JTY2hlZHVsZXJSdW4iLCJqb2IiLCJzY2hlZHVsZXJKb2IiLCJTY2hlZHVsZXJKb2IiLCJ0YXNrIiwiY3JvbiIsInNjaGVkdWxlIiwiam9icyIsImludGVydmFsIiwicnVuIl0sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7QUFBQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFDQTs7QUFFQSxNQUFNQSxTQUFTLEdBQUcsMkJBQWxCO0FBQ0EsTUFBTUMsdUJBQXVCLEdBQUcsQ0FBQ0QsU0FBRCxFQUFZLFdBQVosRUFBeUIsT0FBekIsQ0FBaEM7QUFDQSxNQUFNRSxhQUFhLEdBQUcsRUFBdEI7QUFFQTtBQUNBO0FBQ0E7O0FBQ0EsTUFBTUMseUJBQXlCLEdBQUcsZ0JBQWdCQyxPQUFoQixFQUF5QjtBQUN6RCxNQUFJO0FBQ0QscUJBQ0UsNkNBREYsRUFFRSw2REFGRixFQUdFLE9BSEY7QUFNRCxVQUFNQyx3QkFBd0IsQ0FBQ0QsT0FBRCxDQUE5QjtBQUNBLFVBQU1FLGFBQWEsQ0FBQ0YsT0FBRCxDQUFuQjtBQUNBO0FBQ0QsR0FWRCxDQVVFLE9BQU9HLEtBQVAsRUFBYztBQUNiLHFCQUNFLDZDQURGLEVBRUVBLEtBQUssQ0FBQ0MsTUFBTixJQUFlRCxLQUZqQjs7QUFJQSxRQUFHO0FBQ0QsWUFBTSwyQkFBZSxJQUFmLENBQU47QUFDQSxZQUFNSix5QkFBeUIsQ0FBQ0MsT0FBRCxDQUEvQjtBQUNELEtBSEQsQ0FHQyxPQUFNRyxLQUFOLEVBQVksQ0FBRTs7QUFBQTtBQUNqQjtBQUNELENBckJGO0FBd0JDO0FBQ0Q7QUFDQTs7O0FBQ0MsTUFBTUYsd0JBQXdCLEdBQUcsZ0JBQWdCRCxPQUFoQixFQUF5QjtBQUN4RCxNQUFJO0FBQ0YsVUFBTUssSUFBSSxHQUFHLE1BQU1MLE9BQU8sQ0FBQ00sSUFBUixDQUFhQyxhQUFiLENBQTJCQyxNQUEzQixDQUFrQ0MsY0FBbEMsQ0FBaURDLE9BQWpELENBQXlEQyxNQUF6RCxDQUFnRTtBQUNqRkMsTUFBQUEsS0FBSyxFQUFFWixPQUFPLENBQUNhLE1BQVIsQ0FBZUMsTUFBZixDQUFzQkMsTUFBdEIsQ0FBNkJIO0FBRDZDLEtBQWhFLENBQW5CO0FBSUEsV0FBT1AsSUFBSSxDQUFDVyxJQUFaO0FBQ0QsR0FORCxDQU1FLE9BQU9iLEtBQVAsRUFBYztBQUNkLHFCQUFJLDRDQUFKLEVBQWtEQSxLQUFLLENBQUNjLE9BQU4sSUFBaUJkLEtBQW5FO0FBQ0EsV0FBT2UsT0FBTyxDQUFDQyxNQUFSLENBQWVoQixLQUFmLENBQVA7QUFDRDtBQUNGLENBWEQ7QUFjQTtBQUNEO0FBQ0E7OztBQUNBLE1BQU1ELGFBQWEsR0FBRyxnQkFBZ0JGLE9BQWhCLEVBQXlCO0FBQzdDLE1BQUk7QUFDRixxQkFDRSxpQ0FERixFQUVFLGtDQUZGLEVBR0UsT0FIRjtBQU1BLFVBQU1vQixTQUFTLEdBQUcsTUFBTSx5Q0FBeEI7QUFDQSxVQUFNQyxrQkFBa0IsR0FBR0QsU0FBUyxDQUFDLGFBQUQsQ0FBVCxJQUE0QkUsMENBQXZEO0FBQ0EsVUFBTUMsNkJBQTZCLEdBQUdILFNBQVMsQ0FBQyw0QkFBRCxDQUFULElBQTJDSSx3Q0FBakY7QUFDQSxVQUFNQyxPQUFPLEdBQUksR0FBRUosa0JBQW1CLElBQUdFLDZCQUE4QixJQUF2RTs7QUFFQSxRQUFJO0FBQ0Y7QUFDQSxZQUFNRyxlQUFlLEdBQUcsTUFBTTFCLE9BQU8sQ0FBQ00sSUFBUixDQUFhQyxhQUFiLENBQTJCQyxNQUEzQixDQUFrQ0MsY0FBbEMsQ0FBaURDLE9BQWpELENBQXlEaUIsV0FBekQsQ0FBcUU7QUFDakdDLFFBQUFBLElBQUksRUFBRUM7QUFEMkYsT0FBckUsQ0FBOUIsQ0FGRSxDQUtGOztBQUNBQyw2Q0FBbUJDLGNBQW5CLEdBQW9DTCxlQUFlLENBQUNWLElBQWhCLENBQXFCYSx5Q0FBckIsRUFBcURFLGNBQXpGO0FBQ0QsS0FQRCxDQU9DLE9BQU81QixLQUFQLEVBQWM7QUFDYjtBQUNBMkIsNkNBQW1CQyxjQUFuQixHQUFvQyxDQUFDTixPQUFELENBQXBDO0FBQ0QsS0F0QkMsQ0F3QkY7OztBQUNBLFFBQUksQ0FBQ0ssdUNBQW1CQyxjQUFuQixDQUFrQ0MsUUFBbEMsQ0FBMkNQLE9BQTNDLENBQUwsRUFBMEQ7QUFDeERLLDZDQUFtQkMsY0FBbkIsQ0FBa0NFLElBQWxDLENBQXVDUixPQUF2QztBQUNEOztBQUFBLEtBM0JDLENBNkJGOztBQUNBLFVBQU16QixPQUFPLENBQUNNLElBQVIsQ0FBYUMsYUFBYixDQUEyQkMsTUFBM0IsQ0FBa0NDLGNBQWxDLENBQWlEQyxPQUFqRCxDQUF5RHdCLFdBQXpELENBQXFFO0FBQ3pFTixNQUFBQSxJQUFJLEVBQUVDLHlDQURtRTtBQUV6RWIsTUFBQUEsSUFBSSxFQUFFYztBQUZtRSxLQUFyRSxDQUFOO0FBSUEscUJBQ0UsaUNBREYsRUFFRSxpQ0FGRixFQUdFLE9BSEY7QUFLRCxHQXZDRCxDQXVDRSxPQUFPM0IsS0FBUCxFQUFjO0FBQ2QsVUFBTWdDLFlBQVksR0FBSSx5REFBd0RoQyxLQUFLLENBQUNjLE9BQU4sSUFBaUJkLEtBQU0sRUFBckc7QUFDQSxxQkFDRSxpQ0FERixFQUVFZ0MsWUFGRjtBQUlBbkMsSUFBQUEsT0FBTyxDQUFDb0MsS0FBUixDQUFjQyxNQUFkLENBQXFCbEMsS0FBckIsQ0FBMkJOLHVCQUEzQixFQUFvRHNDLFlBQXBEO0FBQ0EsVUFBTWhDLEtBQU47QUFDRDtBQUNGLENBakREOztBQW1ETyxlQUFlbUMsZUFBZixDQUErQnRDLE9BQS9CLEVBQXVDO0FBQzVDO0FBQ0EsUUFBTUQseUJBQXlCLENBQUNDLE9BQUQsQ0FBL0I7O0FBQ0EsT0FBSyxNQUFNdUMsR0FBWCxJQUFrQixvQ0FBZSxFQUFmLENBQWxCLEVBQXNDO0FBQ3BDLFVBQU1DLFlBQTBCLEdBQUcsSUFBSUMsbUJBQUosQ0FBaUJGLEdBQWpCLEVBQXNCdkMsT0FBdEIsQ0FBbkM7QUFDQUYsSUFBQUEsYUFBYSxDQUFDbUMsSUFBZCxDQUFtQk8sWUFBbkI7O0FBQ0EsVUFBTUUsSUFBSSxHQUFHQyxrQkFBS0MsUUFBTCxDQUNYQyxZQUFLTixHQUFMLEVBQVVPLFFBREMsRUFFWCxNQUFNTixZQUFZLENBQUNPLEdBQWIsRUFGSyxDQUFiO0FBSUQ7QUFDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IGpvYnMsIFNjaGVkdWxlckpvYiB9IGZyb20gJy4vaW5kZXgnO1xuaW1wb3J0IHsgY29uZmlndXJlZEpvYnMgfSBmcm9tICcuL2NvbmZpZ3VyZWQtam9icyc7XG5pbXBvcnQgeyBsb2cgfSBmcm9tICcuLi8uLi9saWIvbG9nZ2VyJztcbmltcG9ydCB7IGdldENvbmZpZ3VyYXRpb24gfSBmcm9tICcuLi8uLi9saWIvZ2V0LWNvbmZpZ3VyYXRpb24nO1xuaW1wb3J0IGNyb24gZnJvbSAnbm9kZS1jcm9uJztcbmltcG9ydCB7IFdBWlVIX1NUQVRJU1RJQ1NfREVGQVVMVF9QUkVGSVgsIFdBWlVIX1NUQVRJU1RJQ1NfREVGQVVMVF9OQU1FLCBXQVpVSF9TVEFUSVNUSUNTX1RFTVBMQVRFX05BTUUgfSBmcm9tICcuLi8uLi8uLi9jb21tb24vY29uc3RhbnRzJztcbmltcG9ydCB7IHN0YXRpc3RpY3NUZW1wbGF0ZSB9IGZyb20gJy4uLy4uL2ludGVncmF0aW9uLWZpbGVzL3N0YXRpc3RpY3MtdGVtcGxhdGUnO1xuaW1wb3J0IHsgZGVsYXlBc1Byb21pc2UgfSBmcm9tICcuLi8uLi8uLi9jb21tb24vdXRpbHMnO1xuXG5jb25zdCBibHVlV2F6dWggPSAnXFx1MDAxYlszNG13YXp1aFxcdTAwMWJbMzltJztcbmNvbnN0IHNjaGVkdWxlckVycm9yTG9nQ29sb3JzID0gW2JsdWVXYXp1aCwgJ3NjaGVkdWxlcicsICdlcnJvciddO1xuY29uc3Qgc2NoZWR1bGVySm9icyA9IFtdO1xuXG4vKipcbiogV2FpdCB1bnRpbCBLaWJhbmEgc2VydmVyIGlzIHJlYWR5XG4qL1xuY29uc3QgY2hlY2tQbHVnaW5QbGF0Zm9ybVN0YXR1cyA9IGFzeW5jIGZ1bmN0aW9uIChjb250ZXh0KSB7XG4gIHRyeSB7XG4gICAgIGxvZyhcbiAgICAgICAnc2NoZWR1bGVyLWhhbmRsZXI6Y2hlY2tQbHVnaW5QbGF0Zm9ybVN0YXR1cycsXG4gICAgICAgJ1dhaXRpbmcgZm9yIEtpYmFuYSBhbmQgRWxhc3RpY3NlYXJjaCBzZXJ2ZXJzIHRvIGJlIHJlYWR5Li4uJyxcbiAgICAgICAnZGVidWcnXG4gICAgICk7XG4gXG4gICAgYXdhaXQgY2hlY2tFbGFzdGljc2VhcmNoU2VydmVyKGNvbnRleHQpO1xuICAgIGF3YWl0IGNoZWNrVGVtcGxhdGUoY29udGV4dCk7XG4gICAgcmV0dXJuO1xuICB9IGNhdGNoIChlcnJvcikge1xuICAgICBsb2coXG4gICAgICAgJ3NjaGVkdWxlci1oYW5kbGVyOmNoZWNrUGx1Z2luUGxhdGZvcm1TdGF0dXMnLFxuICAgICAgIGVycm9yLm1lc2FnZSB8fGVycm9yXG4gICAgICk7XG4gICAgIHRyeXtcbiAgICAgICBhd2FpdCBkZWxheUFzUHJvbWlzZSgzMDAwKTtcbiAgICAgICBhd2FpdCBjaGVja1BsdWdpblBsYXRmb3JtU3RhdHVzKGNvbnRleHQpO1xuICAgICB9Y2F0Y2goZXJyb3Ipe307XG4gIH1cbiB9XG4gXG4gXG4gLyoqXG4gICogQ2hlY2sgRWxhc3RpY3NlYXJjaCBTZXJ2ZXIgc3RhdHVzIGFuZCBLaWJhbmEgaW5kZXggcHJlc2VuY2VcbiAgKi9cbiBjb25zdCBjaGVja0VsYXN0aWNzZWFyY2hTZXJ2ZXIgPSBhc3luYyBmdW5jdGlvbiAoY29udGV4dCkge1xuICAgdHJ5IHtcbiAgICAgY29uc3QgZGF0YSA9IGF3YWl0IGNvbnRleHQuY29yZS5lbGFzdGljc2VhcmNoLmNsaWVudC5hc0ludGVybmFsVXNlci5pbmRpY2VzLmV4aXN0cyh7XG4gICAgICAgaW5kZXg6IGNvbnRleHQuc2VydmVyLmNvbmZpZy5raWJhbmEuaW5kZXhcbiAgICAgfSk7XG4gXG4gICAgIHJldHVybiBkYXRhLmJvZHk7XG4gICB9IGNhdGNoIChlcnJvcikge1xuICAgICBsb2coJ3NjaGVkdWxlci1oYW5kbGVyOmNoZWNrRWxhc3RpY3NlYXJjaFNlcnZlcicsIGVycm9yLm1lc3NhZ2UgfHwgZXJyb3IpO1xuICAgICByZXR1cm4gUHJvbWlzZS5yZWplY3QoZXJyb3IpO1xuICAgfVxuIH1cblxuXG4gLyoqXG4gKiBWZXJpZnkgd2F6dWgtc3RhdGlzdGljcyB0ZW1wbGF0ZVxuICovXG5jb25zdCBjaGVja1RlbXBsYXRlID0gYXN5bmMgZnVuY3Rpb24gKGNvbnRleHQpIHtcbiAgdHJ5IHtcbiAgICBsb2coXG4gICAgICAnc2NoZWR1bGVyLWhhbmRsZXI6Y2hlY2tUZW1wbGF0ZScsXG4gICAgICAnVXBkYXRpbmcgdGhlIHN0YXRpc3RpY3MgdGVtcGxhdGUnLFxuICAgICAgJ2RlYnVnJ1xuICAgICk7XG5cbiAgICBjb25zdCBhcHBDb25maWcgPSBhd2FpdCBnZXRDb25maWd1cmF0aW9uKCk7XG4gICAgY29uc3QgcHJlZml4VGVtcGxhdGVOYW1lID0gYXBwQ29uZmlnWydjcm9uLnByZWZpeCddIHx8IFdBWlVIX1NUQVRJU1RJQ1NfREVGQVVMVF9QUkVGSVg7XG4gICAgY29uc3Qgc3RhdGlzdGljc0luZGljZXNUZW1wbGF0ZU5hbWUgPSBhcHBDb25maWdbJ2Nyb24uc3RhdGlzdGljcy5pbmRleC5uYW1lJ10gfHwgV0FaVUhfU1RBVElTVElDU19ERUZBVUxUX05BTUU7XG4gICAgY29uc3QgcGF0dGVybiA9IGAke3ByZWZpeFRlbXBsYXRlTmFtZX0tJHtzdGF0aXN0aWNzSW5kaWNlc1RlbXBsYXRlTmFtZX0tKmA7XG5cbiAgICB0cnkge1xuICAgICAgLy8gQ2hlY2sgaWYgdGhlIHRlbXBsYXRlIGFscmVhZHkgZXhpc3RzXG4gICAgICBjb25zdCBjdXJyZW50VGVtcGxhdGUgPSBhd2FpdCBjb250ZXh0LmNvcmUuZWxhc3RpY3NlYXJjaC5jbGllbnQuYXNJbnRlcm5hbFVzZXIuaW5kaWNlcy5nZXRUZW1wbGF0ZSh7XG4gICAgICAgIG5hbWU6IFdBWlVIX1NUQVRJU1RJQ1NfVEVNUExBVEVfTkFNRVxuICAgICAgfSk7XG4gICAgICAvLyBDb3B5IGFscmVhZHkgY3JlYXRlZCBpbmRleCBwYXR0ZXJuc1xuICAgICAgc3RhdGlzdGljc1RlbXBsYXRlLmluZGV4X3BhdHRlcm5zID0gY3VycmVudFRlbXBsYXRlLmJvZHlbV0FaVUhfU1RBVElTVElDU19URU1QTEFURV9OQU1FXS5pbmRleF9wYXR0ZXJucztcbiAgICB9Y2F0Y2ggKGVycm9yKSB7XG4gICAgICAvLyBJbml0IHdpdGggdGhlIGRlZmF1bHQgaW5kZXggcGF0dGVyblxuICAgICAgc3RhdGlzdGljc1RlbXBsYXRlLmluZGV4X3BhdHRlcm5zID0gW3BhdHRlcm5dO1xuICAgIH1cblxuICAgIC8vIENoZWNrIGlmIHRoZSB1c2VyIGlzIHVzaW5nIGEgY3VzdG9tIHBhdHRlcm4gYW5kIGFkZCBpdCB0byB0aGUgdGVtcGxhdGUgaWYgaXQgZG9lc1xuICAgIGlmICghc3RhdGlzdGljc1RlbXBsYXRlLmluZGV4X3BhdHRlcm5zLmluY2x1ZGVzKHBhdHRlcm4pKSB7XG4gICAgICBzdGF0aXN0aWNzVGVtcGxhdGUuaW5kZXhfcGF0dGVybnMucHVzaChwYXR0ZXJuKTtcbiAgICB9O1xuXG4gICAgLy8gVXBkYXRlIHRoZSBzdGF0aXN0aWNzIHRlbXBsYXRlXG4gICAgYXdhaXQgY29udGV4dC5jb3JlLmVsYXN0aWNzZWFyY2guY2xpZW50LmFzSW50ZXJuYWxVc2VyLmluZGljZXMucHV0VGVtcGxhdGUoe1xuICAgICAgbmFtZTogV0FaVUhfU1RBVElTVElDU19URU1QTEFURV9OQU1FLFxuICAgICAgYm9keTogc3RhdGlzdGljc1RlbXBsYXRlXG4gICAgfSk7XG4gICAgbG9nKFxuICAgICAgJ3NjaGVkdWxlci1oYW5kbGVyOmNoZWNrVGVtcGxhdGUnLFxuICAgICAgJ1VwZGF0ZWQgdGhlIHN0YXRpc3RpY3MgdGVtcGxhdGUnLFxuICAgICAgJ2RlYnVnJ1xuICAgICk7XG4gIH0gY2F0Y2ggKGVycm9yKSB7XG4gICAgY29uc3QgZXJyb3JNZXNzYWdlID0gYFNvbWV0aGluZyB3ZW50IHdyb25nIHVwZGF0aW5nIHRoZSBzdGF0aXN0aWNzIHRlbXBsYXRlICR7ZXJyb3IubWVzc2FnZSB8fCBlcnJvcn1gO1xuICAgIGxvZyhcbiAgICAgICdzY2hlZHVsZXItaGFuZGxlcjpjaGVja1RlbXBsYXRlJyxcbiAgICAgIGVycm9yTWVzc2FnZVxuICAgICk7XG4gICAgY29udGV4dC53YXp1aC5sb2dnZXIuZXJyb3Ioc2NoZWR1bGVyRXJyb3JMb2dDb2xvcnMsIGVycm9yTWVzc2FnZSk7XG4gICAgdGhyb3cgZXJyb3I7XG4gIH1cbn1cblxuZXhwb3J0IGFzeW5jIGZ1bmN0aW9uIGpvYlNjaGVkdWxlclJ1bihjb250ZXh0KXtcbiAgLy8gQ2hlY2sgS2liYW5hIGluZGV4IGFuZCBpZiBpdCBpcyBwcmVwYXJlZCwgc3RhcnQgdGhlIGluaXRpYWxpemF0aW9uIG9mIFdhenVoIEFwcC5cbiAgYXdhaXQgY2hlY2tQbHVnaW5QbGF0Zm9ybVN0YXR1cyhjb250ZXh0KTtcbiAgZm9yIChjb25zdCBqb2IgaW4gY29uZmlndXJlZEpvYnMoe30pKSB7XG4gICAgY29uc3Qgc2NoZWR1bGVySm9iOiBTY2hlZHVsZXJKb2IgPSBuZXcgU2NoZWR1bGVySm9iKGpvYiwgY29udGV4dCk7XG4gICAgc2NoZWR1bGVySm9icy5wdXNoKHNjaGVkdWxlckpvYik7XG4gICAgY29uc3QgdGFzayA9IGNyb24uc2NoZWR1bGUoXG4gICAgICBqb2JzW2pvYl0uaW50ZXJ2YWwsXG4gICAgICAoKSA9PiBzY2hlZHVsZXJKb2IucnVuKCksXG4gICAgKTtcbiAgfVxufSJdfQ==