"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhElasticCtrl = void 0;

var _errorResponse = require("../lib/error-response");

var _logger = require("../lib/logger");

var _getConfiguration = require("../lib/get-configuration");

var _visualizations = require("../integration-files/visualizations");

var _generateAlertsScript = require("../lib/generate-alerts/generate-alerts-script");

var _constants = require("../../common/constants");

var _jwtDecode = _interopRequireDefault(require("jwt-decode"));

var _manageHosts = require("../lib/manage-hosts");

var _cookie = require("../lib/cookie");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class WazuhElasticCtrl {
  constructor() {
    _defineProperty(this, "wzSampleAlertsIndexPrefix", void 0);

    _defineProperty(this, "manageHosts", void 0);

    this.wzSampleAlertsIndexPrefix = this.getSampleAlertPrefix();
    this.manageHosts = new _manageHosts.ManageHosts();
  }
  /**
   * This returns the index according the category
   * @param {string} category
   */


  buildSampleIndexByCategory(category) {
    return `${this.wzSampleAlertsIndexPrefix}sample-${category}`;
  }
  /**
   * This returns the defined config for sample alerts prefix or the default value.
   */


  getSampleAlertPrefix() {
    const config = (0, _getConfiguration.getConfiguration)();
    return config['alerts.sample.prefix'] || _constants.WAZUH_SAMPLE_ALERT_PREFIX;
  }
  /**
   * This retrieves a template from Elasticsearch
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} template or ErrorResponse
   */


  async getTemplate(context, request, response) {
    try {
      const data = await context.core.elasticsearch.client.asInternalUser.cat.templates();
      const templates = data.body;

      if (!templates || typeof templates !== 'string') {
        throw new Error('An unknown error occurred when fetching templates from Elasticseach');
      }

      const lastChar = request.params.pattern[request.params.pattern.length - 1]; // Split into separate patterns

      const tmpdata = templates.match(/\[.*\]/g);
      const tmparray = [];

      for (let item of tmpdata) {
        // A template might use more than one pattern
        if (item.includes(',')) {
          item = item.substr(1).slice(0, -1);
          const subItems = item.split(',');

          for (const subitem of subItems) {
            tmparray.push(`[${subitem.trim()}]`);
          }
        } else {
          tmparray.push(item);
        }
      } // Ensure we are handling just patterns


      const array = tmparray.filter(item => item.includes('[') && item.includes(']'));
      const pattern = lastChar === '*' ? request.params.pattern.slice(0, -1) : request.params.pattern;
      const isIncluded = array.filter(item => {
        item = item.slice(1, -1);
        const lastChar = item[item.length - 1];
        item = lastChar === '*' ? item.slice(0, -1) : item;
        return item.includes(pattern) || pattern.includes(item);
      });
      (0, _logger.log)('wazuh-elastic:getTemplate', `Template is valid: ${isIncluded && Array.isArray(isIncluded) && isIncluded.length ? 'yes' : 'no'}`, 'debug');
      return isIncluded && Array.isArray(isIncluded) && isIncluded.length ? response.ok({
        body: {
          statusCode: 200,
          status: true,
          data: `Template found for ${request.params.pattern}`
        }
      }) : response.ok({
        body: {
          statusCode: 200,
          status: false,
          data: `No template found for ${request.params.pattern}`
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:getTemplate', error.message || error);
      return (0, _errorResponse.ErrorResponse)(`Could not retrieve templates from Elasticsearch due to ${error.message || error}`, 4002, 500, response);
    }
  }
  /**
   * This check index-pattern
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} status obj or ErrorResponse
   */


  async checkPattern(context, request, response) {
    try {
      const data = await context.core.savedObjects.client.find({
        type: 'index-pattern'
      });
      const existsIndexPattern = data.saved_objects.find(item => item.attributes.title === request.params.pattern);
      (0, _logger.log)('wazuh-elastic:checkPattern', `Index pattern found: ${existsIndexPattern ? existsIndexPattern.attributes.title : 'no'}`, 'debug');
      return existsIndexPattern ? response.ok({
        body: {
          statusCode: 200,
          status: true,
          data: 'Index pattern found'
        }
      }) : response.ok({
        body: {
          statusCode: 500,
          status: false,
          error: 10020,
          message: 'Index pattern not found'
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:checkPattern', error.message || error);
      return (0, _errorResponse.ErrorResponse)(`Something went wrong retrieving index-patterns from Elasticsearch due to ${error.message || error}`, 4003, 500, response);
    }
  }
  /**
   * This get the fields keys
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Array<Object>} fields or ErrorResponse
   */


  async getFieldTop(context, request, response) {
    try {
      // Top field payload
      let payload = {
        size: 1,
        query: {
          bool: {
            must: [],
            must_not: {
              term: {
                'agent.id': '000'
              }
            },
            filter: [{
              range: {
                timestamp: {}
              }
            }]
          }
        },
        aggs: {
          '2': {
            terms: {
              field: '',
              size: 1,
              order: {
                _count: 'desc'
              }
            }
          }
        }
      }; // Set up time interval, default to Last 24h

      const timeGTE = 'now-1d';
      const timeLT = 'now';
      payload.query.bool.filter[0].range['timestamp']['gte'] = timeGTE;
      payload.query.bool.filter[0].range['timestamp']['lt'] = timeLT; // Set up match for default cluster name

      payload.query.bool.must.push(request.params.mode === 'cluster' ? {
        match: {
          'cluster.name': request.params.cluster
        }
      } : {
        match: {
          'manager.name': request.params.cluster
        }
      });
      if (request.query.agentsList) payload.query.bool.filter.push({
        terms: {
          'agent.id': request.query.agentsList.split(',')
        }
      });
      payload.aggs['2'].terms.field = request.params.field;
      const data = await context.core.elasticsearch.client.asCurrentUser.search({
        size: 1,
        index: request.params.pattern,
        body: payload
      });
      return data.body.hits.total.value === 0 || typeof data.body.aggregations['2'].buckets[0] === 'undefined' ? response.ok({
        body: {
          statusCode: 200,
          data: ''
        }
      }) : response.ok({
        body: {
          statusCode: 200,
          data: data.body.aggregations['2'].buckets[0].key
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:getFieldTop', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4004, 500, response);
    }
  }
  /**
   * Checks one by one if the requesting user has enough privileges to use
   * an index pattern from the list.
   * @param {Array<Object>} list List of index patterns
   * @param {Object} req
   * @returns {Array<Object>} List of allowed index
   */


  async filterAllowedIndexPatternList(context, list, req) {
    //TODO: review if necesary to delete
    let finalList = [];

    for (let item of list) {
      let results = false,
          forbidden = false;

      try {
        results = await context.core.elasticsearch.client.asCurrentUser.search({
          index: item.title
        });
      } catch (error) {
        forbidden = true;
      }

      if ((((results || {}).body || {}).hits || {}).total.value >= 1 || !forbidden && (((results || {}).body || {}).hits || {}).total === 0) {
        finalList.push(item);
      }
    }

    return finalList;
  }
  /**
   * Checks for minimum index pattern fields in a list of index patterns.
   * @param {Array<Object>} indexPatternList List of index patterns
   */


  validateIndexPattern(indexPatternList) {
    const minimum = ['timestamp', 'rule.groups', 'manager.name', 'agent.id'];
    let list = [];

    for (const index of indexPatternList) {
      let valid, parsed;

      try {
        parsed = JSON.parse(index.attributes.fields);
      } catch (error) {
        continue;
      }

      valid = parsed.filter(item => minimum.includes(item.name));

      if (valid.length === 4) {
        list.push({
          id: index.id,
          title: index.attributes.title
        });
      }
    }

    return list;
  }
  /**
   * Returns current security platform
   * @param {Object} req
   * @param {Object} reply
   * @returns {String}
   */


  async getCurrentPlatform(context, request, response) {
    try {
      return response.ok({
        body: {
          platform: context.wazuh.security.platform
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:getCurrentPlatform', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4011, 500, response);
    }
  }
  /**
   * Replaces visualizations main fields to fit a certain pattern.
   * @param {Array<Object>} app_objects Object containing raw visualizations.
   * @param {String} id Index-pattern id to use in the visualizations. Eg: 'wazuh-alerts'
   */


  async buildVisualizationsRaw(app_objects, id, namespace = false) {
    try {
      const config = (0, _getConfiguration.getConfiguration)();
      let monitoringPattern = (config || {})['wazuh.monitoring.pattern'] || _constants.WAZUH_MONITORING_PATTERN;
      (0, _logger.log)('wazuh-elastic:buildVisualizationsRaw', `Building ${app_objects.length} visualizations`, 'debug');
      (0, _logger.log)('wazuh-elastic:buildVisualizationsRaw', `Index pattern ID: ${id}`, 'debug');
      const visArray = [];
      let aux_source, bulk_content;

      for (let element of app_objects) {
        aux_source = JSON.parse(JSON.stringify(element._source)); // Replace index-pattern for visualizations

        if (aux_source && aux_source.kibanaSavedObjectMeta && aux_source.kibanaSavedObjectMeta.searchSourceJSON && typeof aux_source.kibanaSavedObjectMeta.searchSourceJSON === 'string') {
          const defaultStr = aux_source.kibanaSavedObjectMeta.searchSourceJSON;
          const isMonitoring = defaultStr.includes('wazuh-monitoring');

          if (isMonitoring) {
            if (namespace && namespace !== 'default') {
              if (monitoringPattern.includes(namespace) && monitoringPattern.includes('index-pattern:')) {
                monitoringPattern = monitoringPattern.split('index-pattern:')[1];
              }
            }

            aux_source.kibanaSavedObjectMeta.searchSourceJSON = defaultStr.replace(/wazuh-monitoring/g, monitoringPattern[monitoringPattern.length - 1] === '*' || namespace && namespace !== 'default' ? monitoringPattern : monitoringPattern + '*');
          } else {
            aux_source.kibanaSavedObjectMeta.searchSourceJSON = defaultStr.replace(/wazuh-alerts/g, id);
          }
        } // Replace index-pattern for selector visualizations


        if (typeof (aux_source || {}).visState === 'string') {
          aux_source.visState = aux_source.visState.replace(/wazuh-alerts/g, id);
        } // Bulk source


        bulk_content = {};
        bulk_content[element._type] = aux_source;
        visArray.push({
          attributes: bulk_content.visualization,
          type: element._type,
          id: element._id,
          _version: bulk_content.visualization.version
        });
      }

      return visArray;
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:buildVisualizationsRaw', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * Replaces cluster visualizations main fields.
   * @param {Array<Object>} app_objects Object containing raw visualizations.
   * @param {String} id Index-pattern id to use in the visualizations. Eg: 'wazuh-alerts'
   * @param {Array<String>} nodes Array of node names. Eg: ['node01', 'node02']
   * @param {String} name Cluster name. Eg: 'wazuh'
   * @param {String} master_node Master node name. Eg: 'node01'
   */


  buildClusterVisualizationsRaw(app_objects, id, nodes = [], name, master_node, pattern_name = '*') {
    try {
      const visArray = [];
      let aux_source, bulk_content;

      for (const element of app_objects) {
        // Stringify and replace index-pattern for visualizations
        aux_source = JSON.stringify(element._source);
        aux_source = aux_source.replace(/wazuh-alerts/g, id);
        aux_source = JSON.parse(aux_source); // Bulk source

        bulk_content = {};
        bulk_content[element._type] = aux_source;
        const visState = JSON.parse(bulk_content.visualization.visState);
        const title = visState.title;

        if (visState.type && visState.type === 'timelion') {
          let query = '';

          if (title === 'Wazuh App Cluster Overview') {
            for (const node of nodes) {
              query += `.es(index=${pattern_name},q="cluster.name: ${name} AND cluster.node: ${node.name}").label("${node.name}"),`;
            }

            query = query.substring(0, query.length - 1);
          } else if (title === 'Wazuh App Cluster Overview Manager') {
            query += `.es(index=${pattern_name},q="cluster.name: ${name}").label("${name} cluster")`;
          } else {
            if (title.startsWith('Wazuh App Statistics')) {
              const {
                searchSourceJSON
              } = bulk_content.visualization.kibanaSavedObjectMeta;
              bulk_content.visualization.kibanaSavedObjectMeta.searchSourceJSON = searchSourceJSON.replace('wazuh-statistics-*', pattern_name);
            }

            if (title.startsWith('Wazuh App Statistics') && name !== '-' && name !== 'all' && visState.params.expression.includes('q=')) {
              const expressionRegex = /q='\*'/gi;

              const _visState = bulk_content.visualization.visStateByNode ? JSON.parse(bulk_content.visualization.visStateByNode) : visState;

              query += _visState.params.expression.replace(/wazuh-statistics-\*/g, pattern_name).replace(expressionRegex, `q="nodeName.keyword:${name} AND apiName.keyword:${master_node}"`).replace("NODE_NAME", name);
            } else if (title.startsWith('Wazuh App Statistics')) {
              const expressionRegex = /q='\*'/gi;
              query += visState.params.expression.replace(/wazuh-statistics-\*/g, pattern_name).replace(expressionRegex, `q="apiName.keyword:${master_node}"`);
            } else {
              query = visState.params.expression;
            }
          }

          visState.params.expression = query.replace(/'/g, "\"");
          bulk_content.visualization.visState = JSON.stringify(visState);
        }

        visArray.push({
          attributes: bulk_content.visualization,
          type: element._type,
          id: element._id,
          _version: bulk_content.visualization.version
        });
      }

      return visArray;
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:buildClusterVisualizationsRaw', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This creates a visualization of data in req
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} vis obj or ErrorResponse
   */


  async createVis(context, request, response) {
    try {
      if (!request.params.tab.includes('overview-') && !request.params.tab.includes('agents-')) {
        throw new Error('Missing parameters creating visualizations');
      }

      const tabPrefix = request.params.tab.includes('overview') ? 'overview' : 'agents';
      const tabSplit = request.params.tab.split('-');
      const tabSufix = tabSplit[1];
      const file = tabPrefix === 'overview' ? _visualizations.OverviewVisualizations[tabSufix] : _visualizations.AgentsVisualizations[tabSufix];

      if (!file) {
        return response.notFound({
          body: {
            message: `Visualizations not found for ${request.params.tab}`
          }
        });
      }

      (0, _logger.log)('wazuh-elastic:createVis', `${tabPrefix}[${tabSufix}] with index pattern ${request.params.pattern}`, 'debug');
      const namespace = context.wazuh.plugins.spaces && context.wazuh.plugins.spaces.spacesService && context.wazuh.plugins.spaces.spacesService.getSpaceId(request);
      const raw = await this.buildVisualizationsRaw(file, request.params.pattern, namespace);
      return response.ok({
        body: {
          acknowledge: true,
          raw: raw
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:createVis', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4007, 500, response);
    }
  }
  /**
   * This creates a visualization of cluster
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} vis obj or ErrorResponse
   */


  async createClusterVis(context, request, response) {
    try {
      if (!request.params.pattern || !request.params.tab || !request.body || !request.body.nodes || !request.body.nodes.affected_items || !request.body.nodes.name || request.params.tab && !request.params.tab.includes('cluster-')) {
        throw new Error('Missing parameters creating visualizations');
      }

      const type = request.params.tab.split('-')[1];
      const file = _visualizations.ClusterVisualizations[type];
      const nodes = request.body.nodes.affected_items;
      const name = request.body.nodes.name;
      const masterNode = request.body.nodes.master_node;
      const {
        id: patternID,
        title: patternName
      } = request.body.pattern;
      const raw = await this.buildClusterVisualizationsRaw(file, patternID, nodes, name, masterNode, patternName);
      return response.ok({
        body: {
          acknowledge: true,
          raw: raw
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:createClusterVis', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4009, 500, response);
    }
  }
  /**
   * This checks if there is sample alerts
   * GET /elastic/samplealerts
   * @param {*} context
   * @param {*} request
   * @param {*} response
   * {alerts: [...]} or ErrorResponse
   */


  async haveSampleAlerts(context, request, response) {
    try {
      // Check if wazuh sample alerts index exists
      const results = await Promise.all(Object.keys(_constants.WAZUH_SAMPLE_ALERTS_CATEGORIES_TYPE_ALERTS).map(category => context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: this.buildSampleIndexByCategory(category)
      })));
      return response.ok({
        body: {
          sampleAlertsInstalled: results.some(result => result.body)
        }
      });
    } catch (error) {
      return (0, _errorResponse.ErrorResponse)('Sample Alerts category not valid', 1000, 500, response);
    }
  }
  /**
   * This creates sample alerts in wazuh-sample-alerts
   * GET /elastic/samplealerts/{category}
   * @param {*} context
   * @param {*} request
   * @param {*} response
   * {alerts: [...]} or ErrorResponse
   */


  async haveSampleAlertsOfCategory(context, request, response) {
    try {
      const sampleAlertsIndex = this.buildSampleIndexByCategory(request.params.category); // Check if wazuh sample alerts index exists

      const existsSampleIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: sampleAlertsIndex
      });
      return response.ok({
        body: {
          index: sampleAlertsIndex,
          exists: existsSampleIndex.body
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:haveSampleAlertsOfCategory', `Error checking if there are sample alerts indices: ${error.message || error}`);
      const [statusCode, errorMessage] = this.getErrorDetails(error);
      return (0, _errorResponse.ErrorResponse)(`Error checking if there are sample alerts indices: ${errorMessage || error}`, 1000, statusCode, response);
    }
  }
  /**
   * This creates sample alerts in wazuh-sample-alerts
   * POST /elastic/samplealerts/{category}
   * {
   *   "manager": {
   *      "name": "manager_name"
   *    },
   *    cluster: {
   *      name: "mycluster",
   *      node: "mynode"
   *    }
   * }
   * @param {*} context
   * @param {*} request
   * @param {*} response
   * {index: string, alerts: [...], count: number} or ErrorResponse
   */


  async createSampleAlerts(context, request, response) {
    const sampleAlertsIndex = this.buildSampleIndexByCategory(request.params.category);

    try {
      // Check if user has administrator role in token
      const token = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-token');

      if (!token) {
        return (0, _errorResponse.ErrorResponse)('No token provided', 401, 401, response);
      }

      ;
      const decodedToken = (0, _jwtDecode.default)(token);

      if (!decodedToken) {
        return (0, _errorResponse.ErrorResponse)('No permissions in token', 401, 401, response);
      }

      ;

      if (!decodedToken.rbac_roles || !decodedToken.rbac_roles.includes(_constants.WAZUH_ROLE_ADMINISTRATOR_ID)) {
        return (0, _errorResponse.ErrorResponse)('No administrator role', 401, 401, response);
      }

      ; // Check the provided token is valid

      const apiHostID = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-api');

      if (!apiHostID) {
        return (0, _errorResponse.ErrorResponse)('No API id provided', 401, 401, response);
      }

      ;
      const responseTokenIsWorking = await context.wazuh.api.client.asCurrentUser.request('GET', `//`, {}, {
        apiHostID
      });

      if (responseTokenIsWorking.status !== 200) {
        return (0, _errorResponse.ErrorResponse)('Token is not valid', 500, 500, response);
      }

      ;
      const bulkPrefix = JSON.stringify({
        index: {
          _index: sampleAlertsIndex
        }
      });
      const alertGenerateParams = request.body && request.body.params || {};

      const sampleAlerts = _constants.WAZUH_SAMPLE_ALERTS_CATEGORIES_TYPE_ALERTS[request.params.category].map(typeAlert => (0, _generateAlertsScript.generateAlerts)({ ...typeAlert,
        ...alertGenerateParams
      }, request.body.alerts || typeAlert.alerts || _constants.WAZUH_SAMPLE_ALERTS_DEFAULT_NUMBER_ALERTS)).flat();

      const bulk = sampleAlerts.map(sampleAlert => `${bulkPrefix}\n${JSON.stringify(sampleAlert)}\n`).join(''); // Index alerts
      // Check if wazuh sample alerts index exists

      const existsSampleIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: sampleAlertsIndex
      });

      if (!existsSampleIndex.body) {
        // Create wazuh sample alerts index
        const configuration = {
          settings: {
            index: {
              number_of_shards: _constants.WAZUH_SAMPLE_ALERTS_INDEX_SHARDS,
              number_of_replicas: _constants.WAZUH_SAMPLE_ALERTS_INDEX_REPLICAS
            }
          }
        };
        await context.core.elasticsearch.client.asCurrentUser.indices.create({
          index: sampleAlertsIndex,
          body: configuration
        });
        (0, _logger.log)('wazuh-elastic:createSampleAlerts', `Created ${sampleAlertsIndex} index`, 'debug');
      }

      await context.core.elasticsearch.client.asCurrentUser.bulk({
        index: sampleAlertsIndex,
        body: bulk
      });
      (0, _logger.log)('wazuh-elastic:createSampleAlerts', `Added sample alerts to ${sampleAlertsIndex} index`, 'debug');
      return response.ok({
        body: {
          index: sampleAlertsIndex,
          alertCount: sampleAlerts.length
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:createSampleAlerts', `Error adding sample alerts to ${sampleAlertsIndex} index: ${error.message || error}`);
      const [statusCode, errorMessage] = this.getErrorDetails(error);
      return (0, _errorResponse.ErrorResponse)(errorMessage || error, 1000, statusCode, response);
    }
  }
  /**
   * This deletes sample alerts
   * @param {*} context
   * @param {*} request
   * @param {*} response
   * {result: "deleted", index: string} or ErrorResponse
   */


  async deleteSampleAlerts(context, request, response) {
    // Delete Wazuh sample alert index
    const sampleAlertsIndex = this.buildSampleIndexByCategory(request.params.category);

    try {
      // Check if user has administrator role in token
      const token = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-token');

      if (!token) {
        return (0, _errorResponse.ErrorResponse)('No token provided', 401, 401, response);
      }

      ;
      const decodedToken = (0, _jwtDecode.default)(token);

      if (!decodedToken) {
        return (0, _errorResponse.ErrorResponse)('No permissions in token', 401, 401, response);
      }

      ;

      if (!decodedToken.rbac_roles || !decodedToken.rbac_roles.includes(_constants.WAZUH_ROLE_ADMINISTRATOR_ID)) {
        return (0, _errorResponse.ErrorResponse)('No administrator role', 401, 401, response);
      }

      ; // Check the provided token is valid

      const apiHostID = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-api');

      if (!apiHostID) {
        return (0, _errorResponse.ErrorResponse)('No API id provided', 401, 401, response);
      }

      ;
      const responseTokenIsWorking = await context.wazuh.api.client.asCurrentUser.request('GET', `//`, {}, {
        apiHostID
      });

      if (responseTokenIsWorking.status !== 200) {
        return (0, _errorResponse.ErrorResponse)('Token is not valid', 500, 500, response);
      }

      ; // Check if Wazuh sample alerts index exists

      const existsSampleIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: sampleAlertsIndex
      });

      if (existsSampleIndex.body) {
        // Delete Wazuh sample alerts index
        await context.core.elasticsearch.client.asCurrentUser.indices.delete({
          index: sampleAlertsIndex
        });
        (0, _logger.log)('wazuh-elastic:deleteSampleAlerts', `Deleted ${sampleAlertsIndex} index`, 'debug');
        return response.ok({
          body: {
            result: 'deleted',
            index: sampleAlertsIndex
          }
        });
      } else {
        return (0, _errorResponse.ErrorResponse)(`${sampleAlertsIndex} index doesn't exist`, 1000, 500, response);
      }
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:deleteSampleAlerts', `Error deleting sample alerts of ${sampleAlertsIndex} index: ${error.message || error}`);
      const [statusCode, errorMessage] = this.getErrorDetails(error);
      return (0, _errorResponse.ErrorResponse)(errorMessage || error, 1000, statusCode, response);
    }
  }

  async alerts(context, request, response) {
    try {
      const data = await context.core.elasticsearch.client.asCurrentUser.search(request.body);
      return response.ok({
        body: data.body
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:alerts', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4010, 500, response);
    }
  } // Check if there are indices for Statistics


  async existStatisticsIndices(context, request, response) {
    try {
      const config = (0, _getConfiguration.getConfiguration)();
      const statisticsPattern = `${config['cron.prefix'] || 'wazuh'}-${config['cron.statistics.index.name'] || 'statistics'}*`; //TODO: replace by default as constants instead hardcoded ('wazuh' and 'statistics')

      const existIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: statisticsPattern,
        allow_no_indices: false
      });
      return response.ok({
        body: existIndex.body
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:existsStatisticsIndices', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 1000, 500, response);
    }
  }

  async usingCredentials(context) {
    try {
      const data = await context.core.elasticsearch.client.asInternalUser.cluster.getSettings({
        include_defaults: true
      });
      return (((((data || {}).body || {}).defaults || {}).xpack || {}).security || {}).user !== null;
    } catch (error) {
      return Promise.reject(error);
    }
  }

  getErrorDetails(error) {
    var _error$meta;

    const statusCode = (error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.statusCode) || 500;
    let errorMessage = error.message;

    if (statusCode === 403) {
      var _error$meta2, _error$meta2$body, _error$meta2$body$err;

      errorMessage = (error === null || error === void 0 ? void 0 : (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : (_error$meta2$body$err = _error$meta2$body.error) === null || _error$meta2$body$err === void 0 ? void 0 : _error$meta2$body$err.reason) || 'Permission denied';
    }

    return [statusCode, errorMessage];
  }

}

exports.WazuhElasticCtrl = WazuhElasticCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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