"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhReportingCtrl = void 0;

var _path = _interopRequireDefault(require("path"));

var _fs = _interopRequireDefault(require("fs"));

var _wazuhModules = require("../../common/wazuh-modules");

var TimSort = _interopRequireWildcard(require("timsort"));

var _errorResponse = require("../lib/error-response");

var VulnerabilityRequest = _interopRequireWildcard(require("../lib/reporting/vulnerability-request"));

var OverviewRequest = _interopRequireWildcard(require("../lib/reporting/overview-request"));

var RootcheckRequest = _interopRequireWildcard(require("../lib/reporting/rootcheck-request"));

var PCIRequest = _interopRequireWildcard(require("../lib/reporting/pci-request"));

var GDPRRequest = _interopRequireWildcard(require("../lib/reporting/gdpr-request"));

var TSCRequest = _interopRequireWildcard(require("../lib/reporting/tsc-request"));

var AuditRequest = _interopRequireWildcard(require("../lib/reporting/audit-request"));

var SyscheckRequest = _interopRequireWildcard(require("../lib/reporting/syscheck-request"));

var _pciRequirementsPdfmake = _interopRequireDefault(require("../integration-files/pci-requirements-pdfmake"));

var _gdprRequirementsPdfmake = _interopRequireDefault(require("../integration-files/gdpr-requirements-pdfmake"));

var _tscRequirementsPdfmake = _interopRequireDefault(require("../integration-files/tsc-requirements-pdfmake"));

var _processStateEquivalence = _interopRequireDefault(require("../lib/process-state-equivalence"));

var _csvKeyEquivalence = require("../../common/csv-key-equivalence");

var _agentConfiguration = require("../lib/reporting/agent-configuration");

var _printer = require("../lib/reporting/printer");

var _logger = require("../lib/logger");

var _constants = require("../../common/constants");

var _filesystem = require("../lib/filesystem");

var _moment = _interopRequireDefault(require("moment"));

var _wz_agent_status = require("../../common/services/wz_agent_status");

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class WazuhReportingCtrl {
  constructor() {
    _defineProperty(this, "createReportsModules", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsModules', `Report started`, 'info');
        const {
          array,
          agents,
          browserTimezone,
          searchBar,
          filters,
          time,
          tables,
          section,
          indexPatternTitle,
          apiId
        } = request.body;
        const {
          moduleID
        } = request.params;
        const {
          from,
          to
        } = time || {}; // Init

        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        await this.renderHeader(context, printer, section, moduleID, agents, apiId);
        const [sanitizedFilters, agentsFilter] = filters ? this.sanitizeKibanaFilters(filters, searchBar) : [false, false];

        if (time && sanitizedFilters) {
          printer.addTimeRangeAndFilters(from, to, sanitizedFilters, browserTimezone);
        }

        if (time) {
          await this.extendedInformation(context, printer, section, moduleID, apiId, new Date(from).getTime(), new Date(to).getTime(), sanitizedFilters, indexPatternTitle, agents);
        }

        printer.addVisualizations(array, agents, moduleID);

        if (tables) {
          printer.addTables(tables);
        } //add authorized agents


        if (agentsFilter) {
          printer.addAgentsFilters(agentsFilter);
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      body: {
        agents
      },
      params: {
        moduleID
      }
    }) => `wazuh-module-${agents ? `agents-${agents}` : 'overview'}-${moduleID}-${this.generateReportTimestamp()}.pdf`));

    _defineProperty(this, "createReportsGroups", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsGroups', `Report started`, 'info');
        const {
          components,
          apiId
        } = request.body;
        const {
          groupID
        } = request.params; // Init

        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        let tables = [];
        const equivalences = {
          localfile: 'Local files',
          osquery: 'Osquery',
          command: 'Command',
          syscheck: 'Syscheck',
          'open-scap': 'OpenSCAP',
          'cis-cat': 'CIS-CAT',
          syscollector: 'Syscollector',
          rootcheck: 'Rootcheck',
          labels: 'Labels',
          sca: 'Security configuration assessment'
        };
        printer.addContent({
          text: `Group ${groupID} configuration`,
          style: 'h1'
        }); // Group configuration

        if (components['0']) {
          const {
            data: {
              data: configuration
            }
          } = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${groupID}/configuration`, {}, {
            apiHostID: apiId
          });

          if (configuration.affected_items.length > 0 && Object.keys(configuration.affected_items[0].config).length) {
            printer.addContent({
              text: 'Configurations',
              style: {
                fontSize: 14,
                color: '#000'
              },
              margin: [0, 10, 0, 15]
            });
            const section = {
              labels: [],
              isGroupConfig: true
            };

            for (let config of configuration.affected_items) {
              let filterTitle = '';
              let index = 0;

              for (let filter of Object.keys(config.filters)) {
                filterTitle = filterTitle.concat(`${filter}: ${config.filters[filter]}`);

                if (index < Object.keys(config.filters).length - 1) {
                  filterTitle = filterTitle.concat(' | ');
                }

                index++;
              }

              printer.addContent({
                text: filterTitle,
                style: 'h4',
                margin: [0, 0, 0, 10]
              });
              let idx = 0;
              section.tabs = [];

              for (let _d of Object.keys(config.config)) {
                for (let c of _agentConfiguration.AgentConfiguration.configurations) {
                  for (let s of c.sections) {
                    section.opts = s.opts || {};

                    for (let cn of s.config || []) {
                      if (cn.configuration === _d) {
                        section.labels = s.labels || [[]];
                      }
                    }

                    for (let wo of s.wodle || []) {
                      if (wo.name === _d) {
                        section.labels = s.labels || [[]];
                      }
                    }
                  }
                }

                section.labels[0]['pack'] = 'Packs';
                section.labels[0]['content'] = 'Evaluations';
                section.labels[0]['7'] = 'Scan listening netwotk ports';
                section.tabs.push(equivalences[_d]);

                if (Array.isArray(config.config[_d])) {
                  /* LOG COLLECTOR */
                  if (_d === 'localfile') {
                    let groups = [];

                    config.config[_d].forEach(obj => {
                      if (!groups[obj.logformat]) {
                        groups[obj.logformat] = [];
                      }

                      groups[obj.logformat].push(obj);
                    });

                    Object.keys(groups).forEach(group => {
                      let saveidx = 0;
                      groups[group].forEach((x, i) => {
                        if (Object.keys(x).length > Object.keys(groups[group][saveidx]).length) {
                          saveidx = i;
                        }
                      });
                      const columns = Object.keys(groups[group][saveidx]);
                      const rows = groups[group].map(x => {
                        let row = [];
                        columns.forEach(key => {
                          row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
                            return x + '\n';
                          }) : JSON.stringify(x[key]));
                        });
                        return row;
                      });
                      columns.forEach((col, i) => {
                        columns[i] = col[0].toUpperCase() + col.slice(1);
                      });
                      tables.push({
                        title: 'Local files',
                        type: 'table',
                        columns,
                        rows
                      });
                    });
                  } else if (_d === 'labels') {
                    const obj = config.config[_d][0].label;
                    const columns = Object.keys(obj[0]);

                    if (!columns.includes('hidden')) {
                      columns.push('hidden');
                    }

                    const rows = obj.map(x => {
                      let row = [];
                      columns.forEach(key => {
                        row.push(x[key]);
                      });
                      return row;
                    });
                    columns.forEach((col, i) => {
                      columns[i] = col[0].toUpperCase() + col.slice(1);
                    });
                    tables.push({
                      title: 'Labels',
                      type: 'table',
                      columns,
                      rows
                    });
                  } else {
                    for (let _d2 of config.config[_d]) {
                      tables.push(...this.getConfigTables(_d2, section, idx));
                    }
                  }
                } else {
                  /*INTEGRITY MONITORING MONITORED DIRECTORIES */
                  if (config.config[_d].directories) {
                    const directories = config.config[_d].directories;
                    delete config.config[_d].directories;
                    tables.push(...this.getConfigTables(config.config[_d], section, idx));
                    let diffOpts = [];
                    Object.keys(section.opts).forEach(x => {
                      diffOpts.push(x);
                    });
                    const columns = ['', ...diffOpts.filter(x => x !== 'check_all' && x !== 'check_sum')];
                    let rows = [];
                    directories.forEach(x => {
                      let row = [];
                      row.push(x.path);
                      columns.forEach(y => {
                        if (y !== '') {
                          y = y !== 'check_whodata' ? y : 'whodata';
                          row.push(x[y] ? x[y] : 'no');
                        }
                      });
                      row.push(x.recursion_level);
                      rows.push(row);
                    });
                    columns.forEach((x, idx) => {
                      columns[idx] = section.opts[x];
                    });
                    columns.push('RL');
                    tables.push({
                      title: 'Monitored directories',
                      type: 'table',
                      columns,
                      rows
                    });
                  } else {
                    tables.push(...this.getConfigTables(config.config[_d], section, idx));
                  }
                }

                for (const table of tables) {
                  printer.addConfigTables([table]);
                }

                idx++;
                tables = [];
              }

              tables = [];
            }
          } else {
            printer.addContent({
              text: 'A configuration for this group has not yet been set up.',
              style: {
                fontSize: 12,
                color: '#000'
              },
              margin: [0, 10, 0, 15]
            });
          }
        } // Agents in group


        if (components['1']) {
          await this.renderHeader(context, printer, 'groupConfig', groupID, [], apiId);
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsGroups', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        groupID
      }
    }) => `wazuh-group-configuration-${groupID}-${this.generateReportTimestamp()}.pdf`));

    _defineProperty(this, "createReportsAgentsConfiguration", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Report started`, 'info');
        const {
          components,
          apiId
        } = request.body;
        const {
          agentID
        } = request.params;
        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        let wmodulesResponse = {};
        let tables = [];

        try {
          wmodulesResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents/${agentID}/config/wmodules/wmodules`, {}, {
            apiHostID: apiId
          });
        } catch (error) {
          (0, _logger.log)('reporting:report', error.message || error, 'debug');
        }

        await this.renderHeader(context, printer, 'agentConfig', 'agentConfig', agentID, apiId);
        let idxComponent = 0;

        for (let config of _agentConfiguration.AgentConfiguration.configurations) {
          let titleOfSection = false;
          (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Iterate over ${config.sections.length} configuration sections`, 'debug');

          for (let section of config.sections) {
            let titleOfSubsection = false;

            if (components[idxComponent] && (section.config || section.wodle)) {
              let idx = 0;
              const configs = (section.config || []).concat(section.wodle || []);
              (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Iterate over ${configs.length} configuration blocks`, 'debug');

              for (let conf of configs) {
                let agentConfigResponse = {};

                try {
                  if (!conf['name']) {
                    agentConfigResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents/${agentID}/config/${conf.component}/${conf.configuration}`, {}, {
                      apiHostID: apiId
                    });
                  } else {
                    for (let wodle of wmodulesResponse.data.data['wmodules']) {
                      if (Object.keys(wodle)[0] === conf['name']) {
                        agentConfigResponse.data = {
                          data: wodle
                        };
                      }
                    }
                  }

                  const agentConfig = agentConfigResponse && agentConfigResponse.data && agentConfigResponse.data.data;

                  if (!titleOfSection) {
                    printer.addContent({
                      text: config.title,
                      style: 'h1',
                      margin: [0, 0, 0, 15]
                    });
                    titleOfSection = true;
                  }

                  if (!titleOfSubsection) {
                    printer.addContent({
                      text: section.subtitle,
                      style: 'h4'
                    });
                    printer.addContent({
                      text: section.desc,
                      style: {
                        fontSize: 12,
                        color: '#000'
                      },
                      margin: [0, 0, 0, 10]
                    });
                    titleOfSubsection = true;
                  }

                  if (agentConfig) {
                    for (let agentConfigKey of Object.keys(agentConfig)) {
                      if (Array.isArray(agentConfig[agentConfigKey])) {
                        /* LOG COLLECTOR */
                        if (conf.filterBy) {
                          let groups = [];
                          agentConfig[agentConfigKey].forEach(obj => {
                            if (!groups[obj.logformat]) {
                              groups[obj.logformat] = [];
                            }

                            groups[obj.logformat].push(obj);
                          });
                          Object.keys(groups).forEach(group => {
                            let saveidx = 0;
                            groups[group].forEach((x, i) => {
                              if (Object.keys(x).length > Object.keys(groups[group][saveidx]).length) {
                                saveidx = i;
                              }
                            });
                            const columns = Object.keys(groups[group][saveidx]);
                            const rows = groups[group].map(x => {
                              let row = [];
                              columns.forEach(key => {
                                row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
                                  return x + '\n';
                                }) : JSON.stringify(x[key]));
                              });
                              return row;
                            });
                            columns.forEach((col, i) => {
                              columns[i] = col[0].toUpperCase() + col.slice(1);
                            });
                            tables.push({
                              title: section.labels[0][group],
                              type: 'table',
                              columns,
                              rows
                            });
                          });
                        } else if (agentConfigKey.configuration !== 'socket') {
                          tables.push(...this.getConfigTables(agentConfig[agentConfigKey], section, idx));
                        } else {
                          for (let _d2 of agentConfig[agentConfigKey]) {
                            tables.push(...this.getConfigTables(_d2, section, idx));
                          }
                        }
                      } else {
                        /*INTEGRITY MONITORING MONITORED DIRECTORIES */
                        if (conf.matrix) {
                          const {
                            directories,
                            diff,
                            synchronization,
                            file_limit,
                            ...rest
                          } = agentConfig[agentConfigKey];
                          tables.push(...this.getConfigTables(rest, section, idx), ...(diff && diff.disk_quota ? this.getConfigTables(diff.disk_quota, {
                            tabs: ['Disk quota']
                          }, 0) : []), ...(diff && diff.file_size ? this.getConfigTables(diff.file_size, {
                            tabs: ['File size']
                          }, 0) : []), ...(synchronization ? this.getConfigTables(synchronization, {
                            tabs: ['Synchronization']
                          }, 0) : []), ...(file_limit ? this.getConfigTables(file_limit, {
                            tabs: ['File limit']
                          }, 0) : []));
                          let diffOpts = [];
                          Object.keys(section.opts).forEach(x => {
                            diffOpts.push(x);
                          });
                          const columns = ['', ...diffOpts.filter(x => x !== 'check_all' && x !== 'check_sum')];
                          let rows = [];
                          directories.forEach(x => {
                            let row = [];
                            row.push(x.dir);
                            columns.forEach(y => {
                              if (y !== '') {
                                row.push(x.opts.indexOf(y) > -1 ? 'yes' : 'no');
                              }
                            });
                            row.push(x.recursion_level);
                            rows.push(row);
                          });
                          columns.forEach((x, idx) => {
                            columns[idx] = section.opts[x];
                          });
                          columns.push('RL');
                          tables.push({
                            title: 'Monitored directories',
                            type: 'table',
                            columns,
                            rows
                          });
                        } else {
                          tables.push(...this.getConfigTables(agentConfig[agentConfigKey], section, idx));
                        }
                      }
                    }
                  } else {
                    // Print no configured module and link to the documentation
                    printer.addContent({
                      text: ['This module is not configured. Please take a look on how to configure it in ', {
                        text: `${section.subtitle.toLowerCase()} configuration.`,
                        link: section.docuLink,
                        style: {
                          fontSize: 12,
                          color: '#1a0dab'
                        }
                      }],
                      margin: [0, 0, 0, 20]
                    });
                  }
                } catch (error) {
                  (0, _logger.log)('reporting:report', error.message || error, 'debug');
                }

                idx++;
              }

              for (const table of tables) {
                printer.addConfigTables([table]);
              }
            }

            idxComponent++;
            tables = [];
          }
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsAgentsConfiguration', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        agentID
      }
    }) => `wazuh-agent-configuration-${agentID}-${this.generateReportTimestamp()}.pdf`));

    _defineProperty(this, "createReportsAgentsInventory", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsAgentsInventory', `Report started`, 'info');
        const {
          searchBar,
          filters,
          time,
          indexPatternTitle,
          apiId
        } = request.body;
        const {
          agentID
        } = request.params;
        const {
          from,
          to
        } = time || {}; // Init

        const printer = new _printer.ReportPrinter();
        const {
          hashUsername
        } = await context.wazuh.security.getCurrentUser(request, context);
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername));
        (0, _logger.log)('reporting:createReportsAgentsInventory', `Syscollector report`, 'debug');
        const sanitizedFilters = filters ? this.sanitizeKibanaFilters(filters, searchBar) : false; // Get the agent OS

        let agentOs = '';

        try {
          const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', '/agents', {
            params: {
              q: `id=${agentID}`
            }
          }, {
            apiHostID: apiId
          });
          agentOs = agentResponse.data.data.affected_items[0].os.platform;
        } catch (error) {
          (0, _logger.log)('reporting:createReportsAgentsInventory', error.message || error, 'debug');
        } // Add title


        printer.addContentWithNewLine({
          text: 'Inventory data report',
          style: 'h1'
        }); // Add table with the agent info

        await this.buildAgentsTable(context, printer, [agentID], apiId); // Get syscollector packages and processes

        const agentRequestsInventory = [{
          endpoint: `/syscollector/${agentID}/packages`,
          loggerMessage: `Fetching packages for agent ${agentID}`,
          table: {
            title: 'Packages',
            columns: agentOs === 'windows' ? [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'architecture',
              label: 'Architecture'
            }, {
              id: 'version',
              label: 'Version'
            }, {
              id: 'vendor',
              label: 'Vendor'
            }] : [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'architecture',
              label: 'Architecture'
            }, {
              id: 'version',
              label: 'Version'
            }, {
              id: 'vendor',
              label: 'Vendor'
            }, {
              id: 'description',
              label: 'Description'
            }]
          }
        }, {
          endpoint: `/syscollector/${agentID}/processes`,
          loggerMessage: `Fetching processes for agent ${agentID}`,
          table: {
            title: 'Processes',
            columns: agentOs === 'windows' ? [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'cmd',
              label: 'CMD'
            }, {
              id: 'priority',
              label: 'Priority'
            }, {
              id: 'nlwp',
              label: 'NLWP'
            }] : [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'euser',
              label: 'Effective user'
            }, {
              id: 'nice',
              label: 'Priority'
            }, {
              id: 'state',
              label: 'State'
            }]
          },
          mapResponseItems: item => agentOs === 'windows' ? item : { ...item,
            state: _processStateEquivalence.default[item.state]
          }
        }, {
          endpoint: `/syscollector/${agentID}/ports`,
          loggerMessage: `Fetching ports for agent ${agentID}`,
          table: {
            title: 'Network ports',
            columns: agentOs === 'windows' ? [{
              id: 'local_ip',
              label: 'Local IP'
            }, {
              id: 'local_port',
              label: 'Local port'
            }, {
              id: 'process',
              label: 'Process'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'protocol',
              label: 'Protocol'
            }] : [{
              id: 'local_ip',
              label: 'Local IP'
            }, {
              id: 'local_port',
              label: 'Local port'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'protocol',
              label: 'Protocol'
            }]
          },
          mapResponseItems: item => ({ ...item,
            local_ip: item.local.ip,
            local_port: item.local.port
          })
        }, {
          endpoint: `/syscollector/${agentID}/netiface`,
          loggerMessage: `Fetching netiface for agent ${agentID}`,
          table: {
            title: 'Network interfaces',
            columns: [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'mac',
              label: 'Mac'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'mtu',
              label: 'MTU'
            }, {
              id: 'type',
              label: 'Type'
            }]
          }
        }, {
          endpoint: `/syscollector/${agentID}/netaddr`,
          loggerMessage: `Fetching netaddr for agent ${agentID}`,
          table: {
            title: 'Network settings',
            columns: [{
              id: 'iface',
              label: 'Interface'
            }, {
              id: 'address',
              label: 'address'
            }, {
              id: 'netmask',
              label: 'Netmask'
            }, {
              id: 'proto',
              label: 'Protocol'
            }, {
              id: 'broadcast',
              label: 'Broadcast'
            }]
          }
        }];
        agentOs === 'windows' && agentRequestsInventory.push({
          endpoint: `/syscollector/${agentID}/hotfixes`,
          loggerMessage: `Fetching hotfixes for agent ${agentID}`,
          table: {
            title: 'Windows updates',
            columns: [{
              id: 'hotfix',
              label: 'Update code'
            }]
          }
        });

        const requestInventory = async agentRequestInventory => {
          try {
            (0, _logger.log)('reporting:createReportsAgentsInventory', agentRequestInventory.loggerMessage, 'debug');
            const inventoryResponse = await context.wazuh.api.client.asCurrentUser.request('GET', agentRequestInventory.endpoint, {}, {
              apiHostID: apiId
            });
            const inventory = inventoryResponse && inventoryResponse.data && inventoryResponse.data.data && inventoryResponse.data.data.affected_items;

            if (inventory) {
              return { ...agentRequestInventory.table,
                items: agentRequestInventory.mapResponseItems ? inventory.map(agentRequestInventory.mapResponseItems) : inventory
              };
            }
          } catch (error) {
            (0, _logger.log)('reporting:createReportsAgentsInventory', error.message || error, 'debug');
          }
        };

        if (time) {
          await this.extendedInformation(context, printer, 'agents', 'syscollector', apiId, from, to, sanitizedFilters + ' AND rule.groups: "vulnerability-detector"', indexPatternTitle, agentID);
        } // Add inventory tables


        (await Promise.all(agentRequestsInventory.map(requestInventory))).filter(table => table).forEach(table => printer.addSimpleTable(table)); // Print the document

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsAgents', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        agentID
      }
    }) => `wazuh-agent-inventory-${agentID}-${this.generateReportTimestamp()}.pdf`));

    _defineProperty(this, "getReportByName", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:getReportByName', `Getting ${context.wazuhEndpointParams.pathFilename} report`, 'debug');

        const reportFileBuffer = _fs.default.readFileSync(context.wazuhEndpointParams.pathFilename);

        return response.ok({
          headers: {
            'Content-Type': 'application/pdf'
          },
          body: reportFileBuffer
        });
      } catch (error) {
        (0, _logger.log)('reporting:getReportByName', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5030, 500, response);
      }
    }, request => request.params.name));

    _defineProperty(this, "deleteReportByName", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:deleteReportByName', `Deleting ${context.wazuhEndpointParams.pathFilename} report`, 'debug');

        _fs.default.unlinkSync(context.wazuhEndpointParams.pathFilename);

        (0, _logger.log)('reporting:deleteReportByName', `${context.wazuhEndpointParams.pathFilename} report was deleted`, 'info');
        return response.ok({
          body: {
            error: 0
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:deleteReportByName', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5032, 500, response);
      }
    }, request => request.params.name));
  }
  /**
   * This do format to filters
   * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
   * @param {String} searchBar search term
   */


  sanitizeKibanaFilters(filters, searchBar) {
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `Started to sanitize filters`, 'info');
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `filters: ${filters.length}, searchBar: ${searchBar}`, 'debug');
    let str = '';
    const agentsFilter = []; //separate agents filter

    filters = filters.filter(filter => {
      if (filter.meta.controlledBy === _constants.AUTHORIZED_AGENTS) {
        agentsFilter.push(filter);
        return false;
      }

      return filter;
    });
    const len = filters.length;

    for (let i = 0; i < len; i++) {
      const {
        negate,
        key,
        value,
        params,
        type
      } = filters[i].meta;
      str += `${negate ? 'NOT ' : ''}`;
      str += `${key}: `;
      str += `${type === 'range' ? `${params.gte}-${params.lt}` : type === 'phrases' ? '(' + params.join(" OR ") + ')' : type === 'exists' ? '*' : !!value ? value : (params || {}).query}`;
      str += `${i === len - 1 ? '' : ' AND '}`;
    }

    if (searchBar) {
      str += ` AND (${searchBar})`;
    }

    const agentsFilterStr = agentsFilter.map(filter => filter.meta.value).join(',');
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `str: ${str}, agentsFilterStr: ${agentsFilterStr}`, 'debug');
    return [str, agentsFilterStr];
  }
  /**
   * This performs the rendering of given header
   * @param {String} printer section target
   * @param {String} section section target
   * @param {Object} tab tab target
   * @param {Boolean} isAgents is agents section
   * @param {String} apiId ID of API
   */


  async renderHeader(context, printer, section, tab, isAgents, apiId) {
    try {
      (0, _logger.log)('reporting:renderHeader', `section: ${section}, tab: ${tab}, isAgents: ${isAgents}, apiId: ${apiId}`, 'debug');

      if (section && typeof section === 'string') {
        if (!['agentConfig', 'groupConfig'].includes(section)) {
          printer.addContent({
            text: _wazuhModules.WAZUH_MODULES[tab].title + ' report',
            style: 'h1'
          });
        } else if (section === 'agentConfig') {
          printer.addContent({
            text: `Agent ${isAgents} configuration`,
            style: 'h1'
          });
        } else if (section === 'groupConfig') {
          printer.addContent({
            text: 'Agents in group',
            style: 'h1'
          });
        }

        printer.addNewLine();
      }

      if (isAgents && typeof isAgents === 'object') {
        await this.buildAgentsTable(context, printer, isAgents, apiId, section === 'groupConfig' ? tab : '');
      }

      if (isAgents && typeof isAgents === 'string') {
        const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents`, {
          params: {
            agents_list: isAgents
          }
        }, {
          apiHostID: apiId
        });
        const agentData = agentResponse.data.data.affected_items[0];

        if (agentData && agentData.status !== _constants.API_NAME_AGENT_STATUS.ACTIVE) {
          printer.addContentWithNewLine({
            text: `Warning. Agent is ${(0, _wz_agent_status.agentStatusLabelByAgentStatus)(agentData.status).toLowerCase()}`,
            style: 'standard'
          });
        }

        await this.buildAgentsTable(context, printer, [isAgents], apiId);

        if (agentData && agentData.group) {
          const agentGroups = agentData.group.join(', ');
          printer.addContentWithNewLine({
            text: `Group${agentData.group.length > 1 ? 's' : ''}: ${agentGroups}`,
            style: 'standard'
          });
        }
      }

      if (_wazuhModules.WAZUH_MODULES[tab] && _wazuhModules.WAZUH_MODULES[tab].description) {
        printer.addContentWithNewLine({
          text: _wazuhModules.WAZUH_MODULES[tab].description,
          style: 'standard'
        });
      }
    } catch (error) {
      (0, _logger.log)('reporting:renderHeader', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This build the agents table
   * @param {Array<Strings>} ids ids of agents
   * @param {String} apiId API id
   */


  async buildAgentsTable(context, printer, agentIDs, apiId, groupID = '') {
    const dateFormat = await context.core.uiSettings.client.get('dateFormat');
    if ((!agentIDs || !agentIDs.length) && !groupID) return;
    (0, _logger.log)('reporting:buildAgentsTable', `${agentIDs.length} agents for API ${apiId}`, 'info');

    try {
      let agentsData = [];

      if (groupID) {
        let totalAgentsInGroup = null;

        do {
          const {
            data: {
              data: {
                affected_items,
                total_affected_items
              }
            }
          } = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${groupID}/agents`, {
            params: {
              offset: agentsData.length,
              select: 'dateAdd,id,ip,lastKeepAlive,manager,name,os.name,os.version,version'
            }
          }, {
            apiHostID: apiId
          });
          !totalAgentsInGroup && (totalAgentsInGroup = total_affected_items);
          agentsData = [...agentsData, ...affected_items];
        } while (agentsData.length < totalAgentsInGroup);
      } else {
        for (const agentID of agentIDs) {
          try {
            const {
              data: {
                data: {
                  affected_items: [agent]
                }
              }
            } = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents`, {
              params: {
                q: `id=${agentID}`,
                select: 'dateAdd,id,ip,lastKeepAlive,manager,name,os.name,os.version,version'
              }
            }, {
              apiHostID: apiId
            });
            agentsData.push(agent);
          } catch (error) {
            (0, _logger.log)('reporting:buildAgentsTable', `Skip agent due to: ${error.message || error}`, 'debug');
          }
        }
      }

      if (agentsData.length) {
        // Print a table with agent/s information
        printer.addSimpleTable({
          columns: [{
            id: 'id',
            label: 'ID'
          }, {
            id: 'name',
            label: 'Name'
          }, {
            id: 'ip',
            label: 'IP'
          }, {
            id: 'version',
            label: 'Version'
          }, {
            id: 'manager',
            label: 'Manager'
          }, {
            id: 'os',
            label: 'OS'
          }, {
            id: 'dateAdd',
            label: 'Registration date'
          }, {
            id: 'lastKeepAlive',
            label: 'Last keep alive'
          }],
          items: agentsData.map(agent => {
            return { ...agent,
              os: agent.os && agent.os.name && agent.os.version ? `${agent.os.name} ${agent.os.version}` : '',
              lastKeepAlive: (0, _moment.default)(agent.lastKeepAlive).format(dateFormat),
              dateAdd: (0, _moment.default)(agent.dateAdd).format(dateFormat)
            };
          })
        });
      } else if (!agentsData.length && groupID) {
        // For group reports when there is no agents in the group
        printer.addContent({
          text: 'There are no agents in this group.',
          style: {
            fontSize: 12,
            color: '#000'
          }
        });
      }
    } catch (error) {
      (0, _logger.log)('reporting:buildAgentsTable', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This load more information
   * @param {*} context Endpoint context
   * @param {*} printer printer instance
   * @param {String} section section target
   * @param {Object} tab tab target
   * @param {String} apiId ID of API
   * @param {Number} from Timestamp (ms) from
   * @param {Number} to Timestamp (ms) to
   * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
   * @param {String} pattern
   * @param {Object} agent agent target
   * @returns {Object} Extended information
   */


  async extendedInformation(context, printer, section, tab, apiId, from, to, filters, pattern = _constants.WAZUH_ALERTS_PATTERN, agent = null) {
    try {
      (0, _logger.log)('reporting:extendedInformation', `Section ${section} and tab ${tab}, API is ${apiId}. From ${from} to ${to}. Filters ${filters}. Index pattern ${pattern}`, 'info');

      if (section === 'agents' && !agent) {
        throw new Error('Reporting for specific agent needs an agent ID in order to work properly');
      }

      const agents = await context.wazuh.api.client.asCurrentUser.request('GET', '/agents', {
        params: {
          limit: 1
        }
      }, {
        apiHostID: apiId
      });
      const totalAgents = agents.data.data.total_affected_items;

      if (section === 'overview' && tab === 'vuls') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector metrics', 'debug');
        const vulnerabilitiesLevels = ['Low', 'Medium', 'High', 'Critical'];
        const vulnerabilitiesResponsesCount = (await Promise.all(vulnerabilitiesLevels.map(async vulnerabilitiesLevel => {
          try {
            const count = await VulnerabilityRequest.uniqueSeverityCount(context, from, to, vulnerabilitiesLevel, filters, pattern);
            return count ? `${count} of ${totalAgents} agents have ${vulnerabilitiesLevel.toLocaleLowerCase()} vulnerabilities.` : undefined;
          } catch (error) {}
        }))).filter(vulnerabilitiesResponse => vulnerabilitiesResponse);
        printer.addList({
          title: {
            text: 'Summary',
            style: 'h2'
          },
          list: vulnerabilitiesResponsesCount
        });
        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector top 3 agents by category', 'debug');
        const lowRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Low', filters, pattern);
        const mediumRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Medium', filters, pattern);
        const highRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'High', filters, pattern);
        const criticalRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Critical', filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding overview vulnerability detector top 3 agents by category', 'debug');

        if (criticalRank && criticalRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with critical severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, criticalRank, apiId);
          printer.addNewLine();
        }

        if (highRank && highRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with high severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, highRank, apiId);
          printer.addNewLine();
        }

        if (mediumRank && mediumRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with medium severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, mediumRank, apiId);
          printer.addNewLine();
        }

        if (lowRank && lowRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with low severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, lowRank, apiId);
          printer.addNewLine();
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector top 3 CVEs', 'debug');
        const cveRank = await VulnerabilityRequest.topCVECount(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding overview vulnerability detector top 3 CVEs', 'debug');

        if (cveRank && cveRank.length) {
          printer.addSimpleTable({
            title: {
              text: 'Top 3 CVE',
              style: 'h2'
            },
            columns: [{
              id: 'top',
              label: 'Top'
            }, {
              id: 'cve',
              label: 'CVE'
            }],
            items: cveRank.map(item => ({
              top: cveRank.indexOf(item) + 1,
              cve: item
            }))
          });
        }
      }

      if (section === 'overview' && tab === 'general') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 agents with level 15 alerts', 'debug');
        const level15Rank = await OverviewRequest.topLevel15(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding top 3 agents with level 15 alerts', 'debug');

        if (level15Rank.length) {
          printer.addContent({
            text: 'Top 3 agents with level 15 alerts',
            style: 'h2'
          });
          await this.buildAgentsTable(context, printer, level15Rank, apiId);
        }
      }

      if (section === 'overview' && tab === 'pm') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching most common rootkits', 'debug');
        const top5RootkitsRank = await RootcheckRequest.top5RootkitsDetected(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding most common rootkits', 'debug');

        if (top5RootkitsRank && top5RootkitsRank.length) {
          printer.addContentWithNewLine({
            text: 'Most common rootkits found among your agents',
            style: 'h2'
          }).addContentWithNewLine({
            text: 'Rootkits are a set of software tools that enable an unauthorized user to gain control of a computer system without being detected.',
            style: 'standard'
          }).addSimpleTable({
            items: top5RootkitsRank.map(item => {
              return {
                top: top5RootkitsRank.indexOf(item) + 1,
                name: item
              };
            }),
            columns: [{
              id: 'top',
              label: 'Top'
            }, {
              id: 'name',
              label: 'Rootkit'
            }]
          });
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching hidden pids', 'debug');
        const hiddenPids = await RootcheckRequest.agentsWithHiddenPids(context, from, to, filters, pattern);
        hiddenPids && printer.addContent({
          text: `${hiddenPids} of ${totalAgents} agents have hidden processes`,
          style: 'h3'
        });
        !hiddenPids && printer.addContentWithNewLine({
          text: `No agents have hidden processes`,
          style: 'h3'
        });
        const hiddenPorts = await RootcheckRequest.agentsWithHiddenPorts(context, from, to, filters, pattern);
        hiddenPorts && printer.addContent({
          text: `${hiddenPorts} of ${totalAgents} agents have hidden ports`,
          style: 'h3'
        });
        !hiddenPorts && printer.addContent({
          text: `No agents have hidden ports`,
          style: 'h3'
        });
        printer.addNewLine();
      }

      if (['overview', 'agents'].includes(section) && tab === 'pci') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top PCI DSS requirements', 'debug');
        const topPciRequirements = await PCIRequest.topPCIRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common PCI DSS requirements alerts found',
          style: 'h2'
        });

        for (const item of topPciRequirements) {
          const rules = await PCIRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_pciRequirementsPdfmake.default[item]) {
            const content = typeof _pciRequirementsPdfmake.default[item] === 'string' ? {
              text: _pciRequirementsPdfmake.default[item],
              style: 'standard'
            } : _pciRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }
      }

      if (['overview', 'agents'].includes(section) && tab === 'tsc') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top TSC requirements', 'debug');
        const topTSCRequirements = await TSCRequest.topTSCRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common TSC requirements alerts found',
          style: 'h2'
        });

        for (const item of topTSCRequirements) {
          const rules = await TSCRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_tscRequirementsPdfmake.default[item]) {
            const content = typeof _tscRequirementsPdfmake.default[item] === 'string' ? {
              text: _tscRequirementsPdfmake.default[item],
              style: 'standard'
            } : _tscRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }
      }

      if (['overview', 'agents'].includes(section) && tab === 'gdpr') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top GDPR requirements', 'debug');
        const topGdprRequirements = await GDPRRequest.topGDPRRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common GDPR requirements alerts found',
          style: 'h2'
        });

        for (const item of topGdprRequirements) {
          const rules = await GDPRRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_gdprRequirementsPdfmake.default && _gdprRequirementsPdfmake.default[item]) {
            const content = typeof _gdprRequirementsPdfmake.default[item] === 'string' ? {
              text: _gdprRequirementsPdfmake.default[item],
              style: 'standard'
            } : _gdprRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }

        printer.addNewLine();
      }

      if (section === 'overview' && tab === 'audit') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching agents with high number of failed sudo commands', 'debug');
        const auditAgentsNonSuccess = await AuditRequest.getTop3AgentsSudoNonSuccessful(context, from, to, filters, pattern);

        if (auditAgentsNonSuccess && auditAgentsNonSuccess.length) {
          printer.addContent({
            text: 'Agents with high number of failed sudo commands',
            style: 'h2'
          });
          await this.buildAgentsTable(context, printer, auditAgentsNonSuccess, apiId);
        }

        const auditAgentsFailedSyscall = await AuditRequest.getTop3AgentsFailedSyscalls(context, from, to, filters, pattern);

        if (auditAgentsFailedSyscall && auditAgentsFailedSyscall.length) {
          printer.addSimpleTable({
            columns: [{
              id: 'agent',
              label: 'Agent ID'
            }, {
              id: 'syscall_id',
              label: 'Syscall ID'
            }, {
              id: 'syscall_syscall',
              label: 'Syscall'
            }],
            items: auditAgentsFailedSyscall.map(item => ({
              agent: item.agent,
              syscall_id: item.syscall.id,
              syscall_syscall: item.syscall.syscall
            })),
            title: {
              text: 'Most common failing syscalls',
              style: 'h2'
            }
          });
        }
      }

      if (section === 'overview' && tab === 'fim') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 rules for FIM', 'debug');
        const rules = await SyscheckRequest.top3Rules(context, from, to, filters, pattern);

        if (rules && rules.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 FIM rules',
            style: 'h2'
          }).addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: {
              text: 'Top 3 rules that are generating most alerts.',
              style: 'standard'
            }
          });
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 agents for FIM', 'debug');
        const agents = await SyscheckRequest.top3agents(context, from, to, filters, pattern);

        if (agents && agents.length) {
          printer.addContentWithNewLine({
            text: 'Agents with suspicious FIM activity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'Top 3 agents that have most FIM alerts from level 7 to level 15. Take care about them.',
            style: 'standard'
          });
          await this.buildAgentsTable(context, printer, agents, apiId);
        }
      }

      if (section === 'agents' && tab === 'audit') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching most common failed syscalls`, 'debug');
        const auditFailedSyscall = await AuditRequest.getTopFailedSyscalls(context, from, to, filters, pattern);
        auditFailedSyscall && auditFailedSyscall.length && printer.addSimpleTable({
          columns: [{
            id: 'id',
            label: 'id'
          }, {
            id: 'syscall',
            label: 'Syscall'
          }],
          items: auditFailedSyscall,
          title: 'Most common failing syscalls'
        });
      }

      if (section === 'agents' && tab === 'fim') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching syscheck database for agent ${agent}`, 'debug');
        const lastScanResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/syscheck/${agent}/last_scan`, {}, {
          apiHostID: apiId
        });

        if (lastScanResponse && lastScanResponse.data) {
          const lastScanData = lastScanResponse.data.data.affected_items[0];

          if (lastScanData.start && lastScanData.end) {
            printer.addContent({
              text: `Last file integrity monitoring scan was executed from ${lastScanData.start} to ${lastScanData.end}.`
            });
          } else if (lastScanData.start) {
            printer.addContent({
              text: `File integrity monitoring scan is currently in progress for this agent (started on ${lastScanData.start}).`
            });
          } else {
            printer.addContent({
              text: `File integrity monitoring scan is currently in progress for this agent.`
            });
          }

          printer.addNewLine();
        }

        (0, _logger.log)('reporting:extendedInformation', `Fetching last 10 deleted files for FIM`, 'debug');
        const lastTenDeleted = await SyscheckRequest.lastTenDeletedFiles(context, from, to, filters, pattern);
        lastTenDeleted && lastTenDeleted.length && printer.addSimpleTable({
          columns: [{
            id: 'path',
            label: 'Path'
          }, {
            id: 'date',
            label: 'Date'
          }],
          items: lastTenDeleted,
          title: 'Last 10 deleted files'
        });
        (0, _logger.log)('reporting:extendedInformation', `Fetching last 10 modified files`, 'debug');
        const lastTenModified = await SyscheckRequest.lastTenModifiedFiles(context, from, to, filters, pattern);
        lastTenModified && lastTenModified.length && printer.addSimpleTable({
          columns: [{
            id: 'path',
            label: 'Path'
          }, {
            id: 'date',
            label: 'Date'
          }],
          items: lastTenModified,
          title: 'Last 10 modified files'
        });
      }

      if (section === 'agents' && tab === 'syscollector') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching hardware information for agent ${agent}`, 'debug');
        const requestsSyscollectorLists = [{
          endpoint: `/syscollector/${agent}/hardware`,
          loggerMessage: `Fetching Hardware information for agent ${agent}`,
          list: {
            title: {
              text: 'Hardware information',
              style: 'h2'
            }
          },
          mapResponse: hardware => [hardware.cpu && hardware.cpu.cores && `${hardware.cpu.cores} cores`, hardware.cpu && hardware.cpu.name, hardware.ram && hardware.ram.total && `${Number(hardware.ram.total / 1024 / 1024).toFixed(2)}GB RAM`]
        }, {
          endpoint: `/syscollector/${agent}/os`,
          loggerMessage: `Fetching OS information for agent ${agent}`,
          list: {
            title: {
              text: 'OS information',
              style: 'h2'
            }
          },
          mapResponse: osData => [osData.sysname, osData.version, osData.architecture, osData.release, osData.os && osData.os.name && osData.os.version && `${osData.os.name} ${osData.os.version}`]
        }];
        const syscollectorLists = await Promise.all(requestsSyscollectorLists.map(async requestSyscollector => {
          try {
            (0, _logger.log)('reporting:extendedInformation', requestSyscollector.loggerMessage, 'debug');
            const responseSyscollector = await context.wazuh.api.client.asCurrentUser.request('GET', requestSyscollector.endpoint, {}, {
              apiHostID: apiId
            });
            const [data] = responseSyscollector && responseSyscollector.data && responseSyscollector.data.data && responseSyscollector.data.data.affected_items || [];

            if (data) {
              return { ...requestSyscollector.list,
                list: requestSyscollector.mapResponse(data)
              };
            }
          } catch (error) {
            (0, _logger.log)('reporting:extendedInformation', error.message || error);
          }
        }));

        if (syscollectorLists) {
          syscollectorLists.filter(syscollectorList => syscollectorList).forEach(syscollectorList => printer.addList(syscollectorList));
        }

        const vulnerabilitiesRequests = ['Critical', 'High'];
        const vulnerabilitiesResponsesItems = (await Promise.all(vulnerabilitiesRequests.map(async vulnerabilitiesLevel => {
          try {
            (0, _logger.log)('reporting:extendedInformation', `Fetching top ${vulnerabilitiesLevel} packages`, 'debug');
            return await VulnerabilityRequest.topPackages(context, from, to, vulnerabilitiesLevel, filters, pattern);
          } catch (error) {
            (0, _logger.log)('reporting:extendedInformation', error.message || error);
          }
        }))).filter(vulnerabilitiesResponse => vulnerabilitiesResponse).flat();

        if (vulnerabilitiesResponsesItems && vulnerabilitiesResponsesItems.length) {
          printer.addSimpleTable({
            title: {
              text: 'Vulnerable packages found (last 24 hours)',
              style: 'h2'
            },
            columns: [{
              id: 'package',
              label: 'Package'
            }, {
              id: 'severity',
              label: 'Severity'
            }],
            items: vulnerabilitiesResponsesItems
          });
        }
      }

      if (section === 'agents' && tab === 'vuls') {
        const topCriticalPackages = await VulnerabilityRequest.topPackagesWithCVE(context, from, to, 'Critical', filters, pattern);

        if (topCriticalPackages && topCriticalPackages.length) {
          printer.addContentWithNewLine({
            text: 'Critical severity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'These vulnerabilties are critical, please review your agent. Click on each link to read more about each found vulnerability.',
            style: 'standard'
          });
          const customul = [];

          for (const critical of topCriticalPackages) {
            customul.push({
              text: critical.package,
              style: 'standard'
            });
            customul.push({
              ul: critical.references.map(item => ({
                text: item.substring(0, 80) + '...',
                link: item,
                color: '#1EA5C8'
              }))
            });
          }

          printer.addContentWithNewLine({
            ul: customul
          });
        }

        const topHighPackages = await VulnerabilityRequest.topPackagesWithCVE(context, from, to, 'High', filters, pattern);

        if (topHighPackages && topHighPackages.length) {
          printer.addContentWithNewLine({
            text: 'High severity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'Click on each link to read more about each found vulnerability.',
            style: 'standard'
          });
          const customul = [];

          for (const critical of topHighPackages) {
            customul.push({
              text: critical.package,
              style: 'standard'
            });
            customul.push({
              ul: critical.references.map(item => ({
                text: item,
                color: '#1EA5C8'
              }))
            });
          }

          customul && customul.length && printer.addContent({
            ul: customul
          });
          printer.addNewLine();
        }
      }

      return false;
    } catch (error) {
      (0, _logger.log)('reporting:extendedInformation', error.message || error);
      return Promise.reject(error);
    }
  }

  getConfigRows(data, labels) {
    (0, _logger.log)('reporting:getConfigRows', `Building configuration rows`, 'info');
    const result = [];

    for (let prop in data || []) {
      if (Array.isArray(data[prop])) {
        data[prop].forEach((x, idx) => {
          if (typeof x === 'object') data[prop][idx] = JSON.stringify(x);
        });
      }

      result.push([(labels || {})[prop] || _csvKeyEquivalence.KeyEquivalence[prop] || prop, data[prop] || '-']);
    }

    return result;
  }

  getConfigTables(data, section, tab, array = []) {
    (0, _logger.log)('reporting:getConfigTables', `Building configuration tables`, 'info');
    let plainData = {};
    const nestedData = [];
    const tableData = [];

    if (data.length === 1 && Array.isArray(data)) {
      tableData[section.config[tab].configuration] = data;
    } else {
      for (let key in data) {
        if (typeof data[key] !== 'object' && !Array.isArray(data[key]) || Array.isArray(data[key]) && typeof data[key][0] !== 'object') {
          plainData[key] = Array.isArray(data[key]) && typeof data[key][0] !== 'object' ? data[key].map(x => {
            return typeof x === 'object' ? JSON.stringify(x) : x + '\n';
          }) : data[key];
        } else if (Array.isArray(data[key]) && typeof data[key][0] === 'object') {
          tableData[key] = data[key];
        } else {
          if (section.isGroupConfig && ['pack', 'content'].includes(key)) {
            tableData[key] = [data[key]];
          } else {
            nestedData.push(data[key]);
          }
        }
      }
    }

    array.push({
      title: (section.options || {}).hideHeader ? '' : (section.tabs || [])[tab] || (section.isGroupConfig ? ((section.labels || [])[0] || [])[tab] : ''),
      columns: ['', ''],
      type: 'config',
      rows: this.getConfigRows(plainData, (section.labels || [])[0])
    });

    for (let key in tableData) {
      const columns = Object.keys(tableData[key][0]);
      columns.forEach((col, i) => {
        columns[i] = col[0].toUpperCase() + col.slice(1);
      });
      const rows = tableData[key].map(x => {
        let row = [];

        for (let key in x) {
          row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
            return x + '\n';
          }) : JSON.stringify(x[key]));
        }

        while (row.length < columns.length) {
          row.push('-');
        }

        return row;
      });
      array.push({
        title: ((section.labels || [])[0] || [])[key] || '',
        type: 'table',
        columns,
        rows
      });
    }

    nestedData.forEach(nest => {
      this.getConfigTables(nest, section, tab + 1, array);
    });
    return array;
  }
  /**
   * Create a report for the modules
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {*} reports list or ErrorResponse
   */


  /**
   * Fetch the reports list
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Array<Object>} reports list or ErrorResponse
   */
  async getReports(context, request, response) {
    try {
      (0, _logger.log)('reporting:getReports', `Fetching created reports`, 'info');
      const {
        hashUsername
      } = await context.wazuh.security.getCurrentUser(request, context);
      (0, _filesystem.createDataDirectoryIfNotExists)();
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);

      const userReportsDirectoryPath = _path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername);

      (0, _filesystem.createDirectoryIfNotExists)(userReportsDirectoryPath);
      (0, _logger.log)('reporting:getReports', `Directory: ${userReportsDirectoryPath}`, 'debug');

      const sortReportsByDate = (a, b) => a.date < b.date ? 1 : a.date > b.date ? -1 : 0;

      const reports = _fs.default.readdirSync(userReportsDirectoryPath).map(file => {
        const stats = _fs.default.statSync(userReportsDirectoryPath + '/' + file); // Get the file creation time (bithtime). It returns the first value that is a truthy value of next file stats: birthtime, mtime, ctime and atime.
        // This solves some OSs can have the bithtimeMs equal to 0 and returns the date like 1970-01-01


        const birthTimeField = ['birthtime', 'mtime', 'ctime', 'atime'].find(time => stats[`${time}Ms`]);
        return {
          name: file,
          size: stats.size,
          date: stats[birthTimeField]
        };
      });

      (0, _logger.log)('reporting:getReports', `Using TimSort for sorting ${reports.length} items`, 'debug');
      TimSort.sort(reports, sortReportsByDate);
      (0, _logger.log)('reporting:getReports', `Total reports: ${reports.length}`, 'debug');
      return response.ok({
        body: {
          reports
        }
      });
    } catch (error) {
      (0, _logger.log)('reporting:getReports', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5031, 500, response);
    }
  }
  /**
   * Fetch specific report
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} report or ErrorResponse
   */


  checkReportsUserDirectoryIsValidRouteDecorator(routeHandler, reportFileNameAccessor) {
    return async (context, request, response) => {
      try {
        const {
          username,
          hashUsername
        } = await context.wazuh.security.getCurrentUser(request, context);

        const userReportsDirectoryPath = _path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername);

        const filename = reportFileNameAccessor(request);

        const pathFilename = _path.default.join(userReportsDirectoryPath, filename);

        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', `Checking the user ${username}(${hashUsername}) can do actions in the reports file: ${pathFilename}`, 'debug');

        if (!pathFilename.startsWith(userReportsDirectoryPath) || pathFilename.includes('../')) {
          (0, _logger.log)('security:reporting:checkReportsUserDirectoryIsValidRouteDecorator', `User ${username}(${hashUsername}) tried to access to a non user report file: ${pathFilename}`, 'warn');
          return response.badRequest({
            body: {
              message: '5040 - You shall not pass!'
            }
          });
        }

        ;
        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', 'Checking the user can do actions in the reports file', 'debug');
        return await routeHandler.bind(this)({ ...context,
          wazuhEndpointParams: {
            hashUsername,
            filename,
            pathFilename
          }
        }, request, response);
      } catch (error) {
        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5040, 500, response);
      }
    };
  }

  generateReportTimestamp() {
    return `${Date.now() / 1000 | 0}`;
  }

}

exports.WazuhReportingCtrl = WazuhReportingCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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