"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BaseLogger = void 0;

var _winston = _interopRequireDefault(require("winston"));

var _fs = _interopRequireDefault(require("fs"));

var _path = _interopRequireDefault(require("path"));

var _getConfiguration = require("./get-configuration");

var _filesystem = require("./filesystem");

var _constants = require("../../common/constants");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class BaseLogger {
  constructor(plainLogsFile, rawLogsFile) {
    _defineProperty(this, "allowed", false);

    _defineProperty(this, "wazuhLogger", undefined);

    _defineProperty(this, "wazuhPlainLogger", undefined);

    _defineProperty(this, "PLAIN_LOGS_PATH", '');

    _defineProperty(this, "PLAIN_LOGS_FILE_NAME", '');

    _defineProperty(this, "RAW_LOGS_PATH", '');

    _defineProperty(this, "RAW_LOGS_FILE_NAME", '');

    _defineProperty(this, "initLogger", () => {
      const configurationFile = (0, _getConfiguration.getConfiguration)();
      const level = typeof (configurationFile || {})['logs.level'] !== 'undefined' && ['info', 'debug'].includes(configurationFile['logs.level']) ? configurationFile['logs.level'] : 'info'; // JSON logger

      this.wazuhLogger = _winston.default.createLogger({
        level,
        format: _winston.default.format.json(),
        transports: [new _winston.default.transports.File({
          filename: this.RAW_LOGS_PATH
        })]
      }); // Prevents from exit on error related to the logger.

      this.wazuhLogger.exitOnError = false; // Plain text logger

      this.wazuhPlainLogger = _winston.default.createLogger({
        level,
        format: _winston.default.format.simple(),
        transports: [new _winston.default.transports.File({
          filename: this.PLAIN_LOGS_PATH
        })]
      }); // Prevents from exit on error related to the logger.

      this.wazuhPlainLogger.exitOnError = false;
    });

    _defineProperty(this, "initDirectory", async () => {
      try {
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDataDirectoryIfNotExists)('logs');

        if (typeof this.wazuhLogger === 'undefined' || typeof this.wazuhPlainLogger === 'undefined') {
          this.initLogger();
        }

        this.allowed = true;
        return;
      } catch (error) {
        this.allowed = false;
        return Promise.reject(error);
      }
    });

    _defineProperty(this, "getFilesizeInMegaBytes", filename => {
      if (this.allowed) {
        if (_fs.default.existsSync(filename)) {
          const stats = _fs.default.statSync(filename);

          const fileSizeInMegaBytes = stats.size;
          return fileSizeInMegaBytes / 1000000.0;
        }
      }

      return 0;
    });

    _defineProperty(this, "checkFileExist", filename => {
      return _fs.default.existsSync(filename);
    });

    _defineProperty(this, "rotateFiles", (file, pathFile, log) => {
      if (this.getFilesizeInMegaBytes(pathFile) >= _constants.MAX_MB_LOG_FILES) {
        const fileExtension = _path.default.extname(file);

        const fileName = _path.default.basename(file, fileExtension);

        _fs.default.renameSync(pathFile, `${_constants.WAZUH_DATA_LOGS_DIRECTORY_PATH}/${fileName}-${new Date().getTime()}${fileExtension}`);

        if (log) {
          _fs.default.writeFileSync(pathFile, log + '\n');
        }
      }
    });

    _defineProperty(this, "checkFiles", () => {
      (0, _filesystem.createLogFileIfNotExists)(this.RAW_LOGS_PATH);
      (0, _filesystem.createLogFileIfNotExists)(this.PLAIN_LOGS_PATH);

      if (this.allowed) {
        // check raw log file
        this.rotateFiles(this.RAW_LOGS_FILE_NAME, this.RAW_LOGS_PATH, JSON.stringify({
          date: new Date(),
          level: 'info',
          location: 'logger',
          message: 'Rotated log file'
        })); // check log file

        this.rotateFiles(this.PLAIN_LOGS_FILE_NAME, this.PLAIN_LOGS_PATH);
      }
    });

    _defineProperty(this, "yyyymmdd", () => {
      const now = new Date();
      const y = now.getFullYear();
      const m = now.getMonth() + 1;
      const d = now.getDate();
      const seconds = now.getSeconds();
      const minutes = now.getMinutes();
      const hour = now.getHours();
      return `${y}/${m < 10 ? '0' : ''}${m}/${d < 10 ? '0' : ''}${d} ${hour}:${minutes}:${seconds}`;
    });

    _defineProperty(this, "parseData", data => {
      let parsedData = data instanceof Error ? {
        message: data.message,
        stack: data.stack
      } : data; // when error is AxiosError, it extends from Error

      if (data.isAxiosError) {
        const {
          config
        } = data;
        parsedData = { ...parsedData,
          config: {
            url: config.url,
            method: config.method,
            data: config.data,
            params: config.params
          }
        };
      }

      if (typeof parsedData === 'object') parsedData.toString = () => JSON.stringify(parsedData);
      return parsedData;
    });

    this.PLAIN_LOGS_PATH = _path.default.join(_constants.WAZUH_DATA_LOGS_DIRECTORY_PATH, plainLogsFile);
    this.RAW_LOGS_PATH = _path.default.join(_constants.WAZUH_DATA_LOGS_DIRECTORY_PATH, rawLogsFile);
    this.PLAIN_LOGS_FILE_NAME = plainLogsFile;
    this.RAW_LOGS_FILE_NAME = rawLogsFile;
  }
  /**
   * Initialize loggers, plain and raw logger
   */


  /**
   * Main function to add a new log
   * @param {*} location File where the log is being thrown
   * @param {*} data Message or object to log
   * @param {*} level Optional, default is 'error'
   */
  async log(location, data, level) {
    const parsedData = this.parseData(data);
    return this.initDirectory().then(() => {
      if (this.allowed) {
        this.checkFiles();
        const plainLogData = {
          level: level || 'error',
          message: `${this.yyyymmdd()}: ${location || 'Unknown origin'}: ${parsedData.toString() || 'An error occurred'}`
        };
        this.wazuhPlainLogger.log(plainLogData);
        const logData = {
          date: new Date(),
          level: level || 'error',
          location: location || 'Unknown origin',
          data: parsedData || 'An error occurred'
        };

        if (typeof data == 'string') {
          logData.message = parsedData;
          delete logData.data;
        }

        this.wazuhLogger.log(logData);
      }
    }).catch(error => {
      console.error(`Cannot create the logs directory due to:\n${error.message || error}`);
      throw error;
    });
  }

}

exports.BaseLogger = BaseLogger;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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