"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.maximumAuthenticationAttemptsExceeded = exports.authenticationSuccess = exports.bruteForceTryingAccessSystem = exports.nonExistentUser = exports.passwordCheckFailed = exports.userLoginFailed = exports.windowsInvalidLoginPassword = exports.multipleAuthenticationFailures = exports.invalidLoginUser = exports.invalidLoginPassword = void 0;

/*
 * Wazuh app - Authentication sample alerts
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
const invalidLoginPassword = {
  decoder: {
    parent: 'sshd',
    name: 'sshd'
  },
  full_log: '{predecoder.timestamp} {predecoder.hostname} sshd[5330]: Failed password for {data.srcuser} from {data.srcip} port {data.srcport} ssh2',
  location: '/var/log/auth.log',
  predecoder: {
    program_name: 'sshd',
    timestamp: 'Apr 17 00:17:52',
    hostname: 'ip-10-0-1-50'
  },
  rule: {
    description: 'sshd: authentication failed.',
    groups: ['syslog', 'sshd', 'invalid_login', 'authentication_failed'],
    id: 5716,
    level: 5,
    mail: false,
    pci_dss: ['10.2.4', '10.2.5'],
    gpg13: ['7.1'],
    gdpr: ['IV_35.7.d', 'IV_32.2'],
    hipaa: ['164.312.b'],
    nist_800_53: ['AU.14', 'AC.7']
  }
};
exports.invalidLoginPassword = invalidLoginPassword;
const invalidLoginUser = {
  decoder: {
    parent: 'sshd',
    name: 'sshd'
  },
  full_log: '{predecoder.timestamp} {predecoder.hostname} sshd[10022]: Invalid user {data.srcuser} from {data.srcuser} from {data.srcip} port {data.srcport} ssh2',
  location: '/var/log/secure',
  predecoder: {
    program_name: 'sshd',
    timestamp: 'Apr 17 00:17:52',
    hostname: 'ip-10-0-1-50'
  },
  rule: {
    description: 'sshd: Attempt to login using a non-existent user',
    groups: ['syslog', 'sshd', 'invalid_login', 'authentication_failed'],
    id: 5710,
    level: 5,
    pci_dss: ['10.2.4', '10.2.5', '10.6.1'],
    gpg13: ['7.1'],
    gdpr: ['IV_35.7.d', 'IV_32.2'],
    hipaa: ['164.312.b'],
    nist_800_53: ['AU.14', 'AC.7', 'AU.6']
  }
};
exports.invalidLoginUser = invalidLoginUser;
const multipleAuthenticationFailures = {
  decoder: {
    parent: 'sshd',
    name: 'sshd'
  },
  full_log: `{predecoder.timestamp} {predecoder.hostname} sshd[5413]: Failed password for invalid user {data.srcuser} from {data.srcip} port {data.srcport} ssh2`,
  location: '/var/log/secure',
  rule: {
    description: 'sshd: Multiple authentication failures.',
    id: 5720,
    level: 10,
    frequency: 8,
    groups: ['syslog', 'sshd', 'authentication_failures'],
    pci_dss: ['10.2.4', '10.2.5', '11.4'],
    gpg13: ['7.1'],
    gdpr: ['IV_35.7.d', 'IV_32.2'],
    hipaa: ['164.312.b'],
    nist_800_53: ['AU.14', 'AC.7', 'SI.4']
  },
  predecoder: {
    program_name: 'sshd',
    timestamp: 'Apr 17 00:17:52',
    hostname: 'ip-10-0-1-50'
  }
};
exports.multipleAuthenticationFailures = multipleAuthenticationFailures;
const windowsInvalidLoginPassword = {
  full_log: `{predecoder.timestamp} {predecoder.hostname} sshd[5413]: Failed password for invalid user {data.srcuser} from {data.srcip} port {data.srcport} ssh2`,
  data_win: {
    eventdata: {
      authenticationPackageName: 'NTLM',
      failureReason: '%%2313',
      keyLength: 0,
      logonProcessName: 'NtLmSsp',
      logonType: '3',
      processId: '0x0',
      status: '0xc000006d',
      subStatus: '0xc0000064',
      subjectLogonId: '0x0',
      subjectUserSid: 'S-1-0-0',
      targetUserName: 'DIRECTION'
    },
    system: {
      channel: 'Security',
      keywords: '0x8010000000000000',
      level: '0',
      message: '',
      opcode: '0',
      providerGuid: '{54849625-5478-4994-a5ba-3e3b0328c30d}',
      providerName: 'Microsoft-Windows-Security-Auditing',
      severityValue: 'AUDIT_FAILURE',
      version: '0'
    }
  },
  decoder: {
    parent: 'sshd',
    name: 'windows_eventchannel'
  },
  location: 'EventChannel',
  rule: {
    description: 'Logon Failure - Unknown user or bad password',
    groups: ['windows', 'windows_security', 'win_authentication_failed'],
    id: 60122,
    level: 5,
    pci_dss: ['10.2.4', '10.2.5'],
    gpg13: ['7.1'],
    gdpr: ['IV_35.7.d', 'IV_32.2'],
    hipaa: ['164.312.b'],
    nist_800_53: ['AU.1', 'AC.7']
  }
};
exports.windowsInvalidLoginPassword = windowsInvalidLoginPassword;
const userLoginFailed = {
  rule: {
    id: 5503,
    level: 5,
    description: 'PAM: User login failed.',
    mail: false,
    groups: ['pam', 'syslog', 'authentication_failed'],
    pci_dss: ['10.2.4', '10.2.5'],
    gpg13: ['7.8'],
    gdpr: ['IV_35.7.d', 'IV_32.2'],
    hipaa: ['164.312.b'],
    nist_800_53: ['AU.14', 'AC.7']
  },
  predecoder: {
    program_name: 'sshd',
    timestamp: 'Apr 17 00:04:40',
    hostname: 'ip-10-0-1-178'
  },
  decoder: {
    name: 'pam'
  },
  location: '/var/log/secure',
  full_log: '{predecoder.timestamp} {predecoder.hostname} sshd[11294]: pam_unix(sshd:auth): authentication failure; logname= uid={data.uid} euid={data.euid} tty={data.tty} ruser= rhost={data.srcip}  user={data.dstuser}'
};
exports.userLoginFailed = userLoginFailed;
const passwordCheckFailed = {
  rule: {
    level: 5,
    description: 'unix_chkpwd: Password check failed.',
    id: '5557',
    mail: false,
    groups: ['pam', 'syslog', 'authentication_failed'],
    pci_dss: ['10.2.4', '10.2.5'],
    gpg13: ['4.3'],
    gdpr: ['IV_35.7.d', 'IV_32.2'],
    hipaa: ['164.312.b'],
    nist_800_53: ['AU.14', 'AC.7']
  },
  predecoder: {
    program_name: 'unix_chkpwd',
    timestamp: 'Apr 17 00:07:04',
    hostname: 'ip-10-0-1-132'
  },
  decoder: {
    name: 'unix_chkpwd'
  },
  data: {
    srcuser: 'root'
  },
  location: '/var/log/secure',
  full_log: '{predecoder.timestamp} {predecoder.hostname} {decoder.name}[29593]: password check failed for user ({data.srcuser})'
};
exports.passwordCheckFailed = passwordCheckFailed;
const nonExistentUser = {
  rule: {
    mail: false,
    level: 5,
    pci_dss: ['10.2.4', '10.2.5', '10.6.1'],
    hipaa: ['164.312.b'],
    description: 'sshd: Attempt to login using a non-existent user',
    groups: ['syslog', 'sshd', 'invalid_login', 'authentication_failed'],
    id: '5710',
    nist_800_53: ['AU.14', 'AC.7', 'AU.6'],
    gpg13: ['7.1'],
    gdpr: ['IV_35.7.d', 'IV_32.2']
  },
  full_log: '{predecoder.timestamp} {predecoder.hostname} sshd[15724]: Invalid user {data.srcuser} from {data.srcip} port {data.srcport}',
  location: '/var/log/secure'
};
exports.nonExistentUser = nonExistentUser;
const bruteForceTryingAccessSystem = {
  rule: {
    mail: false,
    level: 10,
    pci_dss: ['11.4', '10.2.4', '10.2.5'],
    hipaa: ['164.312.b'],
    description: 'sshd: brute force trying to get access to the system.',
    groups: ['syslog', 'sshd', 'authentication_failures'],
    mitre: {
      tactic: ['Credential Access', 'Lateral Movement'],
      technique: ['Brute Force', 'Remove Services'],
      id: ['T1110', 'T1021']
    },
    id: '5712',
    nist_800_53: ['SI.4', 'AU.14', 'AC.7'],
    frequency: 8,
    gdpr: ['IV_35.7.d', 'IV_32.2']
  },
  full_log: '{predecoder.timestamp} {predecoder.hostname} sshd[15722]: Invalid user {data.srcuser} from {data.srcip} port {data.srcport}',
  location: '/var/log/secure'
};
exports.bruteForceTryingAccessSystem = bruteForceTryingAccessSystem;
const authenticationSuccess = {
  data: {
    srcip: '84.122.71.89',
    dstuser: 'ec2-user'
  },
  full_log: '{predecoder.timestamp} {predecoder.hostname} sshd[12727]: Accepted publickey for {data.dstuser} from {data.srcip} port {data.srcport} ssh2: RSA SHA256:ET29+nbiHqrKs1gUewWTFRCHWdO/vMoRQXPESWn8ZG4',
  input: {
    type: 'log'
  },
  location: '/var/log/secure',
  rule: {
    mail: false,
    level: 3,
    pci_dss: ['10.2.5'],
    hipaa: ['164.312.b'],
    description: 'sshd: authentication success.',
    groups: ['syslog', 'sshd', 'authentication_success'],
    id: '5715',
    nist_800_53: ['AU.14', 'AC.7'],
    gpg13: ['7.1', '7.2'],
    gdpr: ['IV_32.2']
  }
};
exports.authenticationSuccess = authenticationSuccess;
const maximumAuthenticationAttemptsExceeded = {
  rule: {
    mail: false,
    level: 8,
    description: 'Maximum authentication attempts exceeded.',
    groups: ['syslog', 'sshd', 'authentication_failed'],
    mitre: {
      tactic: ['Credential Access', 'Lateral Movement'],
      technique: ['Brute Force', 'Remove Services'],
      id: ['T1110', 'T1021']
    },
    id: '5758',
    gpg13: ['7.1']
  },
  location: '/var/log/secure',
  full_log: '{predecoder.timestamp} {predecoder.hostname} sshd[19767]: error: maximum authentication attempts exceeded for {data.dstuser} from {data.srcip} port {data.srcport} ssh2 [preauth]'
};
exports.maximumAuthenticationAttemptsExceeded = maximumAuthenticationAttemptsExceeded;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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