"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.initialWazuhConfig = void 0;

var _constants = require("../../common/constants");

var _web_documentation = require("../../common/services/web_documentation");

var _configEquivalences = require("../../common/config-equivalences");

/*
 * Wazuh app - App configuration file
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */

/**
 * Given a string, this function builds a multine string, each line about 70
 * characters long, splitted at the closest whitespace character to that lentgh.
 *
 * This function is used to transform the settings description stored in the
 * configEquivalences map into a multiline string to be used as the setting
 * documentation.
 *
 * The # character is also appended to the beginning of each line.
 *
 * @param text
 * @returns multine string
 */
function splitDescription(text = '') {
  const lines = text.match(/.{1,80}(?=\s|$)/g) || [];
  return lines.map(z => '# ' + z.trim()).join('\n');
}

const initialWazuhConfig = `---
#
# Wazuh app - App configuration file
# Copyright (C) 2015-2022 Wazuh, Inc.
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Find more information about this on the LICENSE file.
#
# ======================== Wazuh app configuration file ========================
#
# Please check the documentation for more information about configuration options:
# ${(0, _web_documentation.webDocumentationLink)('user-manual/wazuh-dashboard/config-file.html')}
#
# Also, you can check our repository:
# https://github.com/wazuh/wazuh-kibana-app
#
# ---------------------------- Unauthorized roles ------------------------------
#
# Disable Wazuh for the Elasticsearch / OpenSearch roles defined here.
# disabled_roles:
#   - wazuh_disabled
#
# ------------------------------- Index patterns -------------------------------
#
${splitDescription(_configEquivalences.configEquivalences.pattern)}
# pattern: ${_constants.WAZUH_ALERTS_PATTERN}
#
# ----------------------------------- Checks -----------------------------------
#
# Define which checks will be executed by the App's HealthCheck.
# Allowed values are: true, false
#
${splitDescription(_configEquivalences.configEquivalences['checks.pattern'])}
# checks.pattern: ${_constants.WAZUH_DEFAULT_APP_CONFIG['checks.pattern']}
#
${splitDescription(_configEquivalences.configEquivalences['checks.template'])}
# checks.template: ${_constants.WAZUH_DEFAULT_APP_CONFIG['checks.template']}
#
${splitDescription(_configEquivalences.configEquivalences['checks.api'])}
# checks.api: ${_constants.WAZUH_DEFAULT_APP_CONFIG['checks.api']}
#
${splitDescription(_configEquivalences.configEquivalences['checks.setup'])}
# checks.setup: ${_constants.WAZUH_DEFAULT_APP_CONFIG['checks.setup']}
#
${splitDescription(_configEquivalences.configEquivalences['checks.fields'])}
# checks.fields: ${_constants.WAZUH_DEFAULT_APP_CONFIG['checks.fields']}
#
${splitDescription(_configEquivalences.configEquivalences['checks.metaFields'])}
# checks.metaFields: ${_constants.WAZUH_DEFAULT_APP_CONFIG['checks.metaFields']}
#
${splitDescription(_configEquivalences.configEquivalences['checks.timeFilter'])}
# checks.timeFilter: ${_constants.WAZUH_DEFAULT_APP_CONFIG['checks.timeFilter']}
#
${splitDescription(_configEquivalences.configEquivalences['checks.maxBuckets'])}
# checks.maxBuckets: ${_constants.WAZUH_DEFAULT_APP_CONFIG['checks.maxBuckets']}
#
# --------------------------------- Extensions ---------------------------------
#
# Define the initial state of the extensions (enabled / disabled) for recently
# added hosts. The extensions can be enabled or disabled anytime using the UI.
# Allowed values are: true, false
#
${splitDescription(_configEquivalences.configEquivalences['extensions.pci'])}
# extensions.pci: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.pci']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.gdpr'])}
# extensions.gdpr: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.gdpr']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.hipaa'])}
# extensions.hipaa: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.hipaa']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.nist'])}
# extensions.nist: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.nist']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.tsc'])}
# extensions.tsc: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.tsc']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.audit'])}
# extensions.audit: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.audit']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.oscap'])}
# extensions.oscap: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.oscap']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.ciscat'])}
# extensions.ciscat: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.ciscat']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.aws'])}
# extensions.aws: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.aws']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.gcp'])}
# extensions.gcp: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.gcp']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.virustotal'])}
# extensions.virustotal: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.virustotal']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.osquery'])}
# extensions.osquery: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.osquery']}
#
${splitDescription(_configEquivalences.configEquivalences['extensions.docker'])}
# extensions.docker: ${_constants.WAZUH_DEFAULT_APP_CONFIG['extensions.docker']}
#
# ------------------------------- Timeout --------------------------------------
#
${splitDescription(_configEquivalences.configEquivalences.timeout)}
# timeout: ${_constants.WAZUH_DEFAULT_APP_CONFIG.timeout}
#
# --------------------------- Index pattern selector ---------------------------
#
${splitDescription(_configEquivalences.configEquivalences['ip.selector'])}
# ip.selector: ${_constants.WAZUH_DEFAULT_APP_CONFIG['ip.selector']}
#
${splitDescription(_configEquivalences.configEquivalences['ip.ignore'])}
# ip.ignore: ${_constants.WAZUH_DEFAULT_APP_CONFIG['ip.ignore']}
#
# ------------------------------ Monitoring ------------------------------------
#
${splitDescription(_configEquivalences.configEquivalences['wazuh.monitoring.enabled'])}
# wazuh.monitoring.enabled: ${_constants.WAZUH_MONITORING_DEFAULT_ENABLED}
#
${splitDescription(_configEquivalences.configEquivalences['wazuh.monitoring.frequency'])}
# wazuh.monitoring.frequency: ${_constants.WAZUH_MONITORING_DEFAULT_FREQUENCY}
#
${splitDescription(_configEquivalences.configEquivalences['wazuh.monitoring.shards'])}
# wazuh.monitoring.shards: ${_constants.WAZUH_MONITORING_DEFAULT_INDICES_SHARDS}
#
${splitDescription(_configEquivalences.configEquivalences['wazuh.monitoring.replicas'])}
# wazuh.monitoring.replicas: ${_constants.WAZUH_MONITORING_DEFAULT_INDICES_REPLICAS}
#
${splitDescription(_configEquivalences.configEquivalences['wazuh.monitoring.creation'])}
# Allowed values are: h (hourly), d (daily), w (weekly), m (monthly)
# wazuh.monitoring.creation: ${_constants.WAZUH_MONITORING_DEFAULT_CREATION}
#
${splitDescription(_configEquivalences.configEquivalences['wazuh.monitoring.pattern'])}
# wazuh.monitoring.pattern: ${_constants.WAZUH_MONITORING_PATTERN}
#
# --------------------------------- Sample data --------------------------------
#
${splitDescription(_configEquivalences.configEquivalences['alerts.sample.prefix'])}
# alerts.sample.prefix: ${_constants.WAZUH_SAMPLE_ALERT_PREFIX}
#
# ------------------------------ Background tasks ------------------------------
#
${splitDescription(_configEquivalences.configEquivalences['cron.prefix'])}
# cron.prefix: ${_constants.WAZUH_STATISTICS_DEFAULT_PREFIX}
#
# ------------------------------ Wazuh Statistics ------------------------------
#
${splitDescription(_configEquivalences.configEquivalences['cron.statistics.status'])}
# cron.statistics.status: ${_constants.WAZUH_STATISTICS_DEFAULT_STATUS}
#
${splitDescription(_configEquivalences.configEquivalences['cron.statistics.apis'])}
# cron.statistics.apis: ${_constants.WAZUH_DEFAULT_APP_CONFIG['cron.statistics.apis']}
#
${splitDescription(_configEquivalences.configEquivalences['cron.statistics.interval'])}
# cron.statistics.interval: ${_constants.WAZUH_STATISTICS_DEFAULT_CRON_FREQ}
#
${splitDescription(_configEquivalences.configEquivalences['cron.statistics.index.name'])}
# cron.statistics.index.name: ${_constants.WAZUH_STATISTICS_DEFAULT_NAME}
#
${splitDescription(_configEquivalences.configEquivalences['cron.statistics.index.creation'])}
# cron.statistics.index.creation: ${_constants.WAZUH_STATISTICS_DEFAULT_CREATION}
#
${splitDescription(_configEquivalences.configEquivalences['cron.statistics.index.shards'])}
# cron.statistics.shards: ${_constants.WAZUH_STATISTICS_DEFAULT_INDICES_SHARDS}
#
${splitDescription(_configEquivalences.configEquivalences['cron.statistics.index.replicas'])}
# cron.statistics.replicas: ${_constants.WAZUH_STATISTICS_DEFAULT_INDICES_REPLICAS}
#
# ------------------------------ Logo customization ----------------------------
#
${splitDescription(_configEquivalences.configEquivalences['customization.logo.app'])}
# customization.logo.app: ${_constants.WAZUH_DEFAULT_APP_CONFIG['customization.logo.app']}
#
${splitDescription(_configEquivalences.configEquivalences['customization.logo.sidebar'])}
# customization.logo.sidebar: ${_constants.WAZUH_DEFAULT_APP_CONFIG['customization.logo.sidebar']}
#
${splitDescription(_configEquivalences.configEquivalences['customization.logo.healthcheck'])}
# customization.logo.healthcheck: ${_constants.WAZUH_DEFAULT_APP_CONFIG['customization.logo.healthcheck']}
#
${splitDescription(_configEquivalences.configEquivalences['customization.logo.reports'])}
# customization.logo.reports: ${_constants.WAZUH_DEFAULT_APP_CONFIG['customization.logo.reports']}
#
# ---------------------------- Hide manager alerts -----------------------------
#
${splitDescription(_configEquivalences.configEquivalences.hideManagerAlerts)}
# hideManagerAlerts: ${_constants.WAZUH_DEFAULT_APP_CONFIG.hideManagerAlerts}
#
# ------------------------------- App logging level ----------------------------
#
${splitDescription(_configEquivalences.configEquivalences['logs.level'])}
# Allowed values are: info, debug
# logs.level: ${_constants.WAZUH_DEFAULT_APP_CONFIG['logs.level']}
#
# ------------------------------- Agent enrollment -----------------------------
#
${splitDescription(_configEquivalences.configEquivalences['enrollment.dns'])}
# enrollment.dns: ${_constants.WAZUH_DEFAULT_APP_CONFIG['enrollment.dns']}
#
${splitDescription(_configEquivalences.configEquivalences['enrollment.password'])}
# enrollment.password: ${_constants.WAZUH_DEFAULT_APP_CONFIG['enrollment.password']}
#
#-------------------------------- Wazuh hosts ----------------------------------
#
# The following configuration is the default structure to define a host.
#
# hosts:
#   # Host ID / name,
#   - env-1:
#       # Host URL
#       url: https://env-1.example
#       # Host / API port
#       port: 55000
#       # Host / API username
#       username: wazuh-wui
#       # Host / API password
#       password: wazuh-wui
#       # Use RBAC or not. If set to true, the username must be "wazuh-wui".
#       run_as: true
#   - env-2:
#       url: https://env-2.example
#       port: 55000
#       username: wazuh-wui
#       password: wazuh-wui
#       run_as: true

hosts:
  - default:
      url: https://localhost
      port: 55000
      username: wazuh-wui
      password: wazuh-wui
      run_as: false
`;
exports.initialWazuhConfig = initialWazuhConfig;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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