"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.lastTenModifiedFiles = exports.lastTenDeletedFiles = exports.top3Rules = exports.top3agents = void 0;

var _baseQuery = require("./base-query");

var _constants = require("../../../common/constants");

/*
 * Wazuh app - Specific methods to fetch Wazuh syscheck data from Elasticsearch
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */

/**
 * Returns top 3 dangerous agents
 * @param {*} context Endpoint context
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Array<String>}
 */
const top3agents = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'agent.id',
          size: 3,
          order: {
            _count: 'desc'
          }
        }
      }
    });
    base.query.bool.must.push({
      range: {
        'rule.level': {
          gte: 7,
          lt: 16
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => item.key);
  } catch (error) {
    return Promise.reject(error);
  }
};
/**
 * Returns top 3 rules
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Array<String>}
 */


exports.top3agents = top3agents;

const top3Rules = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'rule.description',
          size: 3,
          order: {
            _count: 'desc'
          }
        },
        aggs: {
          '3': {
            terms: {
              field: 'rule.id',
              size: 1,
              order: {
                _count: 'desc'
              }
            }
          }
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.reduce((accum, bucket) => {
      if (!bucket || !bucket['3'] || !bucket['3'].buckets || !bucket['3'].buckets[0] || !bucket['3'].buckets[0].key || !bucket.key) {
        return accum;
      }

      ;
      accum.push({
        ruleID: bucket['3'].buckets[0].key,
        ruleDescription: bucket.key
      });
      return accum;
    }, []);
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.top3Rules = top3Rules;

const lastTenDeletedFiles = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'syscheck.path',
          size: 10,
          order: {
            '1': 'desc'
          }
        },
        aggs: {
          '1': {
            max: {
              field: 'timestamp'
            }
          }
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'syscheck.event': {
          query: 'deleted'
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => ({
      date: item['1'].value_as_string,
      path: item.key
    })).sort((a, b) => a.date > b.date ? -1 : a.date < b.date ? 1 : 0);
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.lastTenDeletedFiles = lastTenDeletedFiles;

const lastTenModifiedFiles = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'syscheck.path',
          size: 10,
          order: {
            '1': 'desc'
          }
        },
        aggs: {
          '1': {
            max: {
              field: 'timestamp'
            }
          }
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'syscheck.event': {
          query: 'modified'
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => ({
      date: item['1'].value_as_string,
      path: item.key
    })).sort((a, b) => a.date > b.date ? -1 : a.date < b.date ? 1 : 0);
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.lastTenModifiedFiles = lastTenModifiedFiles;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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