"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRulesByRequirement = exports.topTSCRequirements = void 0;

var _baseQuery = require("./base-query");

var _constants = require("../../../common/constants");

/*
 * Wazuh app - Specific methods to fetch Wazuh TSC data from Elasticsearch
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */

/**
 * Returns top 5 TSC requirements
 * @param {Number} context Endpoint context
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Array<String>}
 */
const topTSCRequirements = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  if (filters.includes('rule.tsc: exists')) {
    filters = filters.replace('AND rule.tsc: exists', '');
  }

  ;

  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'rule.tsc',
          size: 5,
          order: {
            _count: 'desc'
          }
        }
      }
    });
    base.query.bool.must.push({
      exists: {
        field: 'rule.tsc'
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => item.key).sort((a, b) => {
      const a_split = a.split('.');
      const b_split = b.split('.');
      if (parseInt(a_split[0]) > parseInt(b_split[0])) return 1;else if (parseInt(a_split[0]) < parseInt(b_split[0])) return -1;else {
        if (parseInt(a_split[1]) > parseInt(b_split[1])) return 1;else if (parseInt(a_split[1]) < parseInt(b_split[1])) return -1;else {
          if (parseInt(a_split[2]) > parseInt(b_split[2])) return 1;else if (parseInt(a_split[2]) < parseInt(b_split[2])) return -1;
        }
      }
    });
  } catch (error) {
    return Promise.reject(error);
  }
};
/**
 * Returns top 3 rules for specific TSC requirement
 * @param {*} context Endpoint context
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} requirement TSCrequirement. E.g: 'CC7.2'
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Array<String>}
 */


exports.topTSCRequirements = topTSCRequirements;

const getRulesByRequirement = async (context, gte, lte, filters, requirement, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  if (filters.includes('rule.tsc: exists')) {
    filters = filters.replace('AND rule.tsc: exists', '');
  }

  ;

  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'rule.description',
          size: 3,
          order: {
            _count: 'desc'
          }
        },
        aggs: {
          '3': {
            terms: {
              field: 'rule.id',
              size: 1,
              order: {
                _count: 'desc'
              }
            }
          }
        }
      }
    });
    base.query.bool.must[0].query_string.query = base.query.bool.must[0].query_string.query + ' AND rule.tsc: "' + requirement + '"';
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.reduce((accum, bucket) => {
      if (!bucket || !bucket['3'] || !bucket['3'].buckets || !bucket['3'].buckets[0] || !bucket['3'].buckets[0].key || !bucket.key) {
        return accum;
      }

      ;
      accum.push({
        ruleID: bucket['3'].buckets[0].key,
        ruleDescription: bucket.key
      });
      return accum;
    }, []);
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.getRulesByRequirement = getRulesByRequirement;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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