"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.topPackagesWithCVE = exports.topPackages = exports.uniqueSeverityCount = exports.topCVECount = exports.topAgentCount = void 0;

var _baseQuery = require("./base-query");

var _constants = require("../../../common/constants");

/*
 * Wazuh app - Specific methods to fetch Wazuh vulnerability data from Elasticsearch
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */

/**
 * Returns top 3 agents for specific severity
 * @param {*} context Endpoint context
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} severity Low, Medium, High, Critical
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Array<Number>}
 */
const topAgentCount = async (context, gte, lte, severity, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'agent.id',
          size: 3,
          order: {
            _count: 'desc'
          }
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'data.vulnerability.severity': {
          query: severity
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => item.key);
  } catch (error) {
    return Promise.reject(error);
  }
};
/**
 * Returns top 3 CVE
 * @param {Number} context Endpoint context
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Array<Number>}
 */


exports.topAgentCount = topAgentCount;

const topCVECount = async (context, gte, lte, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'data.vulnerability.cve',
          size: 3,
          order: {
            _count: 'desc'
          }
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => item.key);
  } catch (error) {
    return Promise.reject(error);
  }
};
/**
 * Returns unique count of vulnerability alerts using specific severity.
 * @param {*} context Endpoint context
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} severity Low, Medium, High, Critical
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Number}
 */


exports.topCVECount = topCVECount;

const uniqueSeverityCount = async (context, gte, lte, severity, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '1': {
        cardinality: {
          field: 'agent.id'
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'data.vulnerability.severity': {
          query: severity
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    return response.body && response.body.aggregations && response.body.aggregations['1'] && response.body.aggregations['1'].value ? response.body.aggregations['1'].value : 0;
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.uniqueSeverityCount = uniqueSeverityCount;

const topPackages = async (context, gte, lte, severity, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'data.vulnerability.package.name',
          size: 20,
          order: {
            _count: 'desc'
          }
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'data.vulnerability.severity': {
          query: severity
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => ({
      package: item.key,
      severity: severity
    }));
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.topPackages = topPackages;

const topPackagesWithCVE = async (context, gte, lte, severity, filters, pattern = _constants.WAZUH_ALERTS_PATTERN) => {
  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'data.vulnerability.package.name',
          size: 3,
          order: {
            _count: 'desc'
          }
        },
        aggs: {
          '3': {
            terms: {
              field: 'data.vulnerability.reference',
              size: 10,
              order: {
                _count: 'desc'
              }
            }
          }
        }
      }
    });
    base.query.bool.must.push({
      match_phrase: {
        'data.vulnerability.severity': {
          query: severity
        }
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => ({
      package: item.key,
      references: item['3'].buckets.map(ref => ref.key)
    }));
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.topPackagesWithCVE = topPackagesWithCVE;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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