"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhPlugin = void 0;

var _securityFactory = require("./lib/security-factory");

var _routes = require("./routes");

var _start = require("./start");

var _cookie = require("./lib/cookie");

var ApiInterceptor = _interopRequireWildcard(require("./lib/api-interceptor"));

var _operators = require("rxjs/operators");

function _getRequireWildcardCache() { if (typeof WeakMap !== "function") return null; var cache = new WeakMap(); _getRequireWildcardCache = function () { return cache; }; return cache; }

function _interopRequireWildcard(obj) { if (obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class WazuhPlugin {
  constructor(initializerContext) {
    this.initializerContext = initializerContext;

    _defineProperty(this, "logger", void 0);

    this.logger = initializerContext.logger.get();
  }

  async setup(core, plugins) {
    this.logger.debug('Wazuh-wui: Setup');
    const serverInfo = core.http.getServerInfo();
    let wazuhSecurity;
    core.http.registerRouteHandlerContext('wazuh', async (context, request) => {
      !wazuhSecurity && (wazuhSecurity = await (0, _securityFactory.SecurityObj)(plugins, context));
      return {
        logger: this.logger,
        server: {
          info: serverInfo
        },
        plugins,
        security: wazuhSecurity,
        api: {
          client: {
            asInternalUser: {
              authenticate: async apiHostID => await ApiInterceptor.authenticate(apiHostID),
              request: async (method, path, data, options) => await ApiInterceptor.requestAsInternalUser(method, path, data, options)
            },
            asCurrentUser: {
              authenticate: async apiHostID => await ApiInterceptor.authenticate(apiHostID, (await wazuhSecurity.getCurrentUser(request, context)).authContext),
              request: async (method, path, data, options) => await ApiInterceptor.requestAsCurrentUser(method, path, data, { ...options,
                token: (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-token')
              })
            }
          }
        }
      };
    }); // Add custom headers to the responses

    core.http.registerOnPreResponse((request, response, toolkit) => {
      const additionalHeaders = {
        'x-frame-options': 'sameorigin'
      };
      return toolkit.next({
        headers: additionalHeaders
      });
    }); // Routes

    const router = core.http.createRouter();
    (0, _routes.setupRoutes)(router);
    return {};
  }

  async start(core) {
    const globalConfiguration = await this.initializerContext.config.legacy.globalConfig$.pipe((0, _operators.first)()).toPromise();
    const wazuhApiClient = {
      client: {
        asInternalUser: {
          authenticate: async apiHostID => await ApiInterceptor.authenticate(apiHostID),
          request: async (method, path, data, options) => await ApiInterceptor.requestAsInternalUser(method, path, data, options)
        }
      }
    };
    const contextServer = {
      config: globalConfiguration
    }; // Initialize

    (0, _start.jobInitializeRun)({
      core,
      wazuh: {
        logger: this.logger.get('initialize'),
        api: wazuhApiClient
      },
      server: contextServer
    }); // Migration tasks

    (0, _start.jobMigrationTasksRun)({
      core,
      wazuh: {
        logger: this.logger.get('migration-task'),
        api: wazuhApiClient
      },
      server: contextServer
    }); // Monitoring

    (0, _start.jobMonitoringRun)({
      core,
      wazuh: {
        logger: this.logger.get('monitoring'),
        api: wazuhApiClient
      },
      server: contextServer
    }); // Scheduler

    (0, _start.jobSchedulerRun)({
      core,
      wazuh: {
        logger: this.logger.get('cron-scheduler'),
        api: wazuhApiClient
      },
      server: contextServer
    }); // Queue

    (0, _start.jobQueueRun)({
      core,
      wazuh: {
        logger: this.logger.get('queue'),
        api: wazuhApiClient
      },
      server: contextServer
    });
    return {};
  }

  stop() {}

}

exports.WazuhPlugin = WazuhPlugin;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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