"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhReportingRoutes = WazuhReportingRoutes;

var _controllers = require("../controllers");

var _configSchema = require("@kbn/config-schema");

/*
 * Wazuh app - Module for Wazuh reporting routes
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
function WazuhReportingRoutes(router) {
  const ctrl = new _controllers.WazuhReportingCtrl();

  const agentIDValidation = _configSchema.schema.string({
    minLength: 3,
    validate: agentID => /^\d{3,}$/.test(agentID) ? undefined : 'must be 0-9 are allowed'
  });

  const groupIDValidation = _configSchema.schema.string({
    minLength: 1,
    validate: agentID => /^(?!^(\.{1,2}|all)$)[\w\.\-]+$/.test(agentID) ? undefined : 'must be A-z, 0-9, _, . are allowed. It must not be ., .. or all.'
  });

  const ReportFilenameValidation = _configSchema.schema.string({
    validate: agentID => /^[\w\-\.]+\.pdf$/.test(agentID) ? undefined : 'must be A-z, 0-9, _, ., and - are allowed. It must end with .pdf.'
  });

  const moduleIDValidation = _configSchema.schema.oneOf([_configSchema.schema.literal('general'), _configSchema.schema.literal('fim'), _configSchema.schema.literal('aws'), _configSchema.schema.literal('gcp'), _configSchema.schema.literal('pm'), _configSchema.schema.literal('audit'), _configSchema.schema.literal('sca'), _configSchema.schema.literal('office'), _configSchema.schema.literal('github'), _configSchema.schema.literal('ciscat'), _configSchema.schema.literal('vuls'), _configSchema.schema.literal('mitre'), _configSchema.schema.literal('virustotal'), _configSchema.schema.literal('docker'), _configSchema.schema.literal('osquery'), _configSchema.schema.literal('oscap'), _configSchema.schema.literal('pci'), _configSchema.schema.literal('hipaa'), _configSchema.schema.literal('nist'), _configSchema.schema.literal('gdpr'), _configSchema.schema.literal('tsc')]);

  router.post({
    path: '/reports/modules/{moduleID}',
    validate: {
      body: _configSchema.schema.object({
        array: _configSchema.schema.any(),
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        agents: _configSchema.schema.maybe(_configSchema.schema.oneOf([agentIDValidation, _configSchema.schema.boolean()])),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        searchBar: _configSchema.schema.maybe(_configSchema.schema.string()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        tab: _configSchema.schema.string(),
        tables: _configSchema.schema.maybe(_configSchema.schema.any()),
        time: _configSchema.schema.oneOf([_configSchema.schema.object({
          from: _configSchema.schema.string(),
          to: _configSchema.schema.string()
        }), _configSchema.schema.string()]),
        indexPatternTitle: _configSchema.schema.string(),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        moduleID: moduleIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsModules(context, request, response));
  router.post({
    path: '/reports/groups/{groupID}',
    validate: {
      body: _configSchema.schema.object({
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        groupID: groupIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsGroups(context, request, response));
  router.post({
    path: '/reports/agents/{agentID}',
    validate: {
      body: _configSchema.schema.object({
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.any(),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        agentID: agentIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsAgentsConfiguration(context, request, response));
  router.post({
    path: '/reports/agents/{agentID}/inventory',
    validate: {
      body: _configSchema.schema.object({
        array: _configSchema.schema.any(),
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        agents: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.boolean()])),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        searchBar: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.boolean()])),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        tab: _configSchema.schema.string(),
        tables: _configSchema.schema.maybe(_configSchema.schema.any()),
        time: _configSchema.schema.oneOf([_configSchema.schema.object({
          from: _configSchema.schema.string(),
          to: _configSchema.schema.string()
        }), _configSchema.schema.string()]),
        indexPatternTitle: _configSchema.schema.string(),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        agentID: agentIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsAgentsInventory(context, request, response)); // Fetch specific report

  router.get({
    path: '/reports/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: ReportFilenameValidation
      })
    }
  }, (context, request, response) => ctrl.getReportByName(context, request, response)); // Delete specific report

  router.delete({
    path: '/reports/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: ReportFilenameValidation
      })
    }
  }, (context, request, response) => ctrl.deleteReportByName(context, request, response)); // Fetch the reports list

  router.get({
    path: '/reports',
    validate: false
  }, (context, request, response) => ctrl.getReports(context, request, response));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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