"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SaveDocument = void 0;

var _getConfiguration = require("../../lib/get-configuration");

var _logger = require("../../lib/logger");

var _indexDate = require("../../lib/index-date");

var _constants = require("../../../common/constants");

var _tryCatchForIndexPermissionError = require("../tryCatchForIndexPermissionError");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class SaveDocument {
  constructor(context) {
    _defineProperty(this, "context", void 0);

    _defineProperty(this, "esClientInternalUser", void 0);

    _defineProperty(this, "logPath", 'cron-scheduler|SaveDocument');

    this.context = context;
    this.esClientInternalUser = context.core.elasticsearch.client.asInternalUser;
  }

  async save(doc, indexConfig) {
    const {
      name,
      creation,
      mapping,
      shards,
      replicas
    } = indexConfig;
    const index = this.addIndexPrefix(name);
    const indexCreation = `${index}-${(0, _indexDate.indexDate)(creation)}`;

    try {
      await this.checkIndexAndCreateIfNotExists(indexCreation, shards, replicas);
      const createDocumentObject = this.createDocument(doc, indexCreation, mapping);
      const response = await this.esClientInternalUser.bulk(createDocumentObject);
      (0, _logger.log)(this.logPath, `Response of create new document ${JSON.stringify(response)}`, 'debug'); // await this.checkIndexPatternAndCreateIfNotExists(index);
    } catch (error) {
      if (error.status === 403) throw {
        error: 403,
        message: `Authorization Exception in the index "${index}"`
      };
      if (error.status === 409) throw {
        error: 409,
        message: `Duplicate index-pattern: ${index}`
      };
      throw error;
    }
  }

  async checkIndexAndCreateIfNotExists(index, shards, replicas) {
    try {
      await (0, _tryCatchForIndexPermissionError.tryCatchForIndexPermissionError)(index)(async () => {
        const exists = await this.esClientInternalUser.indices.exists({
          index
        });
        (0, _logger.log)(this.logPath, `Index '${index}' exists? ${exists.body}`, 'debug');

        if (!exists.body) {
          const response = await this.esClientInternalUser.indices.create({
            index,
            body: {
              settings: {
                index: {
                  number_of_shards: shards !== null && shards !== void 0 ? shards : _constants.WAZUH_STATISTICS_DEFAULT_INDICES_SHARDS,
                  number_of_replicas: replicas !== null && replicas !== void 0 ? replicas : _constants.WAZUH_STATISTICS_DEFAULT_INDICES_REPLICAS
                }
              }
            }
          });
          (0, _logger.log)(this.logPath, `Status of create a new index: ${JSON.stringify(response)}`, 'debug');
        }
      })();
    } catch (error) {
      this.checkDuplicateIndexError(error);
    }
  }

  checkDuplicateIndexError(error) {
    const {
      type
    } = ((error || {}).body || {}).error || {};
    if (!['resource_already_exists_exception'].includes(type)) throw error;
  }

  createDocument(doc, index, mapping) {
    const createDocumentObject = {
      index,
      type: '_doc',
      body: doc.flatMap(item => [{
        index: {
          _index: index
        }
      }, { ...this.buildData(item, mapping),
        timestamp: new Date(Date.now()).toISOString()
      }])
    };
    (0, _logger.log)(this.logPath, `Document object: ${JSON.stringify(createDocumentObject)}`, 'debug');
    return createDocumentObject;
  }

  buildData(item, mapping) {
    const getItemArray = (array, index) => {
      return JSON.stringify(array[index || 0]);
    };

    const getValue = (key, item) => {
      const keys = key.split('.');

      if (keys.length === 1) {
        if (key.match(/\[.*\]/)) {
          return getItemArray(item[key.replace(/\[.*\]/, '')], key.match(/\[(.*)\]/)[1]);
        }

        return JSON.stringify(item[key]);
      }

      return getValue(keys.slice(1).join('.'), item[keys[0]]);
    };

    if (mapping) {
      let data;
      data = mapping.replace(/\${([a-z|A-Z|0-9|\.\-\_\[.*\]]+)}/gi, (...key) => getValue(key[1], item));
      return JSON.parse(data);
    }

    if (typeof item.data === 'object') {
      return item.data;
    }

    return {
      data: item.data
    };
  }

  addIndexPrefix(index) {
    const configFile = (0, _getConfiguration.getConfiguration)();
    const prefix = configFile['cron.prefix'] || 'wazuh';
    return `${prefix}-${index}`;
  }

}

exports.SaveDocument = SaveDocument;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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