"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobInitializeRun = jobInitializeRun;

var _logger = require("../../lib/logger");

var _package = _interopRequireDefault(require("../../../package.json"));

var _kibanaTemplate = require("../../integration-files/kibana-template");

var _getConfiguration = require("../../lib/get-configuration");

var _os = require("os");

var _fs = _interopRequireDefault(require("fs"));

var _constants = require("../../../common/constants");

var _filesystem = require("../../lib/filesystem");

var _lodash = _interopRequireDefault(require("lodash"));

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Wazuh app - Module for app initialization
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
function jobInitializeRun(context) {
  const PLUGIN_PLATFORM_INDEX = context.server.config.kibana.index;
  (0, _logger.log)('initialize', `${_constants.PLUGIN_PLATFORM_NAME} index: ${PLUGIN_PLATFORM_INDEX}`, 'info');
  (0, _logger.log)('initialize', `App revision: ${_package.default.revision}`, 'info');
  let configurationFile = {};
  let pattern = null; // Read config from package.json and wazuh.yml

  try {
    configurationFile = (0, _getConfiguration.getConfiguration)();
    pattern = configurationFile && typeof configurationFile.pattern !== 'undefined' ? configurationFile.pattern : _constants.WAZUH_ALERTS_PATTERN;
  } catch (error) {
    (0, _logger.log)('initialize', error.message || error);
    context.wazuh.logger.error('Something went wrong while reading the configuration.' + (error.message || error));
  }

  try {
    // RAM in MB
    const ram = Math.ceil((0, _os.totalmem)() / 1024 / 1024);
    (0, _logger.log)('initialize', `Total RAM: ${ram}MB`, 'info');
  } catch (error) {
    (0, _logger.log)('initialize', `Could not check total RAM due to: ${error.message || error}`);
  } // Save Wazuh App setup


  const saveConfiguration = async (hosts = {}) => {
    try {
      const commonDate = new Date().toISOString();
      const configuration = {
        name: 'Wazuh App',
        'app-version': _package.default.version,
        revision: _package.default.revision,
        installationDate: commonDate,
        lastRestart: commonDate,
        hosts
      };

      try {
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDataDirectoryIfNotExists)('config');
        (0, _logger.log)('initialize:saveConfiguration', `Saving configuration in registry file: ${JSON.stringify(configuration)}`, 'debug');
        await _fs.default.writeFileSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH, JSON.stringify(configuration), 'utf8');
        (0, _logger.log)('initialize:saveConfiguration', 'Wazuh configuration registry saved.', 'debug');
      } catch (error) {
        (0, _logger.log)('initialize:saveConfiguration', error.message || error);
        context.wazuh.logger.error('Could not create Wazuh configuration registry');
      }
    } catch (error) {
      (0, _logger.log)('initialize:saveConfiguration', error.message || error);
      context.wazuh.logger.error('Error creating wazuh-registry.json file.');
    }
  };
  /**
   * Checks if the .wazuh-registry.json file exists:
   * - yes: check the plugin version and revision match the values stored in the registry file.
   *  If not, then it migrates the data rebuilding the registry file.
   * - no: create the file with empty hosts
   */


  const checkWazuhRegistry = async () => {
    (0, _logger.log)('initialize:checkwazuhRegistry', 'Checking wazuh-registry.json file.', 'debug');

    if (!_fs.default.existsSync(_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH)) {
      throw new Error(`The data directory is missing in the ${_constants.PLUGIN_PLATFORM_NAME} root instalation. Create the directory in ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH} and give it the required permissions (sudo mkdir ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH};sudo chown -R ${_constants.PLUGIN_PLATFORM_INSTALLATION_USER}:${_constants.PLUGIN_PLATFORM_INSTALLATION_USER_GROUP} ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH}). After restart the ${_constants.PLUGIN_PLATFORM_NAME} service.`);
    }

    ;

    if (!_fs.default.existsSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH)) {
      (0, _logger.log)('initialize:checkwazuhRegistry', 'wazuh-registry.json file does not exist. Initializing configuration.', 'debug'); // Create the app registry file for the very first time

      await saveConfiguration();
    } else {
      // If this function fails, it throws an exception
      const source = JSON.parse(_fs.default.readFileSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH, 'utf8')); // Check if the stored revision differs from the package.json revision

      const isUpgradedApp = _package.default.revision !== source.revision || _package.default.version !== source['app-version']; // Rebuild the registry file if revision or version fields are differents

      if (isUpgradedApp) {
        (0, _logger.log)('initialize:checkwazuhRegistry', 'Wazuh app revision or version changed, regenerating wazuh-registry.json.', 'info'); // Rebuild the registry file `wazuh-registry.json`
        // Get the supported extensions for the installed plugin

        const supportedDefaultExtensionsConfiguration = Object.entries(_constants.WAZUH_DEFAULT_APP_CONFIG).filter(([setting]) => setting.startsWith('extensions.')).map(([setting, settingValue]) => {
          return [setting.split('.')[1], settingValue];
        }); // Get the supported extensions by ID

        const supportedDefaultExtensionsNames = supportedDefaultExtensionsConfiguration.map(([setting]) => setting); // Generate the hosts data, migrating the extensions.
        // Keep the supported and existent extensions for the installed plugin with the configurated value
        // Add the extensions with default values that didn't exist in the previous configuration
        // Remove the unsupported extensions for the installed plugin

        const registryHostsData = Object.entries(source.hosts).reduce((accum, [hostID, hostData]) => {
          accum[hostID] = hostData;

          if (accum[hostID].extensions) {
            // Migrate extensions to those supported by the installed plugin
            const defaultHostExtentionsConfiguration = Object.fromEntries(supportedDefaultExtensionsConfiguration); // Select of current configuration the extension IDs that are supported in the installed plugin

            const currentHostConfiguration = _lodash.default.pick(accum[hostID].extensions, supportedDefaultExtensionsNames); // Merge the default extensions configuration with the configuration stored in the registry file


            accum[hostID].extensions = _lodash.default.merge(defaultHostExtentionsConfiguration, currentHostConfiguration);
          }

          return accum;
        }, {}); // Rebuild the registry file with the migrated host data (extensions are migrated to these supported by the installed plugin).

        await saveConfiguration(registryHostsData);
        (0, _logger.log)('initialize:checkwazuhRegistry', 'Migrated the registry file.', 'info');
      }
    }
  }; // Init function. Check for wazuh-registry.json file exists.


  const init = async () => {
    await checkWazuhRegistry();
  };

  const createKibanaTemplate = () => {
    (0, _logger.log)('initialize:createKibanaTemplate', `Creating template for ${PLUGIN_PLATFORM_INDEX}`, 'debug');

    try {
      _kibanaTemplate.pluginPlatformTemplate.template = PLUGIN_PLATFORM_INDEX + '*';
    } catch (error) {
      (0, _logger.log)('initialize:createKibanaTemplate', error.message || error);
      context.wazuh.logger.error('Exception: ' + error.message || error);
    }

    return context.core.elasticsearch.client.asInternalUser.indices.putTemplate({
      name: _constants.WAZUH_PLUGIN_PLATFORM_TEMPLATE_NAME,
      order: 0,
      create: true,
      body: _kibanaTemplate.pluginPlatformTemplate
    });
  };

  const createEmptyKibanaIndex = async () => {
    try {
      (0, _logger.log)('initialize:createEmptyKibanaIndex', `Creating ${PLUGIN_PLATFORM_INDEX} index.`, 'info');
      await context.core.elasticsearch.client.asInternalUser.indices.create({
        index: PLUGIN_PLATFORM_INDEX
      });
      (0, _logger.log)('initialize:createEmptyKibanaIndex', `Successfully created ${PLUGIN_PLATFORM_INDEX} index.`, 'debug');
      await init();
    } catch (error) {
      return Promise.reject(new Error(`Error creating ${PLUGIN_PLATFORM_INDEX} index due to ${error.message || error}`));
    }
  };

  const fixKibanaTemplate = async () => {
    try {
      await createKibanaTemplate();
      (0, _logger.log)('initialize:fixKibanaTemplate', `Successfully created ${PLUGIN_PLATFORM_INDEX} template.`, 'debug');
      await createEmptyKibanaIndex();
    } catch (error) {
      return Promise.reject(new Error(`Error creating template for ${PLUGIN_PLATFORM_INDEX} due to ${error.message || error}`));
    }
  };

  const getTemplateByName = async () => {
    try {
      await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
        name: _constants.WAZUH_PLUGIN_PLATFORM_TEMPLATE_NAME
      });
      (0, _logger.log)('initialize:getTemplateByName', `No need to create the ${PLUGIN_PLATFORM_INDEX} template, already exists.`, 'debug');
      await createEmptyKibanaIndex();
    } catch (error) {
      (0, _logger.log)('initialize:getTemplateByName', error.message || error);
      return fixKibanaTemplate();
    }
  }; // Does Kibana index exist?


  const checkKibanaStatus = async () => {
    try {
      const response = await context.core.elasticsearch.client.asInternalUser.indices.exists({
        index: PLUGIN_PLATFORM_INDEX
      });

      if (response.body) {
        // It exists, initialize!
        await init();
      } else {
        // No Kibana index created...
        (0, _logger.log)('initialize:checkKibanaStatus', `Not found ${PLUGIN_PLATFORM_INDEX} index`, 'info');
        await getTemplateByName();
      }
    } catch (error) {
      (0, _logger.log)('initialize:checkKibanaStatus', error.message || error);
      context.wazuh.logger.error(error.message || error);
    }
  }; // Wait until Elasticsearch js is ready


  const checkStatus = async () => {
    try {
      // TODO: wait until elasticsearch is ready?
      // await server.plugins.elasticsearch.waitUntilReady();
      return await checkKibanaStatus();
    } catch (error) {
      (0, _logger.log)('initialize:checkStatus', 'Waiting for elasticsearch plugin to be ready...', 'debug');
      setTimeout(() => checkStatus(), 3000);
    }
  }; // Check Kibana index and if it is prepared, start the initialization of Wazuh App.


  return checkStatus();
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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