"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobMonitoringRun = jobMonitoringRun;

var _nodeCron = _interopRequireDefault(require("node-cron"));

var _logger = require("../../lib/logger");

var _monitoringTemplate = require("../../integration-files/monitoring-template");

var _getConfiguration = require("../../lib/get-configuration");

var _parseCron = require("../../lib/parse-cron");

var _indexDate = require("../../lib/index-date");

var _buildIndexSettings = require("../../lib/build-index-settings");

var _wazuhHosts = require("../../controllers/wazuh-hosts");

var _constants = require("../../../common/constants");

var _tryCatchForIndexPermissionError = require("../tryCatchForIndexPermissionError");

var _utils = require("../../../common/utils");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

/*
 * Wazuh app - Module for agent info fetching functions
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
const blueWazuh = '\u001b[34mwazuh\u001b[39m';
const monitoringErrorLogColors = [blueWazuh, 'monitoring', 'error'];
const wazuhHostController = new _wazuhHosts.WazuhHostsCtrl();
let MONITORING_ENABLED, MONITORING_FREQUENCY, MONITORING_CRON_FREQ, MONITORING_CREATION, MONITORING_INDEX_PATTERN, MONITORING_INDEX_PREFIX; // Utils functions

/**
 * Get the setting value from the configuration
 * @param setting
 * @param configuration
 * @param defaultValue
 */

function getAppConfigurationSetting(setting, configuration, defaultValue) {
  return typeof configuration[setting] !== 'undefined' ? configuration[setting] : defaultValue;
}

;
/**
 * Set the monitoring variables
 * @param context
 */

function initMonitoringConfiguration(context) {
  try {
    const appConfig = (0, _getConfiguration.getConfiguration)();
    MONITORING_ENABLED = appConfig && typeof appConfig['wazuh.monitoring.enabled'] !== 'undefined' ? appConfig['wazuh.monitoring.enabled'] && appConfig['wazuh.monitoring.enabled'] !== 'worker' : _constants.WAZUH_MONITORING_DEFAULT_ENABLED;
    MONITORING_FREQUENCY = getAppConfigurationSetting('wazuh.monitoring.frequency', appConfig, _constants.WAZUH_MONITORING_DEFAULT_FREQUENCY);
    MONITORING_CRON_FREQ = (0, _parseCron.parseCron)(MONITORING_FREQUENCY);
    MONITORING_CREATION = getAppConfigurationSetting('wazuh.monitoring.creation', appConfig, _constants.WAZUH_MONITORING_DEFAULT_CREATION);
    MONITORING_INDEX_PATTERN = getAppConfigurationSetting('wazuh.monitoring.pattern', appConfig, _constants.WAZUH_MONITORING_PATTERN);
    const lastCharIndexPattern = MONITORING_INDEX_PATTERN[MONITORING_INDEX_PATTERN.length - 1];

    if (lastCharIndexPattern !== '*') {
      MONITORING_INDEX_PATTERN += '*';
    }

    ;
    MONITORING_INDEX_PREFIX = MONITORING_INDEX_PATTERN.slice(0, MONITORING_INDEX_PATTERN.length - 1);
    (0, _logger.log)('monitoring:initMonitoringConfiguration', `wazuh.monitoring.enabled: ${MONITORING_ENABLED}`, 'debug');
    (0, _logger.log)('monitoring:initMonitoringConfiguration', `wazuh.monitoring.frequency: ${MONITORING_FREQUENCY} (${MONITORING_CRON_FREQ})`, 'debug');
    (0, _logger.log)('monitoring:initMonitoringConfiguration', `wazuh.monitoring.pattern: ${MONITORING_INDEX_PATTERN} (index prefix: ${MONITORING_INDEX_PREFIX})`, 'debug');
  } catch (error) {
    const errorMessage = error.message || error;
    (0, _logger.log)('monitoring:initMonitoringConfiguration', errorMessage);
    context.wazuh.logger.error(errorMessage);
  }
}

;
/**
 * Main. First execution when installing / loading App.
 * @param context
 */

async function init(context) {
  try {
    if (MONITORING_ENABLED) {
      await checkTemplate(context);
    }

    ;
  } catch (error) {
    const errorMessage = error.message || error;
    (0, _logger.log)('monitoring:init', error.message || error);
    context.wazuh.logger.error(errorMessage);
  }
}
/**
 * Verify wazuh-agent template
 */


async function checkTemplate(context) {
  try {
    (0, _logger.log)('monitoring:checkTemplate', 'Updating the monitoring template', 'debug');

    try {
      // Check if the template already exists
      const currentTemplate = await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
        name: _constants.WAZUH_MONITORING_TEMPLATE_NAME
      }); // Copy already created index patterns

      _monitoringTemplate.monitoringTemplate.index_patterns = currentTemplate.body[_constants.WAZUH_MONITORING_TEMPLATE_NAME].index_patterns;
    } catch (error) {
      // Init with the default index pattern
      _monitoringTemplate.monitoringTemplate.index_patterns = [_constants.WAZUH_MONITORING_PATTERN];
    } // Check if the user is using a custom pattern and add it to the template if it does


    if (!_monitoringTemplate.monitoringTemplate.index_patterns.includes(MONITORING_INDEX_PATTERN)) {
      _monitoringTemplate.monitoringTemplate.index_patterns.push(MONITORING_INDEX_PATTERN);
    }

    ; // Update the monitoring template

    await context.core.elasticsearch.client.asInternalUser.indices.putTemplate({
      name: _constants.WAZUH_MONITORING_TEMPLATE_NAME,
      body: _monitoringTemplate.monitoringTemplate
    });
    (0, _logger.log)('monitoring:checkTemplate', 'Updated the monitoring template', 'debug');
  } catch (error) {
    const errorMessage = `Something went wrong updating the monitoring template ${error.message || error}`;
    (0, _logger.log)('monitoring:checkTemplate', errorMessage);
    context.wazuh.logger.error(monitoringErrorLogColors, errorMessage);
    throw error;
  }
}
/**
 * Save agent status into elasticsearch, create index and/or insert document
 * @param {*} context
 * @param {*} data
 */


async function insertMonitoringDataElasticsearch(context, data) {
  const monitoringIndexName = MONITORING_INDEX_PREFIX + (0, _indexDate.indexDate)(MONITORING_CREATION);

  if (!MONITORING_ENABLED) {
    return;
  }

  ;

  try {
    await (0, _tryCatchForIndexPermissionError.tryCatchForIndexPermissionError)(monitoringIndexName)(async () => {
      const exists = await context.core.elasticsearch.client.asInternalUser.indices.exists({
        index: monitoringIndexName
      });

      if (!exists.body) {
        await createIndex(context, monitoringIndexName);
      }

      ; // Update the index configuration

      const appConfig = (0, _getConfiguration.getConfiguration)();
      const indexConfiguration = (0, _buildIndexSettings.buildIndexSettings)(appConfig, 'wazuh.monitoring', _constants.WAZUH_MONITORING_DEFAULT_INDICES_SHARDS); // To update the index settings with this client is required close the index, update the settings and open it
      // Number of shards is not dynamic so delete that setting if it's given

      delete indexConfiguration.settings.index.number_of_shards;
      await context.core.elasticsearch.client.asInternalUser.indices.putSettings({
        index: monitoringIndexName,
        body: indexConfiguration
      }); // Insert data to the monitoring index

      await insertDataToIndex(context, monitoringIndexName, data);
    })();
  } catch (error) {
    (0, _logger.log)('monitoring:insertMonitoringDataElasticsearch', error.message || error);
    context.wazuh.logger.error(error.message);
  }
}
/**
 * Inserting one document per agent into Elastic. Bulk.
 * @param {*} context Endpoint
 * @param {String} indexName The name for the index (e.g. daily: wazuh-monitoring-YYYY.MM.DD)
 * @param {*} data
 */


async function insertDataToIndex(context, indexName, data) {
  const {
    agents,
    apiHost
  } = data;

  try {
    if (agents.length > 0) {
      (0, _logger.log)('monitoring:insertDataToIndex', `Bulk data to index ${indexName} for ${agents.length} agents`, 'debug');
      const bodyBulk = agents.map(agent => {
        const agentInfo = { ...agent
        };
        agentInfo['timestamp'] = new Date(Date.now()).toISOString();
        agentInfo.host = agent.manager;
        agentInfo.cluster = {
          name: apiHost.clusterName ? apiHost.clusterName : 'disabled'
        };
        return `{ "index":  { "_index": "${indexName}" } }\n${JSON.stringify(agentInfo)}\n`;
      }).join('');
      await context.core.elasticsearch.client.asInternalUser.bulk({
        index: indexName,
        body: bodyBulk
      });
      (0, _logger.log)('monitoring:insertDataToIndex', `Bulk data to index ${indexName} for ${agents.length} agents completed`, 'debug');
    }
  } catch (error) {
    (0, _logger.log)('monitoring:insertDataToIndex', `Error inserting agent data into elasticsearch. Bulk request failed due to ${error.message || error}`);
  }
}
/**
 * Create the wazuh-monitoring index
 * @param {*} context context
 * @param {String} indexName The name for the index (e.g. daily: wazuh-monitoring-YYYY.MM.DD)
 */


async function createIndex(context, indexName) {
  try {
    if (!MONITORING_ENABLED) return;
    const appConfig = (0, _getConfiguration.getConfiguration)();
    const IndexConfiguration = {
      settings: {
        index: {
          number_of_shards: getAppConfigurationSetting('wazuh.monitoring.shards', appConfig, _constants.WAZUH_MONITORING_DEFAULT_INDICES_SHARDS),
          number_of_replicas: getAppConfigurationSetting('wazuh.monitoring.replicas', appConfig, _constants.WAZUH_MONITORING_DEFAULT_INDICES_REPLICAS)
        }
      }
    };
    await context.core.elasticsearch.client.asInternalUser.indices.create({
      index: indexName,
      body: IndexConfiguration
    });
    (0, _logger.log)('monitoring:createIndex', `Successfully created new index: ${indexName}`, 'debug');
  } catch (error) {
    const errorMessage = `Could not create ${indexName} index on elasticsearch due to ${error.message || error}`;
    (0, _logger.log)('monitoring:createIndex', errorMessage);
    context.wazuh.logger.error(errorMessage);
  }
}
/**
* Wait until Kibana server is ready
*/


async function checkPluginPlatformStatus(context) {
  try {
    (0, _logger.log)('monitoring:checkPluginPlatformStatus', 'Waiting for Kibana and Elasticsearch servers to be ready...', 'debug');
    await checkElasticsearchServer(context);
    await init(context);
    return;
  } catch (error) {
    (0, _logger.log)('monitoring:checkPluginPlatformStatus', error.mesage || error);

    try {
      await (0, _utils.delayAsPromise)(3000);
      await checkPluginPlatformStatus(context);
    } catch (error) {}

    ;
  }
}
/**
 * Check Elasticsearch Server status and Kibana index presence
 */


async function checkElasticsearchServer(context) {
  try {
    const data = await context.core.elasticsearch.client.asInternalUser.indices.exists({
      index: context.server.config.kibana.index
    });
    return data.body; // TODO: check if Elasticsearch can receive requests
    // if (data) {
    //   const pluginsData = await this.server.plugins.elasticsearch.waitUntilReady();
    //   return pluginsData;
    // }

    return Promise.reject(data);
  } catch (error) {
    (0, _logger.log)('monitoring:checkElasticsearchServer', error.message || error);
    return Promise.reject(error);
  }
}

const fakeResponseEndpoint = {
  ok: body => body,
  custom: body => body
};
/**
 * Get API configuration from elastic and callback to loadCredentials
 */

async function getHostsConfiguration() {
  try {
    const hosts = await wazuhHostController.getHostsEntries(false, false, fakeResponseEndpoint);

    if (hosts.body.length) {
      return hosts.body;
    }

    ;
    (0, _logger.log)('monitoring:getConfig', 'There are no Wazuh API entries yet', 'debug');
    return Promise.reject({
      error: 'no credentials',
      error_code: 1
    });
  } catch (error) {
    (0, _logger.log)('monitoring:getHostsConfiguration', error.message || error);
    return Promise.reject({
      error: 'no wazuh hosts',
      error_code: 2
    });
  }
}
/**
   * Task used by the cron job.
   */


async function cronTask(context) {
  try {
    const templateMonitoring = await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
      name: _constants.WAZUH_MONITORING_TEMPLATE_NAME
    });
    const apiHosts = await getHostsConfiguration();
    const apiHostsUnique = (apiHosts || []).filter((apiHost, index, self) => index === self.findIndex(t => t.user === apiHost.user && t.password === apiHost.password && t.url === apiHost.url && t.port === apiHost.port));

    for (let apiHost of apiHostsUnique) {
      try {
        const {
          agents,
          apiHost: host
        } = await getApiInfo(context, apiHost);
        await insertMonitoringDataElasticsearch(context, {
          agents,
          apiHost: host
        });
      } catch (error) {}

      ;
    }
  } catch (error) {
    // Retry to call itself again if Kibana index is not ready yet
    // try {
    //   if (
    //     this.wzWrapper.buildingKibanaIndex ||
    //     ((error || {}).status === 404 &&
    //       (error || {}).displayName === 'NotFound')
    //   ) {
    //     await delayAsPromise(1000);
    //     return cronTask(context);
    //   }
    // } catch (error) {} //eslint-disable-line
    (0, _logger.log)('monitoring:cronTask', error.message || error);
    context.wazuh.logger.error(error.message || error);
  }
}
/**
 * Get API and agents info
 * @param context
 * @param apiHost
 */


async function getApiInfo(context, apiHost) {
  try {
    (0, _logger.log)('monitoring:getApiInfo', `Getting API info for ${apiHost.id}`, 'debug');
    const responseIsCluster = await context.wazuh.api.client.asInternalUser.request('GET', '/cluster/status', {}, {
      apiHostID: apiHost.id
    });
    const isCluster = (((responseIsCluster || {}).data || {}).data || {}).enabled === 'yes';

    if (isCluster) {
      const responseClusterInfo = await context.wazuh.api.client.asInternalUser.request('GET', `/cluster/local/info`, {}, {
        apiHostID: apiHost.id
      });
      apiHost.clusterName = responseClusterInfo.data.data.affected_items[0].cluster;
    }

    ;
    const agents = await fetchAllAgentsFromApiHost(context, apiHost);
    return {
      agents,
      apiHost
    };
  } catch (error) {
    (0, _logger.log)('monitoring:getApiInfo', error.message || error);
    throw error;
  }
}

;
/**
 * Fetch all agents for the API provided
 * @param context
 * @param apiHost
 */

async function fetchAllAgentsFromApiHost(context, apiHost) {
  let agents = [];

  try {
    (0, _logger.log)('monitoring:fetchAllAgentsFromApiHost', `Getting all agents from ApiID: ${apiHost.id}`, 'debug');
    const responseAgentsCount = await context.wazuh.api.client.asInternalUser.request('GET', '/agents', {
      params: {
        offset: 0,
        limit: 1,
        q: 'id!=000'
      }
    }, {
      apiHostID: apiHost.id
    });
    const agentsCount = responseAgentsCount.data.data.total_affected_items;
    (0, _logger.log)('monitoring:fetchAllAgentsFromApiHost', `ApiID: ${apiHost.id}, Agent count: ${agentsCount}`, 'debug');
    let payload = {
      offset: 0,
      limit: 500,
      q: 'id!=000'
    };

    while (agents.length < agentsCount && payload.offset < agentsCount) {
      try {
        /* 
        TODO: Improve the performance of request with:
          - Reduce the number of requests to the Wazuh API
          - Reduce (if possible) the quantity of data to index by document
         Requirements:
          - Research about the neccesary data to index.
         How to do:
          - Wazuh API request:
            - select the required data to retrieve depending on is required to index (using the `select` query param)
            - increase the limit of results to retrieve (currently, the requests use the recommended value: 500).
              See the allowed values. This depends on the selected data because the response could fail if contains a lot of data
        */
        const responseAgents = await context.wazuh.api.client.asInternalUser.request('GET', `/agents`, {
          params: payload
        }, {
          apiHostID: apiHost.id
        });
        agents = [...agents, ...responseAgents.data.data.affected_items];
        payload.offset += payload.limit;
      } catch (error) {
        (0, _logger.log)('monitoring:fetchAllAgentsFromApiHost', `ApiID: ${apiHost.id}, Error request with offset/limit ${payload.offset}/${payload.limit}: ${error.message || error}`);
      }
    }

    return agents;
  } catch (error) {
    (0, _logger.log)('monitoring:fetchAllAgentsFromApiHost', `ApiID: ${apiHost.id}. Error: ${error.message || error}`);
    throw error;
  }
}

;
/**
 * Start the cron job
 */

async function jobMonitoringRun(context) {
  // Init the monitoring variables
  initMonitoringConfiguration(context); // Check Kibana index and if it is prepared, start the initialization of Wazuh App.

  await checkPluginPlatformStatus(context); // // Run the cron job only it it's enabled

  if (MONITORING_ENABLED) {
    cronTask(context);

    _nodeCron.default.schedule(MONITORING_CRON_FREQ, () => cronTask(context));
  }
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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