"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhElasticCtrl = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _errorResponse = require("../lib/error-response");

var _logger = require("../lib/logger");

var _getConfiguration = require("../lib/get-configuration");

var _visualizations = require("../integration-files/visualizations");

var _generateAlertsScript = require("../lib/generate-alerts/generate-alerts-script");

var _constants = require("../../common/constants");

var _jwtDecode = _interopRequireDefault(require("jwt-decode"));

var _manageHosts = require("../lib/manage-hosts");

var _cookie = require("../lib/cookie");

/*
 * Wazuh app - Class for Wazuh-Elastic functions
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
class WazuhElasticCtrl {
  constructor() {
    (0, _defineProperty2.default)(this, "wzSampleAlertsIndexPrefix", void 0);
    (0, _defineProperty2.default)(this, "manageHosts", void 0);
    this.wzSampleAlertsIndexPrefix = this.getSampleAlertPrefix();
    this.manageHosts = new _manageHosts.ManageHosts();
  }
  /**
   * This returns the index according the category
   * @param {string} category
   */


  buildSampleIndexByCategory(category) {
    return `${this.wzSampleAlertsIndexPrefix}sample-${category}`;
  }
  /**
   * This returns the defined config for sample alerts prefix or the default value.
   */


  getSampleAlertPrefix() {
    const config = (0, _getConfiguration.getConfiguration)();
    return config['alerts.sample.prefix'] || _constants.WAZUH_SAMPLE_ALERT_PREFIX;
  }
  /**
   * This retrieves a template from Elasticsearch
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} template or ErrorResponse
   */


  async getTemplate(context, request, response) {
    try {
      const data = await context.core.elasticsearch.client.asInternalUser.cat.templates();
      const templates = data.body;

      if (!templates || typeof templates !== 'string') {
        throw new Error('An unknown error occurred when fetching templates from Elasticseach');
      }

      const lastChar = request.params.pattern[request.params.pattern.length - 1]; // Split into separate patterns

      const tmpdata = templates.match(/\[.*\]/g);
      const tmparray = [];

      for (let item of tmpdata) {
        // A template might use more than one pattern
        if (item.includes(',')) {
          item = item.substr(1).slice(0, -1);
          const subItems = item.split(',');

          for (const subitem of subItems) {
            tmparray.push(`[${subitem.trim()}]`);
          }
        } else {
          tmparray.push(item);
        }
      } // Ensure we are handling just patterns


      const array = tmparray.filter(item => item.includes('[') && item.includes(']'));
      const pattern = lastChar === '*' ? request.params.pattern.slice(0, -1) : request.params.pattern;
      const isIncluded = array.filter(item => {
        item = item.slice(1, -1);
        const lastChar = item[item.length - 1];
        item = lastChar === '*' ? item.slice(0, -1) : item;
        return item.includes(pattern) || pattern.includes(item);
      });
      (0, _logger.log)('wazuh-elastic:getTemplate', `Template is valid: ${isIncluded && Array.isArray(isIncluded) && isIncluded.length ? 'yes' : 'no'}`, 'debug');
      return isIncluded && Array.isArray(isIncluded) && isIncluded.length ? response.ok({
        body: {
          statusCode: 200,
          status: true,
          data: `Template found for ${request.params.pattern}`
        }
      }) : response.ok({
        body: {
          statusCode: 200,
          status: false,
          data: `No template found for ${request.params.pattern}`
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:getTemplate', error.message || error);
      return (0, _errorResponse.ErrorResponse)(`Could not retrieve templates from Elasticsearch due to ${error.message || error}`, 4002, 500, response);
    }
  }
  /**
   * This check index-pattern
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} status obj or ErrorResponse
   */


  async checkPattern(context, request, response) {
    try {
      const data = await context.core.savedObjects.client.find({
        type: 'index-pattern'
      });
      const existsIndexPattern = data.saved_objects.find(item => item.attributes.title === request.params.pattern);
      (0, _logger.log)('wazuh-elastic:checkPattern', `Index pattern found: ${existsIndexPattern ? existsIndexPattern.attributes.title : 'no'}`, 'debug');
      return existsIndexPattern ? response.ok({
        body: {
          statusCode: 200,
          status: true,
          data: 'Index pattern found'
        }
      }) : response.ok({
        body: {
          statusCode: 500,
          status: false,
          error: 10020,
          message: 'Index pattern not found'
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:checkPattern', error.message || error);
      return (0, _errorResponse.ErrorResponse)(`Something went wrong retrieving index-patterns from Elasticsearch due to ${error.message || error}`, 4003, 500, response);
    }
  }
  /**
   * This get the fields keys
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Array<Object>} fields or ErrorResponse
   */


  async getFieldTop(context, request, response) {
    try {
      // Top field payload
      let payload = {
        size: 1,
        query: {
          bool: {
            must: [],
            must_not: {
              term: {
                'agent.id': '000'
              }
            },
            filter: [{
              range: {
                timestamp: {}
              }
            }]
          }
        },
        aggs: {
          '2': {
            terms: {
              field: '',
              size: 1,
              order: {
                _count: 'desc'
              }
            }
          }
        }
      }; // Set up time interval, default to Last 24h

      const timeGTE = 'now-1d';
      const timeLT = 'now';
      payload.query.bool.filter[0].range['timestamp']['gte'] = timeGTE;
      payload.query.bool.filter[0].range['timestamp']['lt'] = timeLT; // Set up match for default cluster name

      payload.query.bool.must.push(request.params.mode === 'cluster' ? {
        match: {
          'cluster.name': request.params.cluster
        }
      } : {
        match: {
          'manager.name': request.params.cluster
        }
      });
      if (request.query.agentsList) payload.query.bool.filter.push({
        terms: {
          'agent.id': request.query.agentsList.split(',')
        }
      });
      payload.aggs['2'].terms.field = request.params.field;
      const data = await context.core.elasticsearch.client.asCurrentUser.search({
        size: 1,
        index: request.params.pattern,
        body: payload
      });
      return data.body.hits.total.value === 0 || typeof data.body.aggregations['2'].buckets[0] === 'undefined' ? response.ok({
        body: {
          statusCode: 200,
          data: ''
        }
      }) : response.ok({
        body: {
          statusCode: 200,
          data: data.body.aggregations['2'].buckets[0].key
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:getFieldTop', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4004, 500, response);
    }
  }
  /**
   * Checks one by one if the requesting user has enough privileges to use
   * an index pattern from the list.
   * @param {Array<Object>} list List of index patterns
   * @param {Object} req
   * @returns {Array<Object>} List of allowed index
   */


  async filterAllowedIndexPatternList(context, list, req) {
    //TODO: review if necesary to delete
    let finalList = [];

    for (let item of list) {
      let results = false,
          forbidden = false;

      try {
        results = await context.core.elasticsearch.client.asCurrentUser.search({
          index: item.title
        });
      } catch (error) {
        forbidden = true;
      }

      if ((((results || {}).body || {}).hits || {}).total.value >= 1 || !forbidden && (((results || {}).body || {}).hits || {}).total === 0) {
        finalList.push(item);
      }
    }

    return finalList;
  }
  /**
   * Checks for minimum index pattern fields in a list of index patterns.
   * @param {Array<Object>} indexPatternList List of index patterns
   */


  validateIndexPattern(indexPatternList) {
    const minimum = ['timestamp', 'rule.groups', 'manager.name', 'agent.id'];
    let list = [];

    for (const index of indexPatternList) {
      let valid, parsed;

      try {
        parsed = JSON.parse(index.attributes.fields);
      } catch (error) {
        continue;
      }

      valid = parsed.filter(item => minimum.includes(item.name));

      if (valid.length === 4) {
        list.push({
          id: index.id,
          title: index.attributes.title
        });
      }
    }

    return list;
  }
  /**
   * Returns current security platform
   * @param {Object} req
   * @param {Object} reply
   * @returns {String}
   */


  async getCurrentPlatform(context, request, response) {
    try {
      return response.ok({
        body: {
          platform: context.wazuh.security.platform
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:getCurrentPlatform', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4011, 500, response);
    }
  }
  /**
   * Replaces visualizations main fields to fit a certain pattern.
   * @param {Array<Object>} app_objects Object containing raw visualizations.
   * @param {String} id Index-pattern id to use in the visualizations. Eg: 'wazuh-alerts'
   */


  async buildVisualizationsRaw(app_objects, id, namespace = false) {
    try {
      const config = (0, _getConfiguration.getConfiguration)();
      let monitoringPattern = (config || {})['wazuh.monitoring.pattern'] || _constants.WAZUH_MONITORING_PATTERN;
      (0, _logger.log)('wazuh-elastic:buildVisualizationsRaw', `Building ${app_objects.length} visualizations`, 'debug');
      (0, _logger.log)('wazuh-elastic:buildVisualizationsRaw', `Index pattern ID: ${id}`, 'debug');
      const visArray = [];
      let aux_source, bulk_content;

      for (let element of app_objects) {
        aux_source = JSON.parse(JSON.stringify(element._source)); // Replace index-pattern for visualizations

        if (aux_source && aux_source.kibanaSavedObjectMeta && aux_source.kibanaSavedObjectMeta.searchSourceJSON && typeof aux_source.kibanaSavedObjectMeta.searchSourceJSON === 'string') {
          const defaultStr = aux_source.kibanaSavedObjectMeta.searchSourceJSON;
          const isMonitoring = defaultStr.includes('wazuh-monitoring');

          if (isMonitoring) {
            if (namespace && namespace !== 'default') {
              if (monitoringPattern.includes(namespace) && monitoringPattern.includes('index-pattern:')) {
                monitoringPattern = monitoringPattern.split('index-pattern:')[1];
              }
            }

            aux_source.kibanaSavedObjectMeta.searchSourceJSON = defaultStr.replace(/wazuh-monitoring/g, monitoringPattern[monitoringPattern.length - 1] === '*' || namespace && namespace !== 'default' ? monitoringPattern : monitoringPattern + '*');
          } else {
            aux_source.kibanaSavedObjectMeta.searchSourceJSON = defaultStr.replace(/wazuh-alerts/g, id);
          }
        } // Replace index-pattern for selector visualizations


        if (typeof (aux_source || {}).visState === 'string') {
          aux_source.visState = aux_source.visState.replace(/wazuh-alerts/g, id);
        } // Bulk source


        bulk_content = {};
        bulk_content[element._type] = aux_source;
        visArray.push({
          attributes: bulk_content.visualization,
          type: element._type,
          id: element._id,
          _version: bulk_content.visualization.version
        });
      }

      return visArray;
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:buildVisualizationsRaw', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * Replaces cluster visualizations main fields.
   * @param {Array<Object>} app_objects Object containing raw visualizations.
   * @param {String} id Index-pattern id to use in the visualizations. Eg: 'wazuh-alerts'
   * @param {Array<String>} nodes Array of node names. Eg: ['node01', 'node02']
   * @param {String} name Cluster name. Eg: 'wazuh'
   * @param {String} master_node Master node name. Eg: 'node01'
   */


  buildClusterVisualizationsRaw(app_objects, id, nodes = [], name, master_node, pattern_name = '*') {
    try {
      const visArray = [];
      let aux_source, bulk_content;

      for (const element of app_objects) {
        // Stringify and replace index-pattern for visualizations
        aux_source = JSON.stringify(element._source);
        aux_source = aux_source.replace(/wazuh-alerts/g, id);
        aux_source = JSON.parse(aux_source); // Bulk source

        bulk_content = {};
        bulk_content[element._type] = aux_source;
        const visState = JSON.parse(bulk_content.visualization.visState);
        const title = visState.title;

        if (visState.type && visState.type === 'timelion') {
          let query = '';

          if (title === 'Wazuh App Cluster Overview') {
            for (const node of nodes) {
              query += `.es(index=${pattern_name},q="cluster.name: ${name} AND cluster.node: ${node.name}").label("${node.name}"),`;
            }

            query = query.substring(0, query.length - 1);
          } else if (title === 'Wazuh App Cluster Overview Manager') {
            query += `.es(index=${pattern_name},q="cluster.name: ${name}").label("${name} cluster")`;
          } else {
            if (title.startsWith('Wazuh App Statistics')) {
              const {
                searchSourceJSON
              } = bulk_content.visualization.kibanaSavedObjectMeta;
              bulk_content.visualization.kibanaSavedObjectMeta.searchSourceJSON = searchSourceJSON.replace('wazuh-statistics-*', pattern_name);
            }

            if (title.startsWith('Wazuh App Statistics') && name !== '-' && name !== 'all' && visState.params.expression.includes('q=')) {
              const expressionRegex = /q='\*'/gi;

              const _visState = bulk_content.visualization.visStateByNode ? JSON.parse(bulk_content.visualization.visStateByNode) : visState;

              query += _visState.params.expression.replace(/wazuh-statistics-\*/g, pattern_name).replace(expressionRegex, `q="nodeName.keyword:${name} AND apiName.keyword:${master_node}"`).replace("NODE_NAME", name);
            } else if (title.startsWith('Wazuh App Statistics')) {
              const expressionRegex = /q='\*'/gi;
              query += visState.params.expression.replace(/wazuh-statistics-\*/g, pattern_name).replace(expressionRegex, `q="apiName.keyword:${master_node}"`);
            } else {
              query = visState.params.expression;
            }
          }

          visState.params.expression = query.replace(/'/g, "\"");
          bulk_content.visualization.visState = JSON.stringify(visState);
        }

        visArray.push({
          attributes: bulk_content.visualization,
          type: element._type,
          id: element._id,
          _version: bulk_content.visualization.version
        });
      }

      return visArray;
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:buildClusterVisualizationsRaw', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This creates a visualization of data in req
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} vis obj or ErrorResponse
   */


  async createVis(context, request, response) {
    try {
      if (!request.params.tab.includes('overview-') && !request.params.tab.includes('agents-')) {
        throw new Error('Missing parameters creating visualizations');
      }

      const tabPrefix = request.params.tab.includes('overview') ? 'overview' : 'agents';
      const tabSplit = request.params.tab.split('-');
      const tabSufix = tabSplit[1];
      const file = tabPrefix === 'overview' ? _visualizations.OverviewVisualizations[tabSufix] : _visualizations.AgentsVisualizations[tabSufix];

      if (!file) {
        return response.notFound({
          body: {
            message: `Visualizations not found for ${request.params.tab}`
          }
        });
      }

      (0, _logger.log)('wazuh-elastic:createVis', `${tabPrefix}[${tabSufix}] with index pattern ${request.params.pattern}`, 'debug');
      const namespace = context.wazuh.plugins.spaces && context.wazuh.plugins.spaces.spacesService && context.wazuh.plugins.spaces.spacesService.getSpaceId(request);
      const raw = await this.buildVisualizationsRaw(file, request.params.pattern, namespace);
      return response.ok({
        body: {
          acknowledge: true,
          raw: raw
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:createVis', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4007, 500, response);
    }
  }
  /**
   * This creates a visualization of cluster
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} vis obj or ErrorResponse
   */


  async createClusterVis(context, request, response) {
    try {
      if (!request.params.pattern || !request.params.tab || !request.body || !request.body.nodes || !request.body.nodes.affected_items || !request.body.nodes.name || request.params.tab && !request.params.tab.includes('cluster-')) {
        throw new Error('Missing parameters creating visualizations');
      }

      const type = request.params.tab.split('-')[1];
      const file = _visualizations.ClusterVisualizations[type];
      const nodes = request.body.nodes.affected_items;
      const name = request.body.nodes.name;
      const masterNode = request.body.nodes.master_node;
      const {
        id: patternID,
        title: patternName
      } = request.body.pattern;
      const raw = await this.buildClusterVisualizationsRaw(file, patternID, nodes, name, masterNode, patternName);
      return response.ok({
        body: {
          acknowledge: true,
          raw: raw
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:createClusterVis', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4009, 500, response);
    }
  }
  /**
   * This checks if there is sample alerts
   * GET /elastic/samplealerts
   * @param {*} context
   * @param {*} request
   * @param {*} response
   * {alerts: [...]} or ErrorResponse
   */


  async haveSampleAlerts(context, request, response) {
    try {
      // Check if wazuh sample alerts index exists
      const results = await Promise.all(Object.keys(_constants.WAZUH_SAMPLE_ALERTS_CATEGORIES_TYPE_ALERTS).map(category => context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: this.buildSampleIndexByCategory(category)
      })));
      return response.ok({
        body: {
          sampleAlertsInstalled: results.some(result => result.body)
        }
      });
    } catch (error) {
      return (0, _errorResponse.ErrorResponse)('Sample Alerts category not valid', 1000, 500, response);
    }
  }
  /**
   * This creates sample alerts in wazuh-sample-alerts
   * GET /elastic/samplealerts/{category}
   * @param {*} context
   * @param {*} request
   * @param {*} response
   * {alerts: [...]} or ErrorResponse
   */


  async haveSampleAlertsOfCategory(context, request, response) {
    try {
      const sampleAlertsIndex = this.buildSampleIndexByCategory(request.params.category); // Check if wazuh sample alerts index exists

      const existsSampleIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: sampleAlertsIndex
      });
      return response.ok({
        body: {
          index: sampleAlertsIndex,
          exists: existsSampleIndex.body
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:haveSampleAlertsOfCategory', `Error checking if there are sample alerts indices: ${error.message || error}`);
      const [statusCode, errorMessage] = this.getErrorDetails(error);
      return (0, _errorResponse.ErrorResponse)(`Error checking if there are sample alerts indices: ${errorMessage || error}`, 1000, statusCode, response);
    }
  }
  /**
   * This creates sample alerts in wazuh-sample-alerts
   * POST /elastic/samplealerts/{category}
   * {
   *   "manager": {
   *      "name": "manager_name"
   *    },
   *    cluster: {
   *      name: "mycluster",
   *      node: "mynode"
   *    }
   * }
   * @param {*} context
   * @param {*} request
   * @param {*} response
   * {index: string, alerts: [...], count: number} or ErrorResponse
   */


  async createSampleAlerts(context, request, response) {
    const sampleAlertsIndex = this.buildSampleIndexByCategory(request.params.category);

    try {
      // Check if user has administrator role in token
      const token = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-token');

      if (!token) {
        return (0, _errorResponse.ErrorResponse)('No token provided', 401, 401, response);
      }

      ;
      const decodedToken = (0, _jwtDecode.default)(token);

      if (!decodedToken) {
        return (0, _errorResponse.ErrorResponse)('No permissions in token', 401, 401, response);
      }

      ;

      if (!decodedToken.rbac_roles || !decodedToken.rbac_roles.includes(_constants.WAZUH_ROLE_ADMINISTRATOR_ID)) {
        return (0, _errorResponse.ErrorResponse)('No administrator role', 401, 401, response);
      }

      ; // Check the provided token is valid

      const apiHostID = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-api');

      if (!apiHostID) {
        return (0, _errorResponse.ErrorResponse)('No API id provided', 401, 401, response);
      }

      ;
      const responseTokenIsWorking = await context.wazuh.api.client.asCurrentUser.request('GET', `//`, {}, {
        apiHostID
      });

      if (responseTokenIsWorking.status !== 200) {
        return (0, _errorResponse.ErrorResponse)('Token is not valid', 500, 500, response);
      }

      ;
      const bulkPrefix = JSON.stringify({
        index: {
          _index: sampleAlertsIndex
        }
      });
      const alertGenerateParams = request.body && request.body.params || {};

      const sampleAlerts = _constants.WAZUH_SAMPLE_ALERTS_CATEGORIES_TYPE_ALERTS[request.params.category].map(typeAlert => (0, _generateAlertsScript.generateAlerts)({ ...typeAlert,
        ...alertGenerateParams
      }, request.body.alerts || typeAlert.alerts || _constants.WAZUH_SAMPLE_ALERTS_DEFAULT_NUMBER_ALERTS)).flat();

      const bulk = sampleAlerts.map(sampleAlert => `${bulkPrefix}\n${JSON.stringify(sampleAlert)}\n`).join(''); // Index alerts
      // Check if wazuh sample alerts index exists

      const existsSampleIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: sampleAlertsIndex
      });

      if (!existsSampleIndex.body) {
        // Create wazuh sample alerts index
        const configuration = {
          settings: {
            index: {
              number_of_shards: _constants.WAZUH_SAMPLE_ALERTS_INDEX_SHARDS,
              number_of_replicas: _constants.WAZUH_SAMPLE_ALERTS_INDEX_REPLICAS
            }
          }
        };
        await context.core.elasticsearch.client.asCurrentUser.indices.create({
          index: sampleAlertsIndex,
          body: configuration
        });
        (0, _logger.log)('wazuh-elastic:createSampleAlerts', `Created ${sampleAlertsIndex} index`, 'debug');
      }

      await context.core.elasticsearch.client.asCurrentUser.bulk({
        index: sampleAlertsIndex,
        body: bulk
      });
      (0, _logger.log)('wazuh-elastic:createSampleAlerts', `Added sample alerts to ${sampleAlertsIndex} index`, 'debug');
      return response.ok({
        body: {
          index: sampleAlertsIndex,
          alertCount: sampleAlerts.length
        }
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:createSampleAlerts', `Error adding sample alerts to ${sampleAlertsIndex} index: ${error.message || error}`);
      const [statusCode, errorMessage] = this.getErrorDetails(error);
      return (0, _errorResponse.ErrorResponse)(errorMessage || error, 1000, statusCode, response);
    }
  }
  /**
   * This deletes sample alerts
   * @param {*} context
   * @param {*} request
   * @param {*} response
   * {result: "deleted", index: string} or ErrorResponse
   */


  async deleteSampleAlerts(context, request, response) {
    // Delete Wazuh sample alert index
    const sampleAlertsIndex = this.buildSampleIndexByCategory(request.params.category);

    try {
      // Check if user has administrator role in token
      const token = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-token');

      if (!token) {
        return (0, _errorResponse.ErrorResponse)('No token provided', 401, 401, response);
      }

      ;
      const decodedToken = (0, _jwtDecode.default)(token);

      if (!decodedToken) {
        return (0, _errorResponse.ErrorResponse)('No permissions in token', 401, 401, response);
      }

      ;

      if (!decodedToken.rbac_roles || !decodedToken.rbac_roles.includes(_constants.WAZUH_ROLE_ADMINISTRATOR_ID)) {
        return (0, _errorResponse.ErrorResponse)('No administrator role', 401, 401, response);
      }

      ; // Check the provided token is valid

      const apiHostID = (0, _cookie.getCookieValueByName)(request.headers.cookie, 'wz-api');

      if (!apiHostID) {
        return (0, _errorResponse.ErrorResponse)('No API id provided', 401, 401, response);
      }

      ;
      const responseTokenIsWorking = await context.wazuh.api.client.asCurrentUser.request('GET', `//`, {}, {
        apiHostID
      });

      if (responseTokenIsWorking.status !== 200) {
        return (0, _errorResponse.ErrorResponse)('Token is not valid', 500, 500, response);
      }

      ; // Check if Wazuh sample alerts index exists

      const existsSampleIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: sampleAlertsIndex
      });

      if (existsSampleIndex.body) {
        // Delete Wazuh sample alerts index
        await context.core.elasticsearch.client.asCurrentUser.indices.delete({
          index: sampleAlertsIndex
        });
        (0, _logger.log)('wazuh-elastic:deleteSampleAlerts', `Deleted ${sampleAlertsIndex} index`, 'debug');
        return response.ok({
          body: {
            result: 'deleted',
            index: sampleAlertsIndex
          }
        });
      } else {
        return (0, _errorResponse.ErrorResponse)(`${sampleAlertsIndex} index doesn't exist`, 1000, 500, response);
      }
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:deleteSampleAlerts', `Error deleting sample alerts of ${sampleAlertsIndex} index: ${error.message || error}`);
      const [statusCode, errorMessage] = this.getErrorDetails(error);
      return (0, _errorResponse.ErrorResponse)(errorMessage || error, 1000, statusCode, response);
    }
  }

  async alerts(context, request, response) {
    try {
      const data = await context.core.elasticsearch.client.asCurrentUser.search(request.body);
      return response.ok({
        body: data.body
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:alerts', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 4010, 500, response);
    }
  } // Check if there are indices for Statistics


  async existStatisticsIndices(context, request, response) {
    try {
      const config = (0, _getConfiguration.getConfiguration)();
      const statisticsPattern = `${config['cron.prefix'] || 'wazuh'}-${config['cron.statistics.index.name'] || 'statistics'}*`; //TODO: replace by default as constants instead hardcoded ('wazuh' and 'statistics')

      const existIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: statisticsPattern,
        allow_no_indices: false
      });
      return response.ok({
        body: existIndex.body
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:existsStatisticsIndices', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 1000, 500, response);
    }
  } // Check if there are indices for Monitoring


  async existMonitoringIndices(context, request, response) {
    try {
      const config = (0, _getConfiguration.getConfiguration)();
      const monitoringIndexPattern = config['wazuh.monitoring.pattern'] || _constants.WAZUH_MONITORING_PATTERN;
      const existIndex = await context.core.elasticsearch.client.asCurrentUser.indices.exists({
        index: monitoringIndexPattern,
        allow_no_indices: false
      });
      return response.ok({
        body: existIndex.body
      });
    } catch (error) {
      (0, _logger.log)('wazuh-elastic:existsMonitoringIndices', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 1000, 500, response);
    }
  }

  async usingCredentials(context) {
    try {
      const data = await context.core.elasticsearch.client.asInternalUser.cluster.getSettings({
        include_defaults: true
      });
      return (((((data || {}).body || {}).defaults || {}).xpack || {}).security || {}).user !== null;
    } catch (error) {
      return Promise.reject(error);
    }
  }

  getErrorDetails(error) {
    var _error$meta;

    const statusCode = (error === null || error === void 0 ? void 0 : (_error$meta = error.meta) === null || _error$meta === void 0 ? void 0 : _error$meta.statusCode) || 500;
    let errorMessage = error.message;

    if (statusCode === 403) {
      var _error$meta2, _error$meta2$body, _error$meta2$body$err;

      errorMessage = (error === null || error === void 0 ? void 0 : (_error$meta2 = error.meta) === null || _error$meta2 === void 0 ? void 0 : (_error$meta2$body = _error$meta2.body) === null || _error$meta2$body === void 0 ? void 0 : (_error$meta2$body$err = _error$meta2$body.error) === null || _error$meta2$body$err === void 0 ? void 0 : _error$meta2$body$err.reason) || 'Permission denied';
    }

    return [statusCode, errorMessage];
  }

}

exports.WazuhElasticCtrl = WazuhElasticCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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