"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhReportingCtrl = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _path = _interopRequireDefault(require("path"));

var _fs = _interopRequireDefault(require("fs"));

var _wazuhModules = require("../../common/wazuh-modules");

var TimSort = _interopRequireWildcard(require("timsort"));

var _errorResponse = require("../lib/error-response");

var VulnerabilityRequest = _interopRequireWildcard(require("../lib/reporting/vulnerability-request"));

var OverviewRequest = _interopRequireWildcard(require("../lib/reporting/overview-request"));

var RootcheckRequest = _interopRequireWildcard(require("../lib/reporting/rootcheck-request"));

var PCIRequest = _interopRequireWildcard(require("../lib/reporting/pci-request"));

var GDPRRequest = _interopRequireWildcard(require("../lib/reporting/gdpr-request"));

var TSCRequest = _interopRequireWildcard(require("../lib/reporting/tsc-request"));

var AuditRequest = _interopRequireWildcard(require("../lib/reporting/audit-request"));

var SyscheckRequest = _interopRequireWildcard(require("../lib/reporting/syscheck-request"));

var _pciRequirementsPdfmake = _interopRequireDefault(require("../integration-files/pci-requirements-pdfmake"));

var _gdprRequirementsPdfmake = _interopRequireDefault(require("../integration-files/gdpr-requirements-pdfmake"));

var _tscRequirementsPdfmake = _interopRequireDefault(require("../integration-files/tsc-requirements-pdfmake"));

var _processStateEquivalence = _interopRequireDefault(require("../lib/process-state-equivalence"));

var _csvKeyEquivalence = require("../../common/csv-key-equivalence");

var _agentConfiguration = require("../lib/reporting/agent-configuration");

var _printer = require("../lib/reporting/printer");

var _logger = require("../lib/logger");

var _constants = require("../../common/constants");

var _filesystem = require("../lib/filesystem");

var _moment = _interopRequireDefault(require("moment"));

var _wz_agent_status = require("../../common/services/wz_agent_status");

function _getRequireWildcardCache(nodeInterop) { if (typeof WeakMap !== "function") return null; var cacheBabelInterop = new WeakMap(); var cacheNodeInterop = new WeakMap(); return (_getRequireWildcardCache = function (nodeInterop) { return nodeInterop ? cacheNodeInterop : cacheBabelInterop; })(nodeInterop); }

function _interopRequireWildcard(obj, nodeInterop) { if (!nodeInterop && obj && obj.__esModule) { return obj; } if (obj === null || typeof obj !== "object" && typeof obj !== "function") { return { default: obj }; } var cache = _getRequireWildcardCache(nodeInterop); if (cache && cache.has(obj)) { return cache.get(obj); } var newObj = {}; var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var key in obj) { if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) { var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null; if (desc && (desc.get || desc.set)) { Object.defineProperty(newObj, key, desc); } else { newObj[key] = obj[key]; } } } newObj.default = obj; if (cache) { cache.set(obj, newObj); } return newObj; }

/*
 * Wazuh app - Class for Wazuh reporting controller
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
class WazuhReportingCtrl {
  constructor() {
    (0, _defineProperty2.default)(this, "createReportsModules", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsModules', `Report started`, 'info');
        const {
          array,
          agents,
          browserTimezone,
          searchBar,
          filters,
          time,
          tables,
          section,
          indexPatternTitle,
          apiId
        } = request.body;
        const {
          moduleID
        } = request.params;
        const {
          from,
          to
        } = time || {}; // Init

        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        await this.renderHeader(context, printer, section, moduleID, agents, apiId);
        const [sanitizedFilters, agentsFilter] = filters ? this.sanitizeKibanaFilters(filters, searchBar) : [false, false];

        if (time && sanitizedFilters) {
          printer.addTimeRangeAndFilters(from, to, sanitizedFilters, browserTimezone);
        }

        if (time) {
          await this.extendedInformation(context, printer, section, moduleID, apiId, new Date(from).getTime(), new Date(to).getTime(), sanitizedFilters, indexPatternTitle, agents);
        }

        printer.addVisualizations(array, agents, moduleID);

        if (tables) {
          printer.addTables(tables);
        } //add authorized agents


        if (agentsFilter) {
          printer.addAgentsFilters(agentsFilter);
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      body: {
        agents
      },
      params: {
        moduleID
      }
    }) => `wazuh-module-${agents ? `agents-${agents}` : 'overview'}-${moduleID}-${this.generateReportTimestamp()}.pdf`));
    (0, _defineProperty2.default)(this, "createReportsGroups", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsGroups', `Report started`, 'info');
        const {
          components,
          apiId
        } = request.body;
        const {
          groupID
        } = request.params; // Init

        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        let tables = [];
        const equivalences = {
          localfile: 'Local files',
          osquery: 'Osquery',
          command: 'Command',
          syscheck: 'Syscheck',
          'open-scap': 'OpenSCAP',
          'cis-cat': 'CIS-CAT',
          syscollector: 'Syscollector',
          rootcheck: 'Rootcheck',
          labels: 'Labels',
          sca: 'Security configuration assessment'
        };
        printer.addContent({
          text: `Group ${groupID} configuration`,
          style: 'h1'
        }); // Group configuration

        if (components['0']) {
          const {
            data: {
              data: configuration
            }
          } = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${groupID}/configuration`, {}, {
            apiHostID: apiId
          });

          if (configuration.affected_items.length > 0 && Object.keys(configuration.affected_items[0].config).length) {
            printer.addContent({
              text: 'Configurations',
              style: {
                fontSize: 14,
                color: '#000'
              },
              margin: [0, 10, 0, 15]
            });
            const section = {
              labels: [],
              isGroupConfig: true
            };

            for (let config of configuration.affected_items) {
              let filterTitle = '';
              let index = 0;

              for (let filter of Object.keys(config.filters)) {
                filterTitle = filterTitle.concat(`${filter}: ${config.filters[filter]}`);

                if (index < Object.keys(config.filters).length - 1) {
                  filterTitle = filterTitle.concat(' | ');
                }

                index++;
              }

              printer.addContent({
                text: filterTitle,
                style: 'h4',
                margin: [0, 0, 0, 10]
              });
              let idx = 0;
              section.tabs = [];

              for (let _d of Object.keys(config.config)) {
                for (let c of _agentConfiguration.AgentConfiguration.configurations) {
                  for (let s of c.sections) {
                    section.opts = s.opts || {};

                    for (let cn of s.config || []) {
                      if (cn.configuration === _d) {
                        section.labels = s.labels || [[]];
                      }
                    }

                    for (let wo of s.wodle || []) {
                      if (wo.name === _d) {
                        section.labels = s.labels || [[]];
                      }
                    }
                  }
                }

                section.labels[0]['pack'] = 'Packs';
                section.labels[0]['content'] = 'Evaluations';
                section.labels[0]['7'] = 'Scan listening netwotk ports';
                section.tabs.push(equivalences[_d]);

                if (Array.isArray(config.config[_d])) {
                  /* LOG COLLECTOR */
                  if (_d === 'localfile') {
                    let groups = [];

                    config.config[_d].forEach(obj => {
                      if (!groups[obj.logformat]) {
                        groups[obj.logformat] = [];
                      }

                      groups[obj.logformat].push(obj);
                    });

                    Object.keys(groups).forEach(group => {
                      let saveidx = 0;
                      groups[group].forEach((x, i) => {
                        if (Object.keys(x).length > Object.keys(groups[group][saveidx]).length) {
                          saveidx = i;
                        }
                      });
                      const columns = Object.keys(groups[group][saveidx]);
                      const rows = groups[group].map(x => {
                        let row = [];
                        columns.forEach(key => {
                          row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
                            return x + '\n';
                          }) : JSON.stringify(x[key]));
                        });
                        return row;
                      });
                      columns.forEach((col, i) => {
                        columns[i] = col[0].toUpperCase() + col.slice(1);
                      });
                      tables.push({
                        title: 'Local files',
                        type: 'table',
                        columns,
                        rows
                      });
                    });
                  } else if (_d === 'labels') {
                    const obj = config.config[_d][0].label;
                    const columns = Object.keys(obj[0]);

                    if (!columns.includes('hidden')) {
                      columns.push('hidden');
                    }

                    const rows = obj.map(x => {
                      let row = [];
                      columns.forEach(key => {
                        row.push(x[key]);
                      });
                      return row;
                    });
                    columns.forEach((col, i) => {
                      columns[i] = col[0].toUpperCase() + col.slice(1);
                    });
                    tables.push({
                      title: 'Labels',
                      type: 'table',
                      columns,
                      rows
                    });
                  } else {
                    for (let _d2 of config.config[_d]) {
                      tables.push(...this.getConfigTables(_d2, section, idx));
                    }
                  }
                } else {
                  /*INTEGRITY MONITORING MONITORED DIRECTORIES */
                  if (config.config[_d].directories) {
                    const directories = config.config[_d].directories;
                    delete config.config[_d].directories;
                    tables.push(...this.getConfigTables(config.config[_d], section, idx));
                    let diffOpts = [];
                    Object.keys(section.opts).forEach(x => {
                      diffOpts.push(x);
                    });
                    const columns = ['', ...diffOpts.filter(x => x !== 'check_all' && x !== 'check_sum')];
                    let rows = [];
                    directories.forEach(x => {
                      let row = [];
                      row.push(x.path);
                      columns.forEach(y => {
                        if (y !== '') {
                          y = y !== 'check_whodata' ? y : 'whodata';
                          row.push(x[y] ? x[y] : 'no');
                        }
                      });
                      row.push(x.recursion_level);
                      rows.push(row);
                    });
                    columns.forEach((x, idx) => {
                      columns[idx] = section.opts[x];
                    });
                    columns.push('RL');
                    tables.push({
                      title: 'Monitored directories',
                      type: 'table',
                      columns,
                      rows
                    });
                  } else {
                    tables.push(...this.getConfigTables(config.config[_d], section, idx));
                  }
                }

                for (const table of tables) {
                  printer.addConfigTables([table]);
                }

                idx++;
                tables = [];
              }

              tables = [];
            }
          } else {
            printer.addContent({
              text: 'A configuration for this group has not yet been set up.',
              style: {
                fontSize: 12,
                color: '#000'
              },
              margin: [0, 10, 0, 15]
            });
          }
        } // Agents in group


        if (components['1']) {
          await this.renderHeader(context, printer, 'groupConfig', groupID, [], apiId);
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsGroups', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        groupID
      }
    }) => `wazuh-group-configuration-${groupID}-${this.generateReportTimestamp()}.pdf`));
    (0, _defineProperty2.default)(this, "createReportsAgentsConfiguration", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Report started`, 'info');
        const {
          components,
          apiId
        } = request.body;
        const {
          agentID
        } = request.params;
        const printer = new _printer.ReportPrinter();
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, context.wazuhEndpointParams.hashUsername));
        let wmodulesResponse = {};
        let tables = [];

        try {
          wmodulesResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents/${agentID}/config/wmodules/wmodules`, {}, {
            apiHostID: apiId
          });
        } catch (error) {
          (0, _logger.log)('reporting:report', error.message || error, 'debug');
        }

        await this.renderHeader(context, printer, 'agentConfig', 'agentConfig', agentID, apiId);
        let idxComponent = 0;

        for (let config of _agentConfiguration.AgentConfiguration.configurations) {
          let titleOfSection = false;
          (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Iterate over ${config.sections.length} configuration sections`, 'debug');

          for (let section of config.sections) {
            let titleOfSubsection = false;

            if (components[idxComponent] && (section.config || section.wodle)) {
              let idx = 0;
              const configs = (section.config || []).concat(section.wodle || []);
              (0, _logger.log)('reporting:createReportsAgentsConfiguration', `Iterate over ${configs.length} configuration blocks`, 'debug');

              for (let conf of configs) {
                let agentConfigResponse = {};

                try {
                  if (!conf['name']) {
                    agentConfigResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents/${agentID}/config/${conf.component}/${conf.configuration}`, {}, {
                      apiHostID: apiId
                    });
                  } else {
                    for (let wodle of wmodulesResponse.data.data['wmodules']) {
                      if (Object.keys(wodle)[0] === conf['name']) {
                        agentConfigResponse.data = {
                          data: wodle
                        };
                      }
                    }
                  }

                  const agentConfig = agentConfigResponse && agentConfigResponse.data && agentConfigResponse.data.data;

                  if (!titleOfSection) {
                    printer.addContent({
                      text: config.title,
                      style: 'h1',
                      margin: [0, 0, 0, 15]
                    });
                    titleOfSection = true;
                  }

                  if (!titleOfSubsection) {
                    printer.addContent({
                      text: section.subtitle,
                      style: 'h4'
                    });
                    printer.addContent({
                      text: section.desc,
                      style: {
                        fontSize: 12,
                        color: '#000'
                      },
                      margin: [0, 0, 0, 10]
                    });
                    titleOfSubsection = true;
                  }

                  if (agentConfig) {
                    for (let agentConfigKey of Object.keys(agentConfig)) {
                      if (Array.isArray(agentConfig[agentConfigKey])) {
                        /* LOG COLLECTOR */
                        if (conf.filterBy) {
                          let groups = [];
                          agentConfig[agentConfigKey].forEach(obj => {
                            if (!groups[obj.logformat]) {
                              groups[obj.logformat] = [];
                            }

                            groups[obj.logformat].push(obj);
                          });
                          Object.keys(groups).forEach(group => {
                            let saveidx = 0;
                            groups[group].forEach((x, i) => {
                              if (Object.keys(x).length > Object.keys(groups[group][saveidx]).length) {
                                saveidx = i;
                              }
                            });
                            const columns = Object.keys(groups[group][saveidx]);
                            const rows = groups[group].map(x => {
                              let row = [];
                              columns.forEach(key => {
                                row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
                                  return x + '\n';
                                }) : JSON.stringify(x[key]));
                              });
                              return row;
                            });
                            columns.forEach((col, i) => {
                              columns[i] = col[0].toUpperCase() + col.slice(1);
                            });
                            tables.push({
                              title: section.labels[0][group],
                              type: 'table',
                              columns,
                              rows
                            });
                          });
                        } else if (agentConfigKey.configuration !== 'socket') {
                          tables.push(...this.getConfigTables(agentConfig[agentConfigKey], section, idx));
                        } else {
                          for (let _d2 of agentConfig[agentConfigKey]) {
                            tables.push(...this.getConfigTables(_d2, section, idx));
                          }
                        }
                      } else {
                        /*INTEGRITY MONITORING MONITORED DIRECTORIES */
                        if (conf.matrix) {
                          const {
                            directories,
                            diff,
                            synchronization,
                            file_limit,
                            ...rest
                          } = agentConfig[agentConfigKey];
                          tables.push(...this.getConfigTables(rest, section, idx), ...(diff && diff.disk_quota ? this.getConfigTables(diff.disk_quota, {
                            tabs: ['Disk quota']
                          }, 0) : []), ...(diff && diff.file_size ? this.getConfigTables(diff.file_size, {
                            tabs: ['File size']
                          }, 0) : []), ...(synchronization ? this.getConfigTables(synchronization, {
                            tabs: ['Synchronization']
                          }, 0) : []), ...(file_limit ? this.getConfigTables(file_limit, {
                            tabs: ['File limit']
                          }, 0) : []));
                          let diffOpts = [];
                          Object.keys(section.opts).forEach(x => {
                            diffOpts.push(x);
                          });
                          const columns = ['', ...diffOpts.filter(x => x !== 'check_all' && x !== 'check_sum')];
                          let rows = [];
                          directories.forEach(x => {
                            let row = [];
                            row.push(x.dir);
                            columns.forEach(y => {
                              if (y !== '') {
                                row.push(x.opts.indexOf(y) > -1 ? 'yes' : 'no');
                              }
                            });
                            row.push(x.recursion_level);
                            rows.push(row);
                          });
                          columns.forEach((x, idx) => {
                            columns[idx] = section.opts[x];
                          });
                          columns.push('RL');
                          tables.push({
                            title: 'Monitored directories',
                            type: 'table',
                            columns,
                            rows
                          });
                        } else {
                          tables.push(...this.getConfigTables(agentConfig[agentConfigKey], section, idx));
                        }
                      }
                    }
                  } else {
                    // Print no configured module and link to the documentation
                    printer.addContent({
                      text: ['This module is not configured. Please take a look on how to configure it in ', {
                        text: `${section.subtitle.toLowerCase()} configuration.`,
                        link: section.docuLink,
                        style: {
                          fontSize: 12,
                          color: '#1a0dab'
                        }
                      }],
                      margin: [0, 0, 0, 20]
                    });
                  }
                } catch (error) {
                  (0, _logger.log)('reporting:report', error.message || error, 'debug');
                }

                idx++;
              }

              for (const table of tables) {
                printer.addConfigTables([table]);
              }
            }

            idxComponent++;
            tables = [];
          }
        }

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsAgentsConfiguration', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        agentID
      }
    }) => `wazuh-agent-configuration-${agentID}-${this.generateReportTimestamp()}.pdf`));
    (0, _defineProperty2.default)(this, "createReportsAgentsInventory", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:createReportsAgentsInventory', `Report started`, 'info');
        const {
          searchBar,
          filters,
          time,
          indexPatternTitle,
          apiId
        } = request.body;
        const {
          agentID
        } = request.params;
        const {
          from,
          to
        } = time || {}; // Init

        const printer = new _printer.ReportPrinter();
        const {
          hashUsername
        } = await context.wazuh.security.getCurrentUser(request, context);
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);
        (0, _filesystem.createDirectoryIfNotExists)(_path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername));
        (0, _logger.log)('reporting:createReportsAgentsInventory', `Syscollector report`, 'debug');
        const sanitizedFilters = filters ? this.sanitizeKibanaFilters(filters, searchBar) : false; // Get the agent OS

        let agentOs = '';

        try {
          const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', '/agents', {
            params: {
              q: `id=${agentID}`
            }
          }, {
            apiHostID: apiId
          });
          agentOs = agentResponse.data.data.affected_items[0].os.platform;
        } catch (error) {
          (0, _logger.log)('reporting:createReportsAgentsInventory', error.message || error, 'debug');
        } // Add title


        printer.addContentWithNewLine({
          text: 'Inventory data report',
          style: 'h1'
        }); // Add table with the agent info

        await this.buildAgentsTable(context, printer, [agentID], apiId); // Get syscollector packages and processes

        const agentRequestsInventory = [{
          endpoint: `/syscollector/${agentID}/packages`,
          loggerMessage: `Fetching packages for agent ${agentID}`,
          table: {
            title: 'Packages',
            columns: agentOs === 'windows' ? [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'architecture',
              label: 'Architecture'
            }, {
              id: 'version',
              label: 'Version'
            }, {
              id: 'vendor',
              label: 'Vendor'
            }] : [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'architecture',
              label: 'Architecture'
            }, {
              id: 'version',
              label: 'Version'
            }, {
              id: 'vendor',
              label: 'Vendor'
            }, {
              id: 'description',
              label: 'Description'
            }]
          }
        }, {
          endpoint: `/syscollector/${agentID}/processes`,
          loggerMessage: `Fetching processes for agent ${agentID}`,
          table: {
            title: 'Processes',
            columns: agentOs === 'windows' ? [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'cmd',
              label: 'CMD'
            }, {
              id: 'priority',
              label: 'Priority'
            }, {
              id: 'nlwp',
              label: 'NLWP'
            }] : [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'euser',
              label: 'Effective user'
            }, {
              id: 'nice',
              label: 'Priority'
            }, {
              id: 'state',
              label: 'State'
            }]
          },
          mapResponseItems: item => agentOs === 'windows' ? item : { ...item,
            state: _processStateEquivalence.default[item.state]
          }
        }, {
          endpoint: `/syscollector/${agentID}/ports`,
          loggerMessage: `Fetching ports for agent ${agentID}`,
          table: {
            title: 'Network ports',
            columns: agentOs === 'windows' ? [{
              id: 'local_ip',
              label: 'Local IP'
            }, {
              id: 'local_port',
              label: 'Local port'
            }, {
              id: 'process',
              label: 'Process'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'protocol',
              label: 'Protocol'
            }] : [{
              id: 'local_ip',
              label: 'Local IP'
            }, {
              id: 'local_port',
              label: 'Local port'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'protocol',
              label: 'Protocol'
            }]
          },
          mapResponseItems: item => ({ ...item,
            local_ip: item.local.ip,
            local_port: item.local.port
          })
        }, {
          endpoint: `/syscollector/${agentID}/netiface`,
          loggerMessage: `Fetching netiface for agent ${agentID}`,
          table: {
            title: 'Network interfaces',
            columns: [{
              id: 'name',
              label: 'Name'
            }, {
              id: 'mac',
              label: 'Mac'
            }, {
              id: 'state',
              label: 'State'
            }, {
              id: 'mtu',
              label: 'MTU'
            }, {
              id: 'type',
              label: 'Type'
            }]
          }
        }, {
          endpoint: `/syscollector/${agentID}/netaddr`,
          loggerMessage: `Fetching netaddr for agent ${agentID}`,
          table: {
            title: 'Network settings',
            columns: [{
              id: 'iface',
              label: 'Interface'
            }, {
              id: 'address',
              label: 'address'
            }, {
              id: 'netmask',
              label: 'Netmask'
            }, {
              id: 'proto',
              label: 'Protocol'
            }, {
              id: 'broadcast',
              label: 'Broadcast'
            }]
          }
        }];
        agentOs === 'windows' && agentRequestsInventory.push({
          endpoint: `/syscollector/${agentID}/hotfixes`,
          loggerMessage: `Fetching hotfixes for agent ${agentID}`,
          table: {
            title: 'Windows updates',
            columns: [{
              id: 'hotfix',
              label: 'Update code'
            }]
          }
        });

        const requestInventory = async agentRequestInventory => {
          try {
            (0, _logger.log)('reporting:createReportsAgentsInventory', agentRequestInventory.loggerMessage, 'debug');
            const inventoryResponse = await context.wazuh.api.client.asCurrentUser.request('GET', agentRequestInventory.endpoint, {}, {
              apiHostID: apiId
            });
            const inventory = inventoryResponse && inventoryResponse.data && inventoryResponse.data.data && inventoryResponse.data.data.affected_items;

            if (inventory) {
              return { ...agentRequestInventory.table,
                items: agentRequestInventory.mapResponseItems ? inventory.map(agentRequestInventory.mapResponseItems) : inventory
              };
            }
          } catch (error) {
            (0, _logger.log)('reporting:createReportsAgentsInventory', error.message || error, 'debug');
          }
        };

        if (time) {
          await this.extendedInformation(context, printer, 'agents', 'syscollector', apiId, from, to, sanitizedFilters + ' AND rule.groups: "vulnerability-detector"', indexPatternTitle, agentID);
        } // Add inventory tables


        (await Promise.all(agentRequestsInventory.map(requestInventory))).filter(table => table).forEach(table => printer.addSimpleTable(table)); // Print the document

        await printer.print(context.wazuhEndpointParams.pathFilename);
        return response.ok({
          body: {
            success: true,
            message: `Report ${context.wazuhEndpointParams.filename} was created`
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:createReportsAgents', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5029, 500, response);
      }
    }, ({
      params: {
        agentID
      }
    }) => `wazuh-agent-inventory-${agentID}-${this.generateReportTimestamp()}.pdf`));
    (0, _defineProperty2.default)(this, "getReportByName", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:getReportByName', `Getting ${context.wazuhEndpointParams.pathFilename} report`, 'debug');

        const reportFileBuffer = _fs.default.readFileSync(context.wazuhEndpointParams.pathFilename);

        return response.ok({
          headers: {
            'Content-Type': 'application/pdf'
          },
          body: reportFileBuffer
        });
      } catch (error) {
        (0, _logger.log)('reporting:getReportByName', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5030, 500, response);
      }
    }, request => request.params.name));
    (0, _defineProperty2.default)(this, "deleteReportByName", this.checkReportsUserDirectoryIsValidRouteDecorator(async (context, request, response) => {
      try {
        (0, _logger.log)('reporting:deleteReportByName', `Deleting ${context.wazuhEndpointParams.pathFilename} report`, 'debug');

        _fs.default.unlinkSync(context.wazuhEndpointParams.pathFilename);

        (0, _logger.log)('reporting:deleteReportByName', `${context.wazuhEndpointParams.pathFilename} report was deleted`, 'info');
        return response.ok({
          body: {
            error: 0
          }
        });
      } catch (error) {
        (0, _logger.log)('reporting:deleteReportByName', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5032, 500, response);
      }
    }, request => request.params.name));
  }
  /**
   * This do format to filters
   * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
   * @param {String} searchBar search term
   */


  sanitizeKibanaFilters(filters, searchBar) {
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `Started to sanitize filters`, 'info');
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `filters: ${filters.length}, searchBar: ${searchBar}`, 'debug');
    let str = '';
    const agentsFilter = []; //separate agents filter

    filters = filters.filter(filter => {
      if (filter.meta.controlledBy === _constants.AUTHORIZED_AGENTS) {
        agentsFilter.push(filter);
        return false;
      }

      return filter;
    });
    const len = filters.length;

    for (let i = 0; i < len; i++) {
      const {
        negate,
        key,
        value,
        params,
        type
      } = filters[i].meta;
      str += `${negate ? 'NOT ' : ''}`;
      str += `${key}: `;
      str += `${type === 'range' ? `${params.gte}-${params.lt}` : type === 'phrases' ? '(' + params.join(" OR ") + ')' : type === 'exists' ? '*' : !!value ? value : (params || {}).query}`;
      str += `${i === len - 1 ? '' : ' AND '}`;
    }

    if (searchBar) {
      str += ` AND (${searchBar})`;
    }

    const agentsFilterStr = agentsFilter.map(filter => filter.meta.value).join(',');
    (0, _logger.log)('reporting:sanitizeKibanaFilters', `str: ${str}, agentsFilterStr: ${agentsFilterStr}`, 'debug');
    return [str, agentsFilterStr];
  }
  /**
   * This performs the rendering of given header
   * @param {String} printer section target
   * @param {String} section section target
   * @param {Object} tab tab target
   * @param {Boolean} isAgents is agents section
   * @param {String} apiId ID of API
   */


  async renderHeader(context, printer, section, tab, isAgents, apiId) {
    try {
      (0, _logger.log)('reporting:renderHeader', `section: ${section}, tab: ${tab}, isAgents: ${isAgents}, apiId: ${apiId}`, 'debug');

      if (section && typeof section === 'string') {
        if (!['agentConfig', 'groupConfig'].includes(section)) {
          printer.addContent({
            text: _wazuhModules.WAZUH_MODULES[tab].title + ' report',
            style: 'h1'
          });
        } else if (section === 'agentConfig') {
          printer.addContent({
            text: `Agent ${isAgents} configuration`,
            style: 'h1'
          });
        } else if (section === 'groupConfig') {
          printer.addContent({
            text: 'Agents in group',
            style: 'h1'
          });
        }

        printer.addNewLine();
      }

      if (isAgents && typeof isAgents === 'object') {
        await this.buildAgentsTable(context, printer, isAgents, apiId, section === 'groupConfig' ? tab : '');
      }

      if (isAgents && typeof isAgents === 'string') {
        const agentResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents`, {
          params: {
            agents_list: isAgents
          }
        }, {
          apiHostID: apiId
        });
        const agentData = agentResponse.data.data.affected_items[0];

        if (agentData && agentData.status !== _constants.API_NAME_AGENT_STATUS.ACTIVE) {
          printer.addContentWithNewLine({
            text: `Warning. Agent is ${(0, _wz_agent_status.agentStatusLabelByAgentStatus)(agentData.status).toLowerCase()}`,
            style: 'standard'
          });
        }

        await this.buildAgentsTable(context, printer, [isAgents], apiId);

        if (agentData && agentData.group) {
          const agentGroups = agentData.group.join(', ');
          printer.addContentWithNewLine({
            text: `Group${agentData.group.length > 1 ? 's' : ''}: ${agentGroups}`,
            style: 'standard'
          });
        }
      }

      if (_wazuhModules.WAZUH_MODULES[tab] && _wazuhModules.WAZUH_MODULES[tab].description) {
        printer.addContentWithNewLine({
          text: _wazuhModules.WAZUH_MODULES[tab].description,
          style: 'standard'
        });
      }
    } catch (error) {
      (0, _logger.log)('reporting:renderHeader', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This build the agents table
   * @param {Array<Strings>} ids ids of agents
   * @param {String} apiId API id
   */


  async buildAgentsTable(context, printer, agentIDs, apiId, groupID = '') {
    const dateFormat = await context.core.uiSettings.client.get('dateFormat');
    if ((!agentIDs || !agentIDs.length) && !groupID) return;
    (0, _logger.log)('reporting:buildAgentsTable', `${agentIDs.length} agents for API ${apiId}`, 'info');

    try {
      let agentsData = [];

      if (groupID) {
        let totalAgentsInGroup = null;

        do {
          const {
            data: {
              data: {
                affected_items,
                total_affected_items
              }
            }
          } = await context.wazuh.api.client.asCurrentUser.request('GET', `/groups/${groupID}/agents`, {
            params: {
              offset: agentsData.length,
              select: 'dateAdd,id,ip,lastKeepAlive,manager,name,os.name,os.version,version'
            }
          }, {
            apiHostID: apiId
          });
          !totalAgentsInGroup && (totalAgentsInGroup = total_affected_items);
          agentsData = [...agentsData, ...affected_items];
        } while (agentsData.length < totalAgentsInGroup);
      } else {
        for (const agentID of agentIDs) {
          try {
            const {
              data: {
                data: {
                  affected_items: [agent]
                }
              }
            } = await context.wazuh.api.client.asCurrentUser.request('GET', `/agents`, {
              params: {
                q: `id=${agentID}`,
                select: 'dateAdd,id,ip,lastKeepAlive,manager,name,os.name,os.version,version'
              }
            }, {
              apiHostID: apiId
            });
            agentsData.push(agent);
          } catch (error) {
            (0, _logger.log)('reporting:buildAgentsTable', `Skip agent due to: ${error.message || error}`, 'debug');
          }
        }
      }

      if (agentsData.length) {
        // Print a table with agent/s information
        printer.addSimpleTable({
          columns: [{
            id: 'id',
            label: 'ID'
          }, {
            id: 'name',
            label: 'Name'
          }, {
            id: 'ip',
            label: 'IP'
          }, {
            id: 'version',
            label: 'Version'
          }, {
            id: 'manager',
            label: 'Manager'
          }, {
            id: 'os',
            label: 'OS'
          }, {
            id: 'dateAdd',
            label: 'Registration date'
          }, {
            id: 'lastKeepAlive',
            label: 'Last keep alive'
          }],
          items: agentsData.map(agent => {
            return { ...agent,
              os: agent.os && agent.os.name && agent.os.version ? `${agent.os.name} ${agent.os.version}` : '',
              lastKeepAlive: (0, _moment.default)(agent.lastKeepAlive).format(dateFormat),
              dateAdd: (0, _moment.default)(agent.dateAdd).format(dateFormat)
            };
          })
        });
      } else if (!agentsData.length && groupID) {
        // For group reports when there is no agents in the group
        printer.addContent({
          text: 'There are no agents in this group.',
          style: {
            fontSize: 12,
            color: '#000'
          }
        });
      }
    } catch (error) {
      (0, _logger.log)('reporting:buildAgentsTable', error.message || error);
      return Promise.reject(error);
    }
  }
  /**
   * This load more information
   * @param {*} context Endpoint context
   * @param {*} printer printer instance
   * @param {String} section section target
   * @param {Object} tab tab target
   * @param {String} apiId ID of API
   * @param {Number} from Timestamp (ms) from
   * @param {Number} to Timestamp (ms) to
   * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
   * @param {String} pattern
   * @param {Object} agent agent target
   * @returns {Object} Extended information
   */


  async extendedInformation(context, printer, section, tab, apiId, from, to, filters, pattern = _constants.WAZUH_ALERTS_PATTERN, agent = null) {
    try {
      (0, _logger.log)('reporting:extendedInformation', `Section ${section} and tab ${tab}, API is ${apiId}. From ${from} to ${to}. Filters ${filters}. Index pattern ${pattern}`, 'info');

      if (section === 'agents' && !agent) {
        throw new Error('Reporting for specific agent needs an agent ID in order to work properly');
      }

      const agents = await context.wazuh.api.client.asCurrentUser.request('GET', '/agents', {
        params: {
          limit: 1
        }
      }, {
        apiHostID: apiId
      });
      const totalAgents = agents.data.data.total_affected_items;

      if (section === 'overview' && tab === 'vuls') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector metrics', 'debug');
        const vulnerabilitiesLevels = ['Low', 'Medium', 'High', 'Critical'];
        const vulnerabilitiesResponsesCount = (await Promise.all(vulnerabilitiesLevels.map(async vulnerabilitiesLevel => {
          try {
            const count = await VulnerabilityRequest.uniqueSeverityCount(context, from, to, vulnerabilitiesLevel, filters, pattern);
            return count ? `${count} of ${totalAgents} agents have ${vulnerabilitiesLevel.toLocaleLowerCase()} vulnerabilities.` : undefined;
          } catch (error) {}
        }))).filter(vulnerabilitiesResponse => vulnerabilitiesResponse);
        printer.addList({
          title: {
            text: 'Summary',
            style: 'h2'
          },
          list: vulnerabilitiesResponsesCount
        });
        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector top 3 agents by category', 'debug');
        const lowRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Low', filters, pattern);
        const mediumRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Medium', filters, pattern);
        const highRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'High', filters, pattern);
        const criticalRank = await VulnerabilityRequest.topAgentCount(context, from, to, 'Critical', filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding overview vulnerability detector top 3 agents by category', 'debug');

        if (criticalRank && criticalRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with critical severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, criticalRank, apiId);
          printer.addNewLine();
        }

        if (highRank && highRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with high severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, highRank, apiId);
          printer.addNewLine();
        }

        if (mediumRank && mediumRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with medium severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, mediumRank, apiId);
          printer.addNewLine();
        }

        if (lowRank && lowRank.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 agents with low severity vulnerabilities',
            style: 'h3'
          });
          await this.buildAgentsTable(context, printer, lowRank, apiId);
          printer.addNewLine();
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching overview vulnerability detector top 3 CVEs', 'debug');
        const cveRank = await VulnerabilityRequest.topCVECount(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding overview vulnerability detector top 3 CVEs', 'debug');

        if (cveRank && cveRank.length) {
          printer.addSimpleTable({
            title: {
              text: 'Top 3 CVE',
              style: 'h2'
            },
            columns: [{
              id: 'top',
              label: 'Top'
            }, {
              id: 'cve',
              label: 'CVE'
            }],
            items: cveRank.map(item => ({
              top: cveRank.indexOf(item) + 1,
              cve: item
            }))
          });
        }
      }

      if (section === 'overview' && tab === 'general') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 agents with level 15 alerts', 'debug');
        const level15Rank = await OverviewRequest.topLevel15(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding top 3 agents with level 15 alerts', 'debug');

        if (level15Rank.length) {
          printer.addContent({
            text: 'Top 3 agents with level 15 alerts',
            style: 'h2'
          });
          await this.buildAgentsTable(context, printer, level15Rank, apiId);
        }
      }

      if (section === 'overview' && tab === 'pm') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching most common rootkits', 'debug');
        const top5RootkitsRank = await RootcheckRequest.top5RootkitsDetected(context, from, to, filters, pattern);
        (0, _logger.log)('reporting:extendedInformation', 'Adding most common rootkits', 'debug');

        if (top5RootkitsRank && top5RootkitsRank.length) {
          printer.addContentWithNewLine({
            text: 'Most common rootkits found among your agents',
            style: 'h2'
          }).addContentWithNewLine({
            text: 'Rootkits are a set of software tools that enable an unauthorized user to gain control of a computer system without being detected.',
            style: 'standard'
          }).addSimpleTable({
            items: top5RootkitsRank.map(item => {
              return {
                top: top5RootkitsRank.indexOf(item) + 1,
                name: item
              };
            }),
            columns: [{
              id: 'top',
              label: 'Top'
            }, {
              id: 'name',
              label: 'Rootkit'
            }]
          });
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching hidden pids', 'debug');
        const hiddenPids = await RootcheckRequest.agentsWithHiddenPids(context, from, to, filters, pattern);
        hiddenPids && printer.addContent({
          text: `${hiddenPids} of ${totalAgents} agents have hidden processes`,
          style: 'h3'
        });
        !hiddenPids && printer.addContentWithNewLine({
          text: `No agents have hidden processes`,
          style: 'h3'
        });
        const hiddenPorts = await RootcheckRequest.agentsWithHiddenPorts(context, from, to, filters, pattern);
        hiddenPorts && printer.addContent({
          text: `${hiddenPorts} of ${totalAgents} agents have hidden ports`,
          style: 'h3'
        });
        !hiddenPorts && printer.addContent({
          text: `No agents have hidden ports`,
          style: 'h3'
        });
        printer.addNewLine();
      }

      if (['overview', 'agents'].includes(section) && tab === 'pci') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top PCI DSS requirements', 'debug');
        const topPciRequirements = await PCIRequest.topPCIRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common PCI DSS requirements alerts found',
          style: 'h2'
        });

        for (const item of topPciRequirements) {
          const rules = await PCIRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_pciRequirementsPdfmake.default[item]) {
            const content = typeof _pciRequirementsPdfmake.default[item] === 'string' ? {
              text: _pciRequirementsPdfmake.default[item],
              style: 'standard'
            } : _pciRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }
      }

      if (['overview', 'agents'].includes(section) && tab === 'tsc') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top TSC requirements', 'debug');
        const topTSCRequirements = await TSCRequest.topTSCRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common TSC requirements alerts found',
          style: 'h2'
        });

        for (const item of topTSCRequirements) {
          const rules = await TSCRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_tscRequirementsPdfmake.default[item]) {
            const content = typeof _tscRequirementsPdfmake.default[item] === 'string' ? {
              text: _tscRequirementsPdfmake.default[item],
              style: 'standard'
            } : _tscRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }
      }

      if (['overview', 'agents'].includes(section) && tab === 'gdpr') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top GDPR requirements', 'debug');
        const topGdprRequirements = await GDPRRequest.topGDPRRequirements(context, from, to, filters, pattern);
        printer.addContentWithNewLine({
          text: 'Most common GDPR requirements alerts found',
          style: 'h2'
        });

        for (const item of topGdprRequirements) {
          const rules = await GDPRRequest.getRulesByRequirement(context, from, to, filters, item, pattern);
          printer.addContentWithNewLine({
            text: `Requirement ${item}`,
            style: 'h3'
          });

          if (_gdprRequirementsPdfmake.default && _gdprRequirementsPdfmake.default[item]) {
            const content = typeof _gdprRequirementsPdfmake.default[item] === 'string' ? {
              text: _gdprRequirementsPdfmake.default[item],
              style: 'standard'
            } : _gdprRequirementsPdfmake.default[item];
            printer.addContentWithNewLine(content);
          }

          rules && rules.length && printer.addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: `Top rules for ${item} requirement`
          });
        }

        printer.addNewLine();
      }

      if (section === 'overview' && tab === 'audit') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching agents with high number of failed sudo commands', 'debug');
        const auditAgentsNonSuccess = await AuditRequest.getTop3AgentsSudoNonSuccessful(context, from, to, filters, pattern);

        if (auditAgentsNonSuccess && auditAgentsNonSuccess.length) {
          printer.addContent({
            text: 'Agents with high number of failed sudo commands',
            style: 'h2'
          });
          await this.buildAgentsTable(context, printer, auditAgentsNonSuccess, apiId);
        }

        const auditAgentsFailedSyscall = await AuditRequest.getTop3AgentsFailedSyscalls(context, from, to, filters, pattern);

        if (auditAgentsFailedSyscall && auditAgentsFailedSyscall.length) {
          printer.addSimpleTable({
            columns: [{
              id: 'agent',
              label: 'Agent ID'
            }, {
              id: 'syscall_id',
              label: 'Syscall ID'
            }, {
              id: 'syscall_syscall',
              label: 'Syscall'
            }],
            items: auditAgentsFailedSyscall.map(item => ({
              agent: item.agent,
              syscall_id: item.syscall.id,
              syscall_syscall: item.syscall.syscall
            })),
            title: {
              text: 'Most common failing syscalls',
              style: 'h2'
            }
          });
        }
      }

      if (section === 'overview' && tab === 'fim') {
        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 rules for FIM', 'debug');
        const rules = await SyscheckRequest.top3Rules(context, from, to, filters, pattern);

        if (rules && rules.length) {
          printer.addContentWithNewLine({
            text: 'Top 3 FIM rules',
            style: 'h2'
          }).addSimpleTable({
            columns: [{
              id: 'ruleID',
              label: 'Rule ID'
            }, {
              id: 'ruleDescription',
              label: 'Description'
            }],
            items: rules,
            title: {
              text: 'Top 3 rules that are generating most alerts.',
              style: 'standard'
            }
          });
        }

        (0, _logger.log)('reporting:extendedInformation', 'Fetching top 3 agents for FIM', 'debug');
        const agents = await SyscheckRequest.top3agents(context, from, to, filters, pattern);

        if (agents && agents.length) {
          printer.addContentWithNewLine({
            text: 'Agents with suspicious FIM activity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'Top 3 agents that have most FIM alerts from level 7 to level 15. Take care about them.',
            style: 'standard'
          });
          await this.buildAgentsTable(context, printer, agents, apiId);
        }
      }

      if (section === 'agents' && tab === 'audit') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching most common failed syscalls`, 'debug');
        const auditFailedSyscall = await AuditRequest.getTopFailedSyscalls(context, from, to, filters, pattern);
        auditFailedSyscall && auditFailedSyscall.length && printer.addSimpleTable({
          columns: [{
            id: 'id',
            label: 'id'
          }, {
            id: 'syscall',
            label: 'Syscall'
          }],
          items: auditFailedSyscall,
          title: 'Most common failing syscalls'
        });
      }

      if (section === 'agents' && tab === 'fim') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching syscheck database for agent ${agent}`, 'debug');
        const lastScanResponse = await context.wazuh.api.client.asCurrentUser.request('GET', `/syscheck/${agent}/last_scan`, {}, {
          apiHostID: apiId
        });

        if (lastScanResponse && lastScanResponse.data) {
          const lastScanData = lastScanResponse.data.data.affected_items[0];

          if (lastScanData.start && lastScanData.end) {
            printer.addContent({
              text: `Last file integrity monitoring scan was executed from ${lastScanData.start} to ${lastScanData.end}.`
            });
          } else if (lastScanData.start) {
            printer.addContent({
              text: `File integrity monitoring scan is currently in progress for this agent (started on ${lastScanData.start}).`
            });
          } else {
            printer.addContent({
              text: `File integrity monitoring scan is currently in progress for this agent.`
            });
          }

          printer.addNewLine();
        }

        (0, _logger.log)('reporting:extendedInformation', `Fetching last 10 deleted files for FIM`, 'debug');
        const lastTenDeleted = await SyscheckRequest.lastTenDeletedFiles(context, from, to, filters, pattern);
        lastTenDeleted && lastTenDeleted.length && printer.addSimpleTable({
          columns: [{
            id: 'path',
            label: 'Path'
          }, {
            id: 'date',
            label: 'Date'
          }],
          items: lastTenDeleted,
          title: 'Last 10 deleted files'
        });
        (0, _logger.log)('reporting:extendedInformation', `Fetching last 10 modified files`, 'debug');
        const lastTenModified = await SyscheckRequest.lastTenModifiedFiles(context, from, to, filters, pattern);
        lastTenModified && lastTenModified.length && printer.addSimpleTable({
          columns: [{
            id: 'path',
            label: 'Path'
          }, {
            id: 'date',
            label: 'Date'
          }],
          items: lastTenModified,
          title: 'Last 10 modified files'
        });
      }

      if (section === 'agents' && tab === 'syscollector') {
        (0, _logger.log)('reporting:extendedInformation', `Fetching hardware information for agent ${agent}`, 'debug');
        const requestsSyscollectorLists = [{
          endpoint: `/syscollector/${agent}/hardware`,
          loggerMessage: `Fetching Hardware information for agent ${agent}`,
          list: {
            title: {
              text: 'Hardware information',
              style: 'h2'
            }
          },
          mapResponse: hardware => [hardware.cpu && hardware.cpu.cores && `${hardware.cpu.cores} cores`, hardware.cpu && hardware.cpu.name, hardware.ram && hardware.ram.total && `${Number(hardware.ram.total / 1024 / 1024).toFixed(2)}GB RAM`]
        }, {
          endpoint: `/syscollector/${agent}/os`,
          loggerMessage: `Fetching OS information for agent ${agent}`,
          list: {
            title: {
              text: 'OS information',
              style: 'h2'
            }
          },
          mapResponse: osData => [osData.sysname, osData.version, osData.architecture, osData.release, osData.os && osData.os.name && osData.os.version && `${osData.os.name} ${osData.os.version}`]
        }];
        const syscollectorLists = await Promise.all(requestsSyscollectorLists.map(async requestSyscollector => {
          try {
            (0, _logger.log)('reporting:extendedInformation', requestSyscollector.loggerMessage, 'debug');
            const responseSyscollector = await context.wazuh.api.client.asCurrentUser.request('GET', requestSyscollector.endpoint, {}, {
              apiHostID: apiId
            });
            const [data] = responseSyscollector && responseSyscollector.data && responseSyscollector.data.data && responseSyscollector.data.data.affected_items || [];

            if (data) {
              return { ...requestSyscollector.list,
                list: requestSyscollector.mapResponse(data)
              };
            }
          } catch (error) {
            (0, _logger.log)('reporting:extendedInformation', error.message || error);
          }
        }));

        if (syscollectorLists) {
          syscollectorLists.filter(syscollectorList => syscollectorList).forEach(syscollectorList => printer.addList(syscollectorList));
        }

        const vulnerabilitiesRequests = ['Critical', 'High'];
        const vulnerabilitiesResponsesItems = (await Promise.all(vulnerabilitiesRequests.map(async vulnerabilitiesLevel => {
          try {
            (0, _logger.log)('reporting:extendedInformation', `Fetching top ${vulnerabilitiesLevel} packages`, 'debug');
            return await VulnerabilityRequest.topPackages(context, from, to, vulnerabilitiesLevel, filters, pattern);
          } catch (error) {
            (0, _logger.log)('reporting:extendedInformation', error.message || error);
          }
        }))).filter(vulnerabilitiesResponse => vulnerabilitiesResponse).flat();

        if (vulnerabilitiesResponsesItems && vulnerabilitiesResponsesItems.length) {
          printer.addSimpleTable({
            title: {
              text: 'Vulnerable packages found (last 24 hours)',
              style: 'h2'
            },
            columns: [{
              id: 'package',
              label: 'Package'
            }, {
              id: 'severity',
              label: 'Severity'
            }],
            items: vulnerabilitiesResponsesItems
          });
        }
      }

      if (section === 'agents' && tab === 'vuls') {
        const topCriticalPackages = await VulnerabilityRequest.topPackagesWithCVE(context, from, to, 'Critical', filters, pattern);

        if (topCriticalPackages && topCriticalPackages.length) {
          printer.addContentWithNewLine({
            text: 'Critical severity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'These vulnerabilties are critical, please review your agent. Click on each link to read more about each found vulnerability.',
            style: 'standard'
          });
          const customul = [];

          for (const critical of topCriticalPackages) {
            customul.push({
              text: critical.package,
              style: 'standard'
            });
            customul.push({
              ul: critical.references.map(item => ({
                text: item.substring(0, 80) + '...',
                link: item,
                color: '#1EA5C8'
              }))
            });
          }

          printer.addContentWithNewLine({
            ul: customul
          });
        }

        const topHighPackages = await VulnerabilityRequest.topPackagesWithCVE(context, from, to, 'High', filters, pattern);

        if (topHighPackages && topHighPackages.length) {
          printer.addContentWithNewLine({
            text: 'High severity',
            style: 'h2'
          });
          printer.addContentWithNewLine({
            text: 'Click on each link to read more about each found vulnerability.',
            style: 'standard'
          });
          const customul = [];

          for (const critical of topHighPackages) {
            customul.push({
              text: critical.package,
              style: 'standard'
            });
            customul.push({
              ul: critical.references.map(item => ({
                text: item,
                color: '#1EA5C8'
              }))
            });
          }

          customul && customul.length && printer.addContent({
            ul: customul
          });
          printer.addNewLine();
        }
      }

      return false;
    } catch (error) {
      (0, _logger.log)('reporting:extendedInformation', error.message || error);
      return Promise.reject(error);
    }
  }

  getConfigRows(data, labels) {
    (0, _logger.log)('reporting:getConfigRows', `Building configuration rows`, 'info');
    const result = [];

    for (let prop in data || []) {
      if (Array.isArray(data[prop])) {
        data[prop].forEach((x, idx) => {
          if (typeof x === 'object') data[prop][idx] = JSON.stringify(x);
        });
      }

      result.push([(labels || {})[prop] || _csvKeyEquivalence.KeyEquivalence[prop] || prop, data[prop] || '-']);
    }

    return result;
  }

  getConfigTables(data, section, tab, array = []) {
    (0, _logger.log)('reporting:getConfigTables', `Building configuration tables`, 'info');
    let plainData = {};
    const nestedData = [];
    const tableData = [];

    if (data.length === 1 && Array.isArray(data)) {
      tableData[section.config[tab].configuration] = data;
    } else {
      for (let key in data) {
        if (typeof data[key] !== 'object' && !Array.isArray(data[key]) || Array.isArray(data[key]) && typeof data[key][0] !== 'object') {
          plainData[key] = Array.isArray(data[key]) && typeof data[key][0] !== 'object' ? data[key].map(x => {
            return typeof x === 'object' ? JSON.stringify(x) : x + '\n';
          }) : data[key];
        } else if (Array.isArray(data[key]) && typeof data[key][0] === 'object') {
          tableData[key] = data[key];
        } else {
          if (section.isGroupConfig && ['pack', 'content'].includes(key)) {
            tableData[key] = [data[key]];
          } else {
            nestedData.push(data[key]);
          }
        }
      }
    }

    array.push({
      title: (section.options || {}).hideHeader ? '' : (section.tabs || [])[tab] || (section.isGroupConfig ? ((section.labels || [])[0] || [])[tab] : ''),
      columns: ['', ''],
      type: 'config',
      rows: this.getConfigRows(plainData, (section.labels || [])[0])
    });

    for (let key in tableData) {
      const columns = Object.keys(tableData[key][0]);
      columns.forEach((col, i) => {
        columns[i] = col[0].toUpperCase() + col.slice(1);
      });
      const rows = tableData[key].map(x => {
        let row = [];

        for (let key in x) {
          row.push(typeof x[key] !== 'object' ? x[key] : Array.isArray(x[key]) ? x[key].map(x => {
            return x + '\n';
          }) : JSON.stringify(x[key]));
        }

        while (row.length < columns.length) {
          row.push('-');
        }

        return row;
      });
      array.push({
        title: ((section.labels || [])[0] || [])[key] || '',
        type: 'table',
        columns,
        rows
      });
    }

    nestedData.forEach(nest => {
      this.getConfigTables(nest, section, tab + 1, array);
    });
    return array;
  }
  /**
   * Create a report for the modules
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {*} reports list or ErrorResponse
   */


  /**
   * Fetch the reports list
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Array<Object>} reports list or ErrorResponse
   */
  async getReports(context, request, response) {
    try {
      (0, _logger.log)('reporting:getReports', `Fetching created reports`, 'info');
      const {
        hashUsername
      } = await context.wazuh.security.getCurrentUser(request, context);
      (0, _filesystem.createDataDirectoryIfNotExists)();
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_DIRECTORY_PATH);
      (0, _filesystem.createDirectoryIfNotExists)(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH);

      const userReportsDirectoryPath = _path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername);

      (0, _filesystem.createDirectoryIfNotExists)(userReportsDirectoryPath);
      (0, _logger.log)('reporting:getReports', `Directory: ${userReportsDirectoryPath}`, 'debug');

      const sortReportsByDate = (a, b) => a.date < b.date ? 1 : a.date > b.date ? -1 : 0;

      const reports = _fs.default.readdirSync(userReportsDirectoryPath).map(file => {
        const stats = _fs.default.statSync(userReportsDirectoryPath + '/' + file); // Get the file creation time (bithtime). It returns the first value that is a truthy value of next file stats: birthtime, mtime, ctime and atime.
        // This solves some OSs can have the bithtimeMs equal to 0 and returns the date like 1970-01-01


        const birthTimeField = ['birthtime', 'mtime', 'ctime', 'atime'].find(time => stats[`${time}Ms`]);
        return {
          name: file,
          size: stats.size,
          date: stats[birthTimeField]
        };
      });

      (0, _logger.log)('reporting:getReports', `Using TimSort for sorting ${reports.length} items`, 'debug');
      TimSort.sort(reports, sortReportsByDate);
      (0, _logger.log)('reporting:getReports', `Total reports: ${reports.length}`, 'debug');
      return response.ok({
        body: {
          reports
        }
      });
    } catch (error) {
      (0, _logger.log)('reporting:getReports', error.message || error);
      return (0, _errorResponse.ErrorResponse)(error.message || error, 5031, 500, response);
    }
  }
  /**
   * Fetch specific report
   * @param {Object} context
   * @param {Object} request
   * @param {Object} response
   * @returns {Object} report or ErrorResponse
   */


  checkReportsUserDirectoryIsValidRouteDecorator(routeHandler, reportFileNameAccessor) {
    return async (context, request, response) => {
      try {
        const {
          username,
          hashUsername
        } = await context.wazuh.security.getCurrentUser(request, context);

        const userReportsDirectoryPath = _path.default.join(_constants.WAZUH_DATA_DOWNLOADS_REPORTS_DIRECTORY_PATH, hashUsername);

        const filename = reportFileNameAccessor(request);

        const pathFilename = _path.default.join(userReportsDirectoryPath, filename);

        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', `Checking the user ${username}(${hashUsername}) can do actions in the reports file: ${pathFilename}`, 'debug');

        if (!pathFilename.startsWith(userReportsDirectoryPath) || pathFilename.includes('../')) {
          (0, _logger.log)('security:reporting:checkReportsUserDirectoryIsValidRouteDecorator', `User ${username}(${hashUsername}) tried to access to a non user report file: ${pathFilename}`, 'warn');
          return response.badRequest({
            body: {
              message: '5040 - You shall not pass!'
            }
          });
        }

        ;
        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', 'Checking the user can do actions in the reports file', 'debug');
        return await routeHandler.bind(this)({ ...context,
          wazuhEndpointParams: {
            hashUsername,
            filename,
            pathFilename
          }
        }, request, response);
      } catch (error) {
        (0, _logger.log)('reporting:checkReportsUserDirectoryIsValidRouteDecorator', error.message || error);
        return (0, _errorResponse.ErrorResponse)(error.message || error, 5040, 500, response);
      }
    };
  }

  generateReportTimestamp() {
    return `${Date.now() / 1000 | 0}`;
  }

}

exports.WazuhReportingCtrl = WazuhReportingCtrl;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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