"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UpdateConfigurationFile = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _fs = _interopRequireDefault(require("fs"));

var _logger = require("./logger");

var _getConfiguration = require("./get-configuration");

var _constants = require("../../common/constants");

/*
 * Wazuh app - Module to update the configuration file
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
class UpdateConfigurationFile {
  constructor() {
    (0, _defineProperty2.default)(this, "formatValue", value => typeof value === 'string' ? isNaN(Number(value)) ? `'${value}'` : value : typeof value === 'object' ? JSON.stringify(value) : value);
    (0, _defineProperty2.default)(this, "formatValueCachedConfiguration", value => typeof value === 'string' ? isNaN(Number(value)) ? value : Number(value) : value);
    this.busy = false;
    this.file = _constants.WAZUH_DATA_CONFIG_APP_PATH;
  }
  /**
   * Add or replace specific setting from wazuh.yml
   * @param {String} key The setting name.
   * @param {String} value New value for the setting.
   * @param {Boolean} exists If true, it just replaces the value for that key.
   */


  updateLine(key, value, exists = false) {
    try {
      const data = _fs.default.readFileSync(this.file, {
        encoding: 'utf-8'
      });

      const re = new RegExp(`^${key}\\s{0,}:\\s{1,}.*`, 'gm');
      const formatedValue = this.formatValue(value);
      const result = exists ? data.replace(re, `${key}: ${formatedValue}`) : `${data}\n${key}: ${formatedValue}`;

      _fs.default.writeFileSync(this.file, result, 'utf8');

      (0, _logger.log)('update-configuration:updateLine', 'Updating line', 'debug');
      return true;
    } catch (error) {
      (0, _logger.log)('update-configuration:updateLine', error.message || error);
      throw error;
    }
  }

  /**
   * Updates wazuh.yml file. If it fails, it throws the error to the next function.
   * @param {Object} input
   */
  updateConfiguration(input) {
    try {
      if (this.busy) {
        throw new Error('Another process is updating the configuration file');
      }

      this.busy = true;
      const configuration = (0, _getConfiguration.getConfiguration)(true) || {};
      const {
        key,
        value
      } = (input || {}).body || {};
      this.updateLine(key, value, typeof configuration[key] !== 'undefined'); // Update the app configuration server-cached setting in memory with the new value

      configuration[key] = this.formatValueCachedConfiguration(value);
      this.busy = false;
      (0, _logger.log)('update-configuration:updateConfiguration', 'Updating configuration', 'debug');
      return {
        needRestart: _constants.WAZUH_CONFIGURATION_SETTINGS_NEED_RESTART.includes(key),
        needReload: _constants.WAZUH_CONFIGURATION_SETTINGS_NEED_RELOAD.includes(key),
        needHealtCheck: _constants.WAZUH_CONFIGURATION_SETTINGS_NEED_HEALTH_CHECK.includes(key)
      };
    } catch (error) {
      (0, _logger.log)('update-configuration:updateConfiguration', error.message || error);
      this.busy = false;
      throw error;
    }
  }

}

exports.UpdateConfigurationFile = UpdateConfigurationFile;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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