"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.formatLabelValuePair = formatLabelValuePair;
exports.formatSettingValueToFile = formatSettingValueToFile;
exports.getCategorySettingByTitle = getCategorySettingByTitle;
exports.getCustomizationSetting = getCustomizationSetting;
exports.getPluginSettingDescription = getPluginSettingDescription;
exports.getSettingDefaultValue = getSettingDefaultValue;
exports.getSettingsByCategories = getSettingsByCategories;
exports.getSettingsDefault = getSettingsDefault;
exports.getSettingsDefaultList = getSettingsDefaultList;
exports.groupSettingsByCategory = groupSettingsByCategory;

var _constants = require("../constants");

var _fileSize = require("./file-size");

/**
 * Look for a configuration category setting by its name
 * @param categoryTitle
 * @returns category settings
 */
function getCategorySettingByTitle(categoryTitle) {
  var _Object$entries$find;

  return (_Object$entries$find = Object.entries(_constants.PLUGIN_SETTINGS_CATEGORIES).find(([key, category]) => (category === null || category === void 0 ? void 0 : category.title) == categoryTitle)) === null || _Object$entries$find === void 0 ? void 0 : _Object$entries$find[1];
}
/**
 * Get the default value of the plugin setting.
 * @param setting setting key
 * @returns setting default value. It returns `defaultValueIfNotSet` or `defaultValue`.
 */


function getSettingDefaultValue(settingKey) {
  return typeof _constants.PLUGIN_SETTINGS[settingKey].defaultValueIfNotSet !== 'undefined' ? _constants.PLUGIN_SETTINGS[settingKey].defaultValueIfNotSet : _constants.PLUGIN_SETTINGS[settingKey].defaultValue;
}

;
/**
 * Get the default settings configuration. key-value pair
 * @returns an object with key-value pairs whose value is the default one
 */

function getSettingsDefault() {
  return Object.entries(_constants.PLUGIN_SETTINGS).reduce((accum, [pluginSettingID, pluginSettingConfiguration]) => ({ ...accum,
    [pluginSettingID]: pluginSettingConfiguration.defaultValue
  }), {});
}

;
/**
 * Get the settings grouped by category
 * @returns an object whose keys are the categories and its value is an array of setting of that category
 */

function getSettingsByCategories() {
  return Object.entries(_constants.PLUGIN_SETTINGS).reduce((accum, [pluginSettingID, pluginSettingConfiguration]) => ({ ...accum,
    [pluginSettingConfiguration.category]: [...(accum[pluginSettingConfiguration.category] || []), { ...pluginSettingConfiguration,
      key: pluginSettingID
    }]
  }), {});
}

;
/**
 * Get the plugin settings as an array
 * @returns an array of plugin setting denifitions including the key
 */

function getSettingsDefaultList() {
  return Object.entries(_constants.PLUGIN_SETTINGS).reduce((accum, [pluginSettingID, pluginSettingConfiguration]) => [...accum, { ...pluginSettingConfiguration,
    key: pluginSettingID
  }], []);
}

;
/**
 * Format the plugin setting value received in the backend to store in the plugin configuration file (.yml).
 * @param value plugin setting value sent to the endpoint
 * @returns valid value to .yml
 */

function formatSettingValueToFile(value) {
  const formatter = formatSettingValueToFileType[typeof value] || formatSettingValueToFileType.default;
  return formatter(value);
}

;
const formatSettingValueToFileType = {
  string: value => `"${value.replace(/"/, '\\"').replace(/\n/g, '\\n')}"`,
  // Escape the " character and new line
  object: value => JSON.stringify(value),
  default: value => value
};
/**
 * Group the settings by category
 * @param settings
 * @returns
 */

function groupSettingsByCategory(settings) {
  const settingsSortedByCategories = settings.sort((settingA, settingB) => {
    var _settingA$key, _settingA$key$localeC;

    return (_settingA$key = settingA.key) === null || _settingA$key === void 0 ? void 0 : (_settingA$key$localeC = _settingA$key.localeCompare) === null || _settingA$key$localeC === void 0 ? void 0 : _settingA$key$localeC.call(_settingA$key, settingB.key);
  }).reduce((accum, pluginSettingConfiguration) => ({ ...accum,
    [pluginSettingConfiguration.category]: [...(accum[pluginSettingConfiguration.category] || []), { ...pluginSettingConfiguration
    }]
  }), {});
  return Object.entries(settingsSortedByCategories).map(([category, settings]) => ({
    category,
    settings
  })).filter(categoryEntry => categoryEntry.settings.length);
}

;
/**
 * Get the plugin setting description composed.
 * @param options
 * @returns
 */

function getPluginSettingDescription({
  description,
  options
}) {
  var _options$file, _options$file2, _options$file2$recomm, _options$file3, _options$file4;

  return [description, ...(options !== null && options !== void 0 && options.select ? [`Allowed values: ${options.select.map(({
    text,
    value
  }) => formatLabelValuePair(text, value)).join(', ')}.`] : []), ...(options !== null && options !== void 0 && options.switch ? [`Allowed values: ${['enabled', 'disabled'].map(s => formatLabelValuePair(options.switch.values[s].label, options.switch.values[s].value)).join(', ')}.`] : []), ...(options !== null && options !== void 0 && options.number && 'min' in options.number ? [`Minimum value: ${options.number.min}.`] : []), ...(options !== null && options !== void 0 && options.number && 'max' in options.number ? [`Maximum value: ${options.number.max}.`] : []), // File extensions
  ...(options !== null && options !== void 0 && (_options$file = options.file) !== null && _options$file !== void 0 && _options$file.extensions ? [`Supported extensions: ${options.file.extensions.join(', ')}.`] : []), // File recommended dimensions
  ...(options !== null && options !== void 0 && (_options$file2 = options.file) !== null && _options$file2 !== void 0 && (_options$file2$recomm = _options$file2.recommended) !== null && _options$file2$recomm !== void 0 && _options$file2$recomm.dimensions ? [`Recommended dimensions: ${options.file.recommended.dimensions.width}x${options.file.recommended.dimensions.height}${options.file.recommended.dimensions.unit || ''}.`] : []), // File size
  ...(options !== null && options !== void 0 && (_options$file3 = options.file) !== null && _options$file3 !== void 0 && _options$file3.size && typeof options.file.size.minBytes !== 'undefined' ? [`Minimum file size: ${(0, _fileSize.formatBytes)(options.file.size.minBytes)}.`] : []), ...(options !== null && options !== void 0 && (_options$file4 = options.file) !== null && _options$file4 !== void 0 && _options$file4.size && typeof options.file.size.maxBytes !== 'undefined' ? [`Maximum file size: ${(0, _fileSize.formatBytes)(options.file.size.maxBytes)}.`] : []), // Multi line text
  ...(options !== null && options !== void 0 && options.maxRows && typeof options.maxRows !== 'undefined' ? [`Maximum amount of lines: ${options.maxRows}.`] : []), ...(options !== null && options !== void 0 && options.minRows && typeof options.minRows !== 'undefined' ? [`Minimum amount of lines: ${options.minRows}.`] : []), ...(options !== null && options !== void 0 && options.maxLength && typeof options.maxLength !== 'undefined' ? [`Maximum lines length is ${options.maxLength} characters.`] : [])].join(' ');
}

;
/**
 * Format the pair value-label to display the pair. If label and the string of value are equals, only displays the value, if not, displays both.
 * @param value
 * @param label
 * @returns
 */

function formatLabelValuePair(label, value) {
  return label !== `${value}` ? `${value} (${label})` : `${value}`;
}

;
/**
 * Get the configuration value if the customization is enabled.
 * @param configuration JSON object from `wazuh.yml`
 * @param settingKey key of the setting
 * @returns 
 */

function getCustomizationSetting(configuration, settingKey) {
  const isCustomizationEnabled = typeof configuration['customization.enabled'] === 'undefined' ? getSettingDefaultValue('customization.enabled') : configuration['customization.enabled'];
  const defaultValue = getSettingDefaultValue(settingKey);

  if (isCustomizationEnabled && settingKey.startsWith('customization') && settingKey !== 'customization.enabled') {
    return typeof configuration[settingKey] !== 'undefined' ? resolveEmptySetting(settingKey, configuration[settingKey]) : defaultValue;
  } else {
    return defaultValue;
  }

  ;
}

;
/**
 * Returns the default value if not set when the setting is an empty string
 * @param settingKey plugin setting
 * @param value value of the plugin setting
 * @returns 
 */

function resolveEmptySetting(settingKey, value) {
  return typeof value === 'string' && value.length === 0 && _constants.PLUGIN_SETTINGS[settingKey].defaultValueIfNotSet ? getSettingDefaultValue(settingKey) : value;
}

;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInNldHRpbmdzLnRzIl0sIm5hbWVzIjpbImdldENhdGVnb3J5U2V0dGluZ0J5VGl0bGUiLCJjYXRlZ29yeVRpdGxlIiwiT2JqZWN0IiwiZW50cmllcyIsIlBMVUdJTl9TRVRUSU5HU19DQVRFR09SSUVTIiwiZmluZCIsImtleSIsImNhdGVnb3J5IiwidGl0bGUiLCJnZXRTZXR0aW5nRGVmYXVsdFZhbHVlIiwic2V0dGluZ0tleSIsIlBMVUdJTl9TRVRUSU5HUyIsImRlZmF1bHRWYWx1ZUlmTm90U2V0IiwiZGVmYXVsdFZhbHVlIiwiZ2V0U2V0dGluZ3NEZWZhdWx0IiwicmVkdWNlIiwiYWNjdW0iLCJwbHVnaW5TZXR0aW5nSUQiLCJwbHVnaW5TZXR0aW5nQ29uZmlndXJhdGlvbiIsImdldFNldHRpbmdzQnlDYXRlZ29yaWVzIiwiZ2V0U2V0dGluZ3NEZWZhdWx0TGlzdCIsImZvcm1hdFNldHRpbmdWYWx1ZVRvRmlsZSIsInZhbHVlIiwiZm9ybWF0dGVyIiwiZm9ybWF0U2V0dGluZ1ZhbHVlVG9GaWxlVHlwZSIsImRlZmF1bHQiLCJzdHJpbmciLCJyZXBsYWNlIiwib2JqZWN0IiwiSlNPTiIsInN0cmluZ2lmeSIsImdyb3VwU2V0dGluZ3NCeUNhdGVnb3J5Iiwic2V0dGluZ3MiLCJzZXR0aW5nc1NvcnRlZEJ5Q2F0ZWdvcmllcyIsInNvcnQiLCJzZXR0aW5nQSIsInNldHRpbmdCIiwibG9jYWxlQ29tcGFyZSIsIm1hcCIsImZpbHRlciIsImNhdGVnb3J5RW50cnkiLCJsZW5ndGgiLCJnZXRQbHVnaW5TZXR0aW5nRGVzY3JpcHRpb24iLCJkZXNjcmlwdGlvbiIsIm9wdGlvbnMiLCJzZWxlY3QiLCJ0ZXh0IiwiZm9ybWF0TGFiZWxWYWx1ZVBhaXIiLCJqb2luIiwic3dpdGNoIiwicyIsInZhbHVlcyIsImxhYmVsIiwibnVtYmVyIiwibWluIiwibWF4IiwiZmlsZSIsImV4dGVuc2lvbnMiLCJyZWNvbW1lbmRlZCIsImRpbWVuc2lvbnMiLCJ3aWR0aCIsImhlaWdodCIsInVuaXQiLCJzaXplIiwibWluQnl0ZXMiLCJtYXhCeXRlcyIsIm1heFJvd3MiLCJtaW5Sb3dzIiwibWF4TGVuZ3RoIiwiZ2V0Q3VzdG9taXphdGlvblNldHRpbmciLCJjb25maWd1cmF0aW9uIiwiaXNDdXN0b21pemF0aW9uRW5hYmxlZCIsInN0YXJ0c1dpdGgiLCJyZXNvbHZlRW1wdHlTZXR0aW5nIl0sIm1hcHBpbmdzIjoiOzs7Ozs7Ozs7Ozs7Ozs7O0FBQUE7O0FBT0E7O0FBRUE7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNPLFNBQVNBLHlCQUFULENBQW1DQyxhQUFuQyxFQUErRDtBQUFBOztBQUNwRSxpQ0FBT0MsTUFBTSxDQUFDQyxPQUFQLENBQWVDLHFDQUFmLEVBQTJDQyxJQUEzQyxDQUFnRCxDQUFDLENBQUNDLEdBQUQsRUFBTUMsUUFBTixDQUFELEtBQXFCLENBQUFBLFFBQVEsU0FBUixJQUFBQSxRQUFRLFdBQVIsWUFBQUEsUUFBUSxDQUFFQyxLQUFWLEtBQW1CUCxhQUF4RixDQUFQLHlEQUFPLHFCQUF5RyxDQUF6RyxDQUFQO0FBQ0Q7QUFFRDtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7QUFDTyxTQUFTUSxzQkFBVCxDQUFnQ0MsVUFBaEMsRUFBeUQ7QUFDL0QsU0FBTyxPQUFPQywyQkFBZ0JELFVBQWhCLEVBQTRCRSxvQkFBbkMsS0FBNEQsV0FBNUQsR0FDSkQsMkJBQWdCRCxVQUFoQixFQUE0QkUsb0JBRHhCLEdBRUpELDJCQUFnQkQsVUFBaEIsRUFBNEJHLFlBRi9CO0FBR0E7O0FBQUE7QUFFRDtBQUNBO0FBQ0E7QUFDQTs7QUFDTyxTQUFTQyxrQkFBVCxHQUF3RTtBQUM5RSxTQUFPWixNQUFNLENBQUNDLE9BQVAsQ0FBZVEsMEJBQWYsRUFBZ0NJLE1BQWhDLENBQXVDLENBQUNDLEtBQUQsRUFBUSxDQUFDQyxlQUFELEVBQWtCQywwQkFBbEIsQ0FBUixNQUEyRCxFQUN4RyxHQUFHRixLQURxRztBQUV4RyxLQUFDQyxlQUFELEdBQW1CQywwQkFBMEIsQ0FBQ0w7QUFGMEQsR0FBM0QsQ0FBdkMsRUFHSCxFQUhHLENBQVA7QUFJQTs7QUFBQTtBQUVEO0FBQ0E7QUFDQTtBQUNBOztBQUNPLFNBQVNNLHVCQUFULEdBQXdFO0FBQzlFLFNBQU9qQixNQUFNLENBQUNDLE9BQVAsQ0FBZVEsMEJBQWYsRUFBZ0NJLE1BQWhDLENBQXVDLENBQUNDLEtBQUQsRUFBUSxDQUFDQyxlQUFELEVBQWtCQywwQkFBbEIsQ0FBUixNQUEyRCxFQUN4RyxHQUFHRixLQURxRztBQUV4RyxLQUFDRSwwQkFBMEIsQ0FBQ1gsUUFBNUIsR0FBdUMsQ0FBQyxJQUFJUyxLQUFLLENBQUNFLDBCQUEwQixDQUFDWCxRQUE1QixDQUFMLElBQThDLEVBQWxELENBQUQsRUFBd0QsRUFBRSxHQUFHVywwQkFBTDtBQUFpQ1osTUFBQUEsR0FBRyxFQUFFVztBQUF0QyxLQUF4RDtBQUZpRSxHQUEzRCxDQUF2QyxFQUdILEVBSEcsQ0FBUDtBQUlBOztBQUFBO0FBRUQ7QUFDQTtBQUNBO0FBQ0E7O0FBQ08sU0FBU0csc0JBQVQsR0FBMkQ7QUFDakUsU0FBT2xCLE1BQU0sQ0FBQ0MsT0FBUCxDQUFlUSwwQkFBZixFQUFnQ0ksTUFBaEMsQ0FBdUMsQ0FBQ0MsS0FBRCxFQUFRLENBQUNDLGVBQUQsRUFBa0JDLDBCQUFsQixDQUFSLEtBQTJELENBQ3hHLEdBQUdGLEtBRHFHLEVBRXhHLEVBQUUsR0FBR0UsMEJBQUw7QUFBaUNaLElBQUFBLEdBQUcsRUFBRVc7QUFBdEMsR0FGd0csQ0FBbEcsRUFHSCxFQUhHLENBQVA7QUFJQTs7QUFBQTtBQUVEO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBQ08sU0FBU0ksd0JBQVQsQ0FBa0NDLEtBQWxDLEVBQThDO0FBQ3BELFFBQU1DLFNBQVMsR0FBR0MsNEJBQTRCLENBQUMsT0FBT0YsS0FBUixDQUE1QixJQUE4Q0UsNEJBQTRCLENBQUNDLE9BQTdGO0FBQ0EsU0FBT0YsU0FBUyxDQUFDRCxLQUFELENBQWhCO0FBQ0E7O0FBQUE7QUFFRCxNQUFNRSw0QkFBNEIsR0FBRztBQUNwQ0UsRUFBQUEsTUFBTSxFQUFHSixLQUFELElBQTRCLElBQUdBLEtBQUssQ0FBQ0ssT0FBTixDQUFjLEdBQWQsRUFBa0IsS0FBbEIsRUFBeUJBLE9BQXpCLENBQWlDLEtBQWpDLEVBQXVDLEtBQXZDLENBQThDLEdBRGpEO0FBQ3FEO0FBQ3pGQyxFQUFBQSxNQUFNLEVBQUdOLEtBQUQsSUFBd0JPLElBQUksQ0FBQ0MsU0FBTCxDQUFlUixLQUFmLENBRkk7QUFHcENHLEVBQUFBLE9BQU8sRUFBR0gsS0FBRCxJQUFxQkE7QUFITSxDQUFyQztBQU1BO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBQ08sU0FBU1MsdUJBQVQsQ0FBaUNDLFFBQWpDLEVBQW1FO0FBQ3pFLFFBQU1DLDBCQUEwQixHQUFHRCxRQUFRLENBQ3pDRSxJQURpQyxDQUM1QixDQUFDQyxRQUFELEVBQVdDLFFBQVg7QUFBQTs7QUFBQSw0QkFBd0JELFFBQVEsQ0FBQzdCLEdBQWpDLDJFQUF3QixjQUFjK0IsYUFBdEMsMERBQXdCLDBDQUE4QkQsUUFBUSxDQUFDOUIsR0FBdkMsQ0FBeEI7QUFBQSxHQUQ0QixFQUVqQ1MsTUFGaUMsQ0FFMUIsQ0FBQ0MsS0FBRCxFQUFRRSwwQkFBUixNQUF3QyxFQUMvQyxHQUFHRixLQUQ0QztBQUUvQyxLQUFDRSwwQkFBMEIsQ0FBQ1gsUUFBNUIsR0FBdUMsQ0FDdEMsSUFBSVMsS0FBSyxDQUFDRSwwQkFBMEIsQ0FBQ1gsUUFBNUIsQ0FBTCxJQUE4QyxFQUFsRCxDQURzQyxFQUV0QyxFQUFFLEdBQUdXO0FBQUwsS0FGc0M7QUFGUSxHQUF4QyxDQUYwQixFQVE5QixFQVI4QixDQUFuQztBQVVBLFNBQU9oQixNQUFNLENBQUNDLE9BQVAsQ0FBZThCLDBCQUFmLEVBQ0xLLEdBREssQ0FDRCxDQUFDLENBQUMvQixRQUFELEVBQVd5QixRQUFYLENBQUQsTUFBMkI7QUFBRXpCLElBQUFBLFFBQUY7QUFBWXlCLElBQUFBO0FBQVosR0FBM0IsQ0FEQyxFQUVMTyxNQUZLLENBRUVDLGFBQWEsSUFBSUEsYUFBYSxDQUFDUixRQUFkLENBQXVCUyxNQUYxQyxDQUFQO0FBR0E7O0FBQUE7QUFFRDtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUNRLFNBQVNDLDJCQUFULENBQXFDO0FBQUNDLEVBQUFBLFdBQUQ7QUFBY0MsRUFBQUE7QUFBZCxDQUFyQyxFQUFvRjtBQUFBOztBQUMzRixTQUFPLENBQ05ELFdBRE0sRUFFTixJQUFJQyxPQUFPLFNBQVAsSUFBQUEsT0FBTyxXQUFQLElBQUFBLE9BQU8sQ0FBRUMsTUFBVCxHQUFrQixDQUFFLG1CQUFrQkQsT0FBTyxDQUFDQyxNQUFSLENBQWVQLEdBQWYsQ0FBbUIsQ0FBQztBQUFDUSxJQUFBQSxJQUFEO0FBQU94QixJQUFBQTtBQUFQLEdBQUQsS0FBbUJ5QixvQkFBb0IsQ0FBQ0QsSUFBRCxFQUFPeEIsS0FBUCxDQUExRCxFQUF5RTBCLElBQXpFLENBQThFLElBQTlFLENBQW9GLEdBQXhHLENBQWxCLEdBQWdJLEVBQXBJLENBRk0sRUFHTixJQUFJSixPQUFPLFNBQVAsSUFBQUEsT0FBTyxXQUFQLElBQUFBLE9BQU8sQ0FBRUssTUFBVCxHQUFrQixDQUFFLG1CQUFrQixDQUFDLFNBQUQsRUFBWSxVQUFaLEVBQXdCWCxHQUF4QixDQUE0QlksQ0FBQyxJQUFJSCxvQkFBb0IsQ0FBQ0gsT0FBTyxDQUFDSyxNQUFSLENBQWVFLE1BQWYsQ0FBc0JELENBQXRCLEVBQXlCRSxLQUExQixFQUFpQ1IsT0FBTyxDQUFDSyxNQUFSLENBQWVFLE1BQWYsQ0FBc0JELENBQXRCLEVBQXlCNUIsS0FBMUQsQ0FBckQsRUFBdUgwQixJQUF2SCxDQUE0SCxJQUE1SCxDQUFrSSxHQUF0SixDQUFsQixHQUE4SyxFQUFsTCxDQUhNLEVBSU4sSUFBSUosT0FBTyxTQUFQLElBQUFBLE9BQU8sV0FBUCxJQUFBQSxPQUFPLENBQUVTLE1BQVQsSUFBbUIsU0FBU1QsT0FBTyxDQUFDUyxNQUFwQyxHQUE2QyxDQUFFLGtCQUFpQlQsT0FBTyxDQUFDUyxNQUFSLENBQWVDLEdBQUksR0FBdEMsQ0FBN0MsR0FBeUYsRUFBN0YsQ0FKTSxFQUtOLElBQUlWLE9BQU8sU0FBUCxJQUFBQSxPQUFPLFdBQVAsSUFBQUEsT0FBTyxDQUFFUyxNQUFULElBQW1CLFNBQVNULE9BQU8sQ0FBQ1MsTUFBcEMsR0FBNkMsQ0FBRSxrQkFBaUJULE9BQU8sQ0FBQ1MsTUFBUixDQUFlRSxHQUFJLEdBQXRDLENBQTdDLEdBQXlGLEVBQTdGLENBTE0sRUFNTjtBQUNBLE1BQUlYLE9BQU8sU0FBUCxJQUFBQSxPQUFPLFdBQVAscUJBQUFBLE9BQU8sQ0FBRVksSUFBVCx3REFBZUMsVUFBZixHQUE0QixDQUFFLHlCQUF3QmIsT0FBTyxDQUFDWSxJQUFSLENBQWFDLFVBQWIsQ0FBd0JULElBQXhCLENBQTZCLElBQTdCLENBQW1DLEdBQTdELENBQTVCLEdBQStGLEVBQW5HLENBUE0sRUFRTjtBQUNBLE1BQUlKLE9BQU8sU0FBUCxJQUFBQSxPQUFPLFdBQVAsc0JBQUFBLE9BQU8sQ0FBRVksSUFBVCxtRkFBZUUsV0FBZix3RUFBNEJDLFVBQTVCLEdBQXlDLENBQUUsMkJBQTBCZixPQUFPLENBQUNZLElBQVIsQ0FBYUUsV0FBYixDQUF5QkMsVUFBekIsQ0FBb0NDLEtBQU0sSUFBR2hCLE9BQU8sQ0FBQ1ksSUFBUixDQUFhRSxXQUFiLENBQXlCQyxVQUF6QixDQUFvQ0UsTUFBTyxHQUFFakIsT0FBTyxDQUFDWSxJQUFSLENBQWFFLFdBQWIsQ0FBeUJDLFVBQXpCLENBQW9DRyxJQUFwQyxJQUE0QyxFQUFHLEdBQXJLLENBQXpDLEdBQW9OLEVBQXhOLENBVE0sRUFVTjtBQUNBLE1BQUtsQixPQUFPLFNBQVAsSUFBQUEsT0FBTyxXQUFQLHNCQUFBQSxPQUFPLENBQUVZLElBQVQsMERBQWVPLElBQWYsSUFBdUIsT0FBT25CLE9BQU8sQ0FBQ1ksSUFBUixDQUFhTyxJQUFiLENBQWtCQyxRQUF6QixLQUFzQyxXQUE5RCxHQUE2RSxDQUFFLHNCQUFxQiwyQkFBWXBCLE9BQU8sQ0FBQ1ksSUFBUixDQUFhTyxJQUFiLENBQWtCQyxRQUE5QixDQUF3QyxHQUEvRCxDQUE3RSxHQUFrSixFQUF0SixDQVhNLEVBWU4sSUFBS3BCLE9BQU8sU0FBUCxJQUFBQSxPQUFPLFdBQVAsc0JBQUFBLE9BQU8sQ0FBRVksSUFBVCwwREFBZU8sSUFBZixJQUF1QixPQUFPbkIsT0FBTyxDQUFDWSxJQUFSLENBQWFPLElBQWIsQ0FBa0JFLFFBQXpCLEtBQXNDLFdBQTlELEdBQTZFLENBQUUsc0JBQXFCLDJCQUFZckIsT0FBTyxDQUFDWSxJQUFSLENBQWFPLElBQWIsQ0FBa0JFLFFBQTlCLENBQXdDLEdBQS9ELENBQTdFLEdBQWtKLEVBQXRKLENBWk0sRUFhTjtBQUNBLE1BQUtyQixPQUFPLFNBQVAsSUFBQUEsT0FBTyxXQUFQLElBQUFBLE9BQU8sQ0FBRXNCLE9BQVQsSUFBb0IsT0FBT3RCLE9BQU8sQ0FBQ3NCLE9BQWYsS0FBMkIsV0FBL0MsR0FBNkQsQ0FBRSw0QkFBMkJ0QixPQUFPLENBQUNzQixPQUFRLEdBQTdDLENBQTdELEdBQWdILEVBQXJILENBZE0sRUFlTixJQUFLdEIsT0FBTyxTQUFQLElBQUFBLE9BQU8sV0FBUCxJQUFBQSxPQUFPLENBQUV1QixPQUFULElBQW9CLE9BQU92QixPQUFPLENBQUN1QixPQUFmLEtBQTJCLFdBQS9DLEdBQTZELENBQUUsNEJBQTJCdkIsT0FBTyxDQUFDdUIsT0FBUSxHQUE3QyxDQUE3RCxHQUFnSCxFQUFySCxDQWZNLEVBZ0JOLElBQUt2QixPQUFPLFNBQVAsSUFBQUEsT0FBTyxXQUFQLElBQUFBLE9BQU8sQ0FBRXdCLFNBQVQsSUFBc0IsT0FBT3hCLE9BQU8sQ0FBQ3dCLFNBQWYsS0FBNkIsV0FBbkQsR0FBaUUsQ0FBRSwyQkFBMEJ4QixPQUFPLENBQUN3QixTQUFVLGNBQTlDLENBQWpFLEdBQWdJLEVBQXJJLENBaEJNLEVBaUJMcEIsSUFqQkssQ0FpQkEsR0FqQkEsQ0FBUDtBQWtCQTs7QUFBQTtBQUVEO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7QUFDTyxTQUFTRCxvQkFBVCxDQUE4QkssS0FBOUIsRUFBcUM5QixLQUFyQyxFQUEyQztBQUNqRCxTQUFPOEIsS0FBSyxLQUFNLEdBQUU5QixLQUFNLEVBQW5CLEdBQ0gsR0FBRUEsS0FBTSxLQUFJOEIsS0FBTSxHQURmLEdBRUgsR0FBRTlCLEtBQU0sRUFGWjtBQUdBOztBQUFBO0FBRUQ7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOztBQUNPLFNBQVMrQyx1QkFBVCxDQUFpQ0MsYUFBakMsRUFBdUU1RCxVQUF2RSxFQUFnRztBQUNyRyxRQUFNNkQsc0JBQXNCLEdBQUcsT0FBT0QsYUFBYSxDQUFDLHVCQUFELENBQXBCLEtBQWtELFdBQWxELEdBQzNCN0Qsc0JBQXNCLENBQUMsdUJBQUQsQ0FESyxHQUUzQjZELGFBQWEsQ0FBQyx1QkFBRCxDQUZqQjtBQUdBLFFBQU16RCxZQUFZLEdBQUdKLHNCQUFzQixDQUFDQyxVQUFELENBQTNDOztBQUVELE1BQUs2RCxzQkFBc0IsSUFBSTdELFVBQVUsQ0FBQzhELFVBQVgsQ0FBc0IsZUFBdEIsQ0FBMUIsSUFBb0U5RCxVQUFVLEtBQUssdUJBQXhGLEVBQWdIO0FBQy9HLFdBQVEsT0FBTzRELGFBQWEsQ0FBQzVELFVBQUQsQ0FBcEIsS0FBcUMsV0FBckMsR0FBbUQrRCxtQkFBbUIsQ0FBQy9ELFVBQUQsRUFBYTRELGFBQWEsQ0FBQzVELFVBQUQsQ0FBMUIsQ0FBdEUsR0FBZ0hHLFlBQXhIO0FBQ0EsR0FGRCxNQUVLO0FBQ0osV0FBT0EsWUFBUDtBQUNBOztBQUFBO0FBQ0Q7O0FBQUE7QUFFRDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7O0FBQ0EsU0FBUzRELG1CQUFULENBQTZCL0QsVUFBN0IsRUFBaURZLEtBQWpELEVBQWlFO0FBQ2hFLFNBQU8sT0FBT0EsS0FBUCxLQUFpQixRQUFqQixJQUE2QkEsS0FBSyxDQUFDbUIsTUFBTixLQUFpQixDQUE5QyxJQUFtRDlCLDJCQUFnQkQsVUFBaEIsRUFBNEJFLG9CQUEvRSxHQUNKSCxzQkFBc0IsQ0FBQ0MsVUFBRCxDQURsQixHQUVKWSxLQUZIO0FBR0E7O0FBQUEiLCJzb3VyY2VzQ29udGVudCI6WyJpbXBvcnQge1xuICBQTFVHSU5fU0VUVElOR1MsXG4gIFBMVUdJTl9TRVRUSU5HU19DQVRFR09SSUVTLFxuICBUUGx1Z2luU2V0dGluZyxcbiAgVFBsdWdpblNldHRpbmdLZXksXG4gIFRQbHVnaW5TZXR0aW5nV2l0aEtleVxufSBmcm9tICcuLi9jb25zdGFudHMnO1xuaW1wb3J0IHsgZm9ybWF0Qnl0ZXMgfSBmcm9tICcuL2ZpbGUtc2l6ZSc7XG5cbi8qKlxuICogTG9vayBmb3IgYSBjb25maWd1cmF0aW9uIGNhdGVnb3J5IHNldHRpbmcgYnkgaXRzIG5hbWVcbiAqIEBwYXJhbSBjYXRlZ29yeVRpdGxlXG4gKiBAcmV0dXJucyBjYXRlZ29yeSBzZXR0aW5nc1xuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0Q2F0ZWdvcnlTZXR0aW5nQnlUaXRsZShjYXRlZ29yeVRpdGxlOiBzdHJpbmcpOiBhbnkge1xuICByZXR1cm4gT2JqZWN0LmVudHJpZXMoUExVR0lOX1NFVFRJTkdTX0NBVEVHT1JJRVMpLmZpbmQoKFtrZXksIGNhdGVnb3J5XSkgPT4gY2F0ZWdvcnk/LnRpdGxlID09IGNhdGVnb3J5VGl0bGUpPy5bMV07XG59XG5cbi8qKlxuICogR2V0IHRoZSBkZWZhdWx0IHZhbHVlIG9mIHRoZSBwbHVnaW4gc2V0dGluZy5cbiAqIEBwYXJhbSBzZXR0aW5nIHNldHRpbmcga2V5XG4gKiBAcmV0dXJucyBzZXR0aW5nIGRlZmF1bHQgdmFsdWUuIEl0IHJldHVybnMgYGRlZmF1bHRWYWx1ZUlmTm90U2V0YCBvciBgZGVmYXVsdFZhbHVlYC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldFNldHRpbmdEZWZhdWx0VmFsdWUoc2V0dGluZ0tleTogc3RyaW5nKTogYW55IHtcblx0cmV0dXJuIHR5cGVvZiBQTFVHSU5fU0VUVElOR1Nbc2V0dGluZ0tleV0uZGVmYXVsdFZhbHVlSWZOb3RTZXQgIT09ICd1bmRlZmluZWQnXG5cdFx0PyBQTFVHSU5fU0VUVElOR1Nbc2V0dGluZ0tleV0uZGVmYXVsdFZhbHVlSWZOb3RTZXRcblx0XHQ6IFBMVUdJTl9TRVRUSU5HU1tzZXR0aW5nS2V5XS5kZWZhdWx0VmFsdWU7XG59O1xuXG4vKipcbiAqIEdldCB0aGUgZGVmYXVsdCBzZXR0aW5ncyBjb25maWd1cmF0aW9uLiBrZXktdmFsdWUgcGFpclxuICogQHJldHVybnMgYW4gb2JqZWN0IHdpdGgga2V5LXZhbHVlIHBhaXJzIHdob3NlIHZhbHVlIGlzIHRoZSBkZWZhdWx0IG9uZVxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0U2V0dGluZ3NEZWZhdWx0KCkgOiB7W2tleSBpbiBUUGx1Z2luU2V0dGluZ0tleV06IHVua25vd259ICAge1xuXHRyZXR1cm4gT2JqZWN0LmVudHJpZXMoUExVR0lOX1NFVFRJTkdTKS5yZWR1Y2UoKGFjY3VtLCBbcGx1Z2luU2V0dGluZ0lELCBwbHVnaW5TZXR0aW5nQ29uZmlndXJhdGlvbl0pID0+ICh7XG5cdFx0Li4uYWNjdW0sXG5cdFx0W3BsdWdpblNldHRpbmdJRF06IHBsdWdpblNldHRpbmdDb25maWd1cmF0aW9uLmRlZmF1bHRWYWx1ZVxuXHR9KSwge30pO1xufTtcblxuLyoqXG4gKiBHZXQgdGhlIHNldHRpbmdzIGdyb3VwZWQgYnkgY2F0ZWdvcnlcbiAqIEByZXR1cm5zIGFuIG9iamVjdCB3aG9zZSBrZXlzIGFyZSB0aGUgY2F0ZWdvcmllcyBhbmQgaXRzIHZhbHVlIGlzIGFuIGFycmF5IG9mIHNldHRpbmcgb2YgdGhhdCBjYXRlZ29yeVxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0U2V0dGluZ3NCeUNhdGVnb3JpZXMoKSA6IHtba2V5OiBzdHJpbmddOiBUUGx1Z2luU2V0dGluZ1tdfSAge1xuXHRyZXR1cm4gT2JqZWN0LmVudHJpZXMoUExVR0lOX1NFVFRJTkdTKS5yZWR1Y2UoKGFjY3VtLCBbcGx1Z2luU2V0dGluZ0lELCBwbHVnaW5TZXR0aW5nQ29uZmlndXJhdGlvbl0pID0+ICh7XG5cdFx0Li4uYWNjdW0sXG5cdFx0W3BsdWdpblNldHRpbmdDb25maWd1cmF0aW9uLmNhdGVnb3J5XTogWy4uLihhY2N1bVtwbHVnaW5TZXR0aW5nQ29uZmlndXJhdGlvbi5jYXRlZ29yeV0gfHwgW10pLCB7IC4uLnBsdWdpblNldHRpbmdDb25maWd1cmF0aW9uLCBrZXk6IHBsdWdpblNldHRpbmdJRCB9XVxuXHR9KSwge30pO1xufTtcblxuLyoqXG4gKiBHZXQgdGhlIHBsdWdpbiBzZXR0aW5ncyBhcyBhbiBhcnJheVxuICogQHJldHVybnMgYW4gYXJyYXkgb2YgcGx1Z2luIHNldHRpbmcgZGVuaWZpdGlvbnMgaW5jbHVkaW5nIHRoZSBrZXlcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGdldFNldHRpbmdzRGVmYXVsdExpc3QoKTogVFBsdWdpblNldHRpbmdXaXRoS2V5W10ge1xuXHRyZXR1cm4gT2JqZWN0LmVudHJpZXMoUExVR0lOX1NFVFRJTkdTKS5yZWR1Y2UoKGFjY3VtLCBbcGx1Z2luU2V0dGluZ0lELCBwbHVnaW5TZXR0aW5nQ29uZmlndXJhdGlvbl0pID0+IChbXG5cdFx0Li4uYWNjdW0sXG5cdFx0eyAuLi5wbHVnaW5TZXR0aW5nQ29uZmlndXJhdGlvbiwga2V5OiBwbHVnaW5TZXR0aW5nSUQgfVxuXHRdKSwgW10pO1xufTtcblxuLyoqXG4gKiBGb3JtYXQgdGhlIHBsdWdpbiBzZXR0aW5nIHZhbHVlIHJlY2VpdmVkIGluIHRoZSBiYWNrZW5kIHRvIHN0b3JlIGluIHRoZSBwbHVnaW4gY29uZmlndXJhdGlvbiBmaWxlICgueW1sKS5cbiAqIEBwYXJhbSB2YWx1ZSBwbHVnaW4gc2V0dGluZyB2YWx1ZSBzZW50IHRvIHRoZSBlbmRwb2ludFxuICogQHJldHVybnMgdmFsaWQgdmFsdWUgdG8gLnltbFxuICovXG5leHBvcnQgZnVuY3Rpb24gZm9ybWF0U2V0dGluZ1ZhbHVlVG9GaWxlKHZhbHVlOiBhbnkpIHtcblx0Y29uc3QgZm9ybWF0dGVyID0gZm9ybWF0U2V0dGluZ1ZhbHVlVG9GaWxlVHlwZVt0eXBlb2YgdmFsdWVdIHx8IGZvcm1hdFNldHRpbmdWYWx1ZVRvRmlsZVR5cGUuZGVmYXVsdDtcblx0cmV0dXJuIGZvcm1hdHRlcih2YWx1ZSk7XG59O1xuXG5jb25zdCBmb3JtYXRTZXR0aW5nVmFsdWVUb0ZpbGVUeXBlID0ge1xuXHRzdHJpbmc6ICh2YWx1ZTogc3RyaW5nKTogc3RyaW5nID0+IGBcIiR7dmFsdWUucmVwbGFjZSgvXCIvLCdcXFxcXCInKS5yZXBsYWNlKC9cXG4vZywnXFxcXG4nKX1cImAsIC8vIEVzY2FwZSB0aGUgXCIgY2hhcmFjdGVyIGFuZCBuZXcgbGluZVxuXHRvYmplY3Q6ICh2YWx1ZTogYW55KTogc3RyaW5nID0+IEpTT04uc3RyaW5naWZ5KHZhbHVlKSxcblx0ZGVmYXVsdDogKHZhbHVlOiBhbnkpOiBhbnkgPT4gdmFsdWVcbn07XG5cbi8qKlxuICogR3JvdXAgdGhlIHNldHRpbmdzIGJ5IGNhdGVnb3J5XG4gKiBAcGFyYW0gc2V0dGluZ3NcbiAqIEByZXR1cm5zXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBncm91cFNldHRpbmdzQnlDYXRlZ29yeShzZXR0aW5nczogVFBsdWdpblNldHRpbmdXaXRoS2V5W10pe1xuXHRjb25zdCBzZXR0aW5nc1NvcnRlZEJ5Q2F0ZWdvcmllcyA9IHNldHRpbmdzXG5cdFx0LnNvcnQoKHNldHRpbmdBLCBzZXR0aW5nQikgPT4gc2V0dGluZ0Eua2V5Py5sb2NhbGVDb21wYXJlPy4oc2V0dGluZ0Iua2V5KSlcblx0XHQucmVkdWNlKChhY2N1bSwgcGx1Z2luU2V0dGluZ0NvbmZpZ3VyYXRpb24pID0+ICh7XG5cdFx0XHQuLi5hY2N1bSxcblx0XHRcdFtwbHVnaW5TZXR0aW5nQ29uZmlndXJhdGlvbi5jYXRlZ29yeV06IFtcblx0XHRcdFx0Li4uKGFjY3VtW3BsdWdpblNldHRpbmdDb25maWd1cmF0aW9uLmNhdGVnb3J5XSB8fCBbXSksXG5cdFx0XHRcdHsgLi4ucGx1Z2luU2V0dGluZ0NvbmZpZ3VyYXRpb24gfVxuXHRcdFx0XVxuXHRcdH0pLCB7fSk7XG5cblx0cmV0dXJuIE9iamVjdC5lbnRyaWVzKHNldHRpbmdzU29ydGVkQnlDYXRlZ29yaWVzKVxuXHRcdC5tYXAoKFtjYXRlZ29yeSwgc2V0dGluZ3NdKSA9PiAoeyBjYXRlZ29yeSwgc2V0dGluZ3MgfSkpXG5cdFx0LmZpbHRlcihjYXRlZ29yeUVudHJ5ID0+IGNhdGVnb3J5RW50cnkuc2V0dGluZ3MubGVuZ3RoKTtcbn07XG5cbi8qKlxuICogR2V0IHRoZSBwbHVnaW4gc2V0dGluZyBkZXNjcmlwdGlvbiBjb21wb3NlZC5cbiAqIEBwYXJhbSBvcHRpb25zXG4gKiBAcmV0dXJuc1xuICovXG4gZXhwb3J0IGZ1bmN0aW9uIGdldFBsdWdpblNldHRpbmdEZXNjcmlwdGlvbih7ZGVzY3JpcHRpb24sIG9wdGlvbnN9OiBUUGx1Z2luU2V0dGluZyk6IHN0cmluZ3tcblx0cmV0dXJuIFtcblx0XHRkZXNjcmlwdGlvbixcblx0XHQuLi4ob3B0aW9ucz8uc2VsZWN0ID8gW2BBbGxvd2VkIHZhbHVlczogJHtvcHRpb25zLnNlbGVjdC5tYXAoKHt0ZXh0LCB2YWx1ZX0pID0+IGZvcm1hdExhYmVsVmFsdWVQYWlyKHRleHQsIHZhbHVlKSkuam9pbignLCAnKX0uYF0gOiBbXSksXG5cdFx0Li4uKG9wdGlvbnM/LnN3aXRjaCA/IFtgQWxsb3dlZCB2YWx1ZXM6ICR7WydlbmFibGVkJywgJ2Rpc2FibGVkJ10ubWFwKHMgPT4gZm9ybWF0TGFiZWxWYWx1ZVBhaXIob3B0aW9ucy5zd2l0Y2gudmFsdWVzW3NdLmxhYmVsLCBvcHRpb25zLnN3aXRjaC52YWx1ZXNbc10udmFsdWUpKS5qb2luKCcsICcpfS5gXSA6IFtdKSxcblx0XHQuLi4ob3B0aW9ucz8ubnVtYmVyICYmICdtaW4nIGluIG9wdGlvbnMubnVtYmVyID8gW2BNaW5pbXVtIHZhbHVlOiAke29wdGlvbnMubnVtYmVyLm1pbn0uYF0gOiBbXSksXG5cdFx0Li4uKG9wdGlvbnM/Lm51bWJlciAmJiAnbWF4JyBpbiBvcHRpb25zLm51bWJlciA/IFtgTWF4aW11bSB2YWx1ZTogJHtvcHRpb25zLm51bWJlci5tYXh9LmBdIDogW10pLFxuXHRcdC8vIEZpbGUgZXh0ZW5zaW9uc1xuXHRcdC4uLihvcHRpb25zPy5maWxlPy5leHRlbnNpb25zID8gW2BTdXBwb3J0ZWQgZXh0ZW5zaW9uczogJHtvcHRpb25zLmZpbGUuZXh0ZW5zaW9ucy5qb2luKCcsICcpfS5gXSA6IFtdKSxcblx0XHQvLyBGaWxlIHJlY29tbWVuZGVkIGRpbWVuc2lvbnNcblx0XHQuLi4ob3B0aW9ucz8uZmlsZT8ucmVjb21tZW5kZWQ/LmRpbWVuc2lvbnMgPyBbYFJlY29tbWVuZGVkIGRpbWVuc2lvbnM6ICR7b3B0aW9ucy5maWxlLnJlY29tbWVuZGVkLmRpbWVuc2lvbnMud2lkdGh9eCR7b3B0aW9ucy5maWxlLnJlY29tbWVuZGVkLmRpbWVuc2lvbnMuaGVpZ2h0fSR7b3B0aW9ucy5maWxlLnJlY29tbWVuZGVkLmRpbWVuc2lvbnMudW5pdCB8fCAnJ30uYF0gOiBbXSksXG5cdFx0Ly8gRmlsZSBzaXplXG5cdFx0Li4uKChvcHRpb25zPy5maWxlPy5zaXplICYmIHR5cGVvZiBvcHRpb25zLmZpbGUuc2l6ZS5taW5CeXRlcyAhPT0gJ3VuZGVmaW5lZCcpID8gW2BNaW5pbXVtIGZpbGUgc2l6ZTogJHtmb3JtYXRCeXRlcyhvcHRpb25zLmZpbGUuc2l6ZS5taW5CeXRlcyl9LmBdIDogW10pLFxuXHRcdC4uLigob3B0aW9ucz8uZmlsZT8uc2l6ZSAmJiB0eXBlb2Ygb3B0aW9ucy5maWxlLnNpemUubWF4Qnl0ZXMgIT09ICd1bmRlZmluZWQnKSA/IFtgTWF4aW11bSBmaWxlIHNpemU6ICR7Zm9ybWF0Qnl0ZXMob3B0aW9ucy5maWxlLnNpemUubWF4Qnl0ZXMpfS5gXSA6IFtdKSxcblx0XHQvLyBNdWx0aSBsaW5lIHRleHRcblx0XHQuLi4oKG9wdGlvbnM/Lm1heFJvd3MgJiYgdHlwZW9mIG9wdGlvbnMubWF4Um93cyAhPT0gJ3VuZGVmaW5lZCcgPyBbYE1heGltdW0gYW1vdW50IG9mIGxpbmVzOiAke29wdGlvbnMubWF4Um93c30uYF0gOiBbXSkpLFxuXHRcdC4uLigob3B0aW9ucz8ubWluUm93cyAmJiB0eXBlb2Ygb3B0aW9ucy5taW5Sb3dzICE9PSAndW5kZWZpbmVkJyA/IFtgTWluaW11bSBhbW91bnQgb2YgbGluZXM6ICR7b3B0aW9ucy5taW5Sb3dzfS5gXSA6IFtdKSksXG5cdFx0Li4uKChvcHRpb25zPy5tYXhMZW5ndGggJiYgdHlwZW9mIG9wdGlvbnMubWF4TGVuZ3RoICE9PSAndW5kZWZpbmVkJyA/IFtgTWF4aW11bSBsaW5lcyBsZW5ndGggaXMgJHtvcHRpb25zLm1heExlbmd0aH0gY2hhcmFjdGVycy5gXSA6IFtdKSksXG5cdF0uam9pbignICcpO1xufTtcblxuLyoqXG4gKiBGb3JtYXQgdGhlIHBhaXIgdmFsdWUtbGFiZWwgdG8gZGlzcGxheSB0aGUgcGFpci4gSWYgbGFiZWwgYW5kIHRoZSBzdHJpbmcgb2YgdmFsdWUgYXJlIGVxdWFscywgb25seSBkaXNwbGF5cyB0aGUgdmFsdWUsIGlmIG5vdCwgZGlzcGxheXMgYm90aC5cbiAqIEBwYXJhbSB2YWx1ZVxuICogQHBhcmFtIGxhYmVsXG4gKiBAcmV0dXJuc1xuICovXG5leHBvcnQgZnVuY3Rpb24gZm9ybWF0TGFiZWxWYWx1ZVBhaXIobGFiZWwsIHZhbHVlKXtcblx0cmV0dXJuIGxhYmVsICE9PSBgJHt2YWx1ZX1gXG5cdFx0PyBgJHt2YWx1ZX0gKCR7bGFiZWx9KWBcblx0XHQ6IGAke3ZhbHVlfWBcbn07XG5cbi8qKlxuICogR2V0IHRoZSBjb25maWd1cmF0aW9uIHZhbHVlIGlmIHRoZSBjdXN0b21pemF0aW9uIGlzIGVuYWJsZWQuXG4gKiBAcGFyYW0gY29uZmlndXJhdGlvbiBKU09OIG9iamVjdCBmcm9tIGB3YXp1aC55bWxgXG4gKiBAcGFyYW0gc2V0dGluZ0tleSBrZXkgb2YgdGhlIHNldHRpbmdcbiAqIEByZXR1cm5zIFxuICovXG5leHBvcnQgZnVuY3Rpb24gZ2V0Q3VzdG9taXphdGlvblNldHRpbmcoY29uZmlndXJhdGlvbjoge1trZXk6IHN0cmluZ106IGFueSB9LCBzZXR0aW5nS2V5OiBzdHJpbmcpOiBhbnkge1xuICBjb25zdCBpc0N1c3RvbWl6YXRpb25FbmFibGVkID0gdHlwZW9mIGNvbmZpZ3VyYXRpb25bJ2N1c3RvbWl6YXRpb24uZW5hYmxlZCddID09PSAndW5kZWZpbmVkJ1xuICAgID8gZ2V0U2V0dGluZ0RlZmF1bHRWYWx1ZSgnY3VzdG9taXphdGlvbi5lbmFibGVkJylcbiAgICA6IGNvbmZpZ3VyYXRpb25bJ2N1c3RvbWl6YXRpb24uZW5hYmxlZCddO1xuICBjb25zdCBkZWZhdWx0VmFsdWUgPSBnZXRTZXR0aW5nRGVmYXVsdFZhbHVlKHNldHRpbmdLZXkpO1xuXG5cdGlmICggaXNDdXN0b21pemF0aW9uRW5hYmxlZCAmJiBzZXR0aW5nS2V5LnN0YXJ0c1dpdGgoJ2N1c3RvbWl6YXRpb24nKSAmJiBzZXR0aW5nS2V5ICE9PSAnY3VzdG9taXphdGlvbi5lbmFibGVkJyl7XG5cdFx0cmV0dXJuICh0eXBlb2YgY29uZmlndXJhdGlvbltzZXR0aW5nS2V5XSAhPT0gJ3VuZGVmaW5lZCcgPyByZXNvbHZlRW1wdHlTZXR0aW5nKHNldHRpbmdLZXksIGNvbmZpZ3VyYXRpb25bc2V0dGluZ0tleV0pIDogZGVmYXVsdFZhbHVlKTtcblx0fWVsc2V7XG5cdFx0cmV0dXJuIGRlZmF1bHRWYWx1ZTtcblx0fTtcbn07XG5cbi8qKlxuICogUmV0dXJucyB0aGUgZGVmYXVsdCB2YWx1ZSBpZiBub3Qgc2V0IHdoZW4gdGhlIHNldHRpbmcgaXMgYW4gZW1wdHkgc3RyaW5nXG4gKiBAcGFyYW0gc2V0dGluZ0tleSBwbHVnaW4gc2V0dGluZ1xuICogQHBhcmFtIHZhbHVlIHZhbHVlIG9mIHRoZSBwbHVnaW4gc2V0dGluZ1xuICogQHJldHVybnMgXG4gKi9cbmZ1bmN0aW9uIHJlc29sdmVFbXB0eVNldHRpbmcoc2V0dGluZ0tleTogc3RyaW5nLCB2YWx1ZSA6IHVua25vd24pe1xuXHRyZXR1cm4gdHlwZW9mIHZhbHVlID09PSAnc3RyaW5nJyAmJiB2YWx1ZS5sZW5ndGggPT09IDAgJiYgUExVR0lOX1NFVFRJTkdTW3NldHRpbmdLZXldLmRlZmF1bHRWYWx1ZUlmTm90U2V0XG5cdFx0PyBnZXRTZXR0aW5nRGVmYXVsdFZhbHVlKHNldHRpbmdLZXkpXG5cdFx0OiB2YWx1ZTtcbn07XG4iXX0=