"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.topPCIRequirements = exports.getRulesByRequirement = void 0;

var _baseQuery = require("./base-query");

var _settings = require("../../../common/services/settings");

/*
 * Wazuh app - Specific methods to fetch Wazuh PCI DSS data from Elasticsearch
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */

/**
 * Returns top 5 PCI DSS requirements
 * @param {*} context Endpoint context
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Array<String>}
 */
const topPCIRequirements = async (context, gte, lte, filters, allowedAgentsFilter, pattern = (0, _settings.getSettingDefaultValue)('pattern')) => {
  if (filters.includes('rule.pci_dss: exists')) {
    filters = filters.replace('AND rule.pci_dss: exists', '');
  }

  ;

  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte, allowedAgentsFilter));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'rule.pci_dss',
          size: 5,
          order: {
            _count: 'desc'
          }
        }
      }
    });
    base.query.bool.must.push({
      exists: {
        field: 'rule.pci_dss'
      }
    });
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.map(item => item.key).sort((a, b) => {
      const a_split = a.split('.');
      const b_split = b.split('.');
      if (parseInt(a_split[0]) > parseInt(b_split[0])) return 1;else if (parseInt(a_split[0]) < parseInt(b_split[0])) return -1;else {
        if (parseInt(a_split[1]) > parseInt(b_split[1])) return 1;else if (parseInt(a_split[1]) < parseInt(b_split[1])) return -1;else {
          if (parseInt(a_split[2]) > parseInt(b_split[2])) return 1;else if (parseInt(a_split[2]) < parseInt(b_split[2])) return -1;
        }
      }
    });
  } catch (error) {
    return Promise.reject(error);
  }
};
/**
 * Returns top 3 rules for specific PCI DSS requirement
 * @param {*} context Endpoint context
 * @param {Number} gte Timestamp (ms) from
 * @param {Number} lte Timestamp (ms) to
 * @param {String} requirement PCI DSS requirement. E.g: '10.2.3'
 * @param {String} filters E.g: cluster.name: wazuh AND rule.groups: vulnerability
 * @returns {Array<String>}
 */


exports.topPCIRequirements = topPCIRequirements;

const getRulesByRequirement = async (context, gte, lte, filters, allowedAgentsFilter, requirement, pattern = (0, _settings.getSettingDefaultValue)('pattern')) => {
  if (filters.includes('rule.pci_dss: exists')) {
    filters = filters.replace('AND rule.pci_dss: exists', '');
  }

  ;

  try {
    const base = {};
    Object.assign(base, (0, _baseQuery.Base)(pattern, filters, gte, lte, allowedAgentsFilter));
    Object.assign(base.aggs, {
      '2': {
        terms: {
          field: 'rule.description',
          size: 3,
          order: {
            _count: 'desc'
          }
        },
        aggs: {
          '3': {
            terms: {
              field: 'rule.id',
              size: 1,
              order: {
                _count: 'desc'
              }
            }
          }
        }
      }
    });
    base.query.bool.must[0].query_string.query = base.query.bool.must[0].query_string.query + ' AND rule.pci_dss: "' + requirement + '"';
    const response = await context.core.elasticsearch.client.asCurrentUser.search({
      index: pattern,
      body: base
    });
    const {
      buckets
    } = response.body.aggregations['2'];
    return buckets.reduce((accum, bucket) => {
      if (!bucket || !bucket['3'] || !bucket['3'].buckets || !bucket['3'].buckets[0] || !bucket['3'].buckets[0].key || !bucket.key) {
        return accum;
      }

      ;
      accum.push({
        ruleID: bucket['3'].buckets[0].key,
        ruleDescription: bucket.key
      });
      return accum;
    }, []);
  } catch (error) {
    return Promise.reject(error);
  }
};

exports.getRulesByRequirement = getRulesByRequirement;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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