"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SaveDocument = void 0;

var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));

var _getConfiguration = require("../../lib/get-configuration");

var _logger = require("../../lib/logger");

var _indexDate = require("../../lib/index-date");

var _constants = require("../../../common/constants");

var _tryCatchForIndexPermissionError = require("../tryCatchForIndexPermissionError");

class SaveDocument {
  constructor(context) {
    (0, _defineProperty2.default)(this, "context", void 0);
    (0, _defineProperty2.default)(this, "esClientInternalUser", void 0);
    (0, _defineProperty2.default)(this, "logPath", 'cron-scheduler|SaveDocument');
    this.context = context;
    this.esClientInternalUser = context.core.elasticsearch.client.asInternalUser;
  }

  async save(doc, indexConfig) {
    const {
      name,
      creation,
      mapping,
      shards,
      replicas
    } = indexConfig;
    const index = this.addIndexPrefix(name);
    const indexCreation = `${index}-${(0, _indexDate.indexDate)(creation)}`;

    try {
      await this.checkIndexAndCreateIfNotExists(indexCreation, shards, replicas);
      const createDocumentObject = this.createDocument(doc, indexCreation, mapping);
      const response = await this.esClientInternalUser.bulk(createDocumentObject);
      (0, _logger.log)(this.logPath, `Response of create new document ${JSON.stringify(response)}`, 'debug'); // await this.checkIndexPatternAndCreateIfNotExists(index);
    } catch (error) {
      if (error.status === 403) throw {
        error: 403,
        message: `Authorization Exception in the index "${index}"`
      };
      if (error.status === 409) throw {
        error: 409,
        message: `Duplicate index-pattern: ${index}`
      };
      throw error;
    }
  }

  async checkIndexAndCreateIfNotExists(index, shards, replicas) {
    try {
      await (0, _tryCatchForIndexPermissionError.tryCatchForIndexPermissionError)(index)(async () => {
        const exists = await this.esClientInternalUser.indices.exists({
          index
        });
        (0, _logger.log)(this.logPath, `Index '${index}' exists? ${exists.body}`, 'debug');

        if (!exists.body) {
          const response = await this.esClientInternalUser.indices.create({
            index,
            body: {
              settings: {
                index: {
                  number_of_shards: shards !== null && shards !== void 0 ? shards : _constants.WAZUH_STATISTICS_DEFAULT_INDICES_SHARDS,
                  number_of_replicas: replicas !== null && replicas !== void 0 ? replicas : _constants.WAZUH_STATISTICS_DEFAULT_INDICES_REPLICAS
                }
              }
            }
          });
          (0, _logger.log)(this.logPath, `Status of create a new index: ${JSON.stringify(response)}`, 'debug');
        }
      })();
    } catch (error) {
      this.checkDuplicateIndexError(error);
    }
  }

  checkDuplicateIndexError(error) {
    const {
      type
    } = ((error || {}).body || {}).error || {};
    if (!['resource_already_exists_exception'].includes(type)) throw error;
  }

  createDocument(doc, index, mapping) {
    const createDocumentObject = {
      index,
      body: doc.map(item => `{"index": { "_index": "${index}" }}\n${JSON.stringify({ ...this.buildData(item, mapping),
        timestamp: new Date(Date.now()).toISOString()
      })}\n`).join('')
    };
    (0, _logger.log)(this.logPath, `Document object: ${JSON.stringify(createDocumentObject)}`, 'debug');
    return createDocumentObject;
  }

  buildData(item, mapping) {
    const getItemArray = (array, index) => {
      return JSON.stringify(array[index || 0]);
    };

    const getValue = (key, item) => {
      const keys = key.split('.');

      if (keys.length === 1) {
        if (key.match(/\[.*\]/)) {
          return getItemArray(item[key.replace(/\[.*\]/, '')], key.match(/\[(.*)\]/)[1]);
        }

        return JSON.stringify(item[key]);
      }

      return getValue(keys.slice(1).join('.'), item[keys[0]]);
    };

    if (mapping) {
      let data;
      data = mapping.replace(/\${([a-z|A-Z|0-9|\.\-\_\[.*\]]+)}/gi, (...key) => getValue(key[1], item));
      return JSON.parse(data);
    }

    if (typeof item.data === 'object') {
      return item.data;
    }

    return {
      data: item.data
    };
  }

  addIndexPrefix(index) {
    const configFile = (0, _getConfiguration.getConfiguration)();
    const prefix = configFile['cron.prefix'] || 'wazuh';
    return `${prefix}-${index}`;
  }

}

exports.SaveDocument = SaveDocument;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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