"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.jobInitializeRun = jobInitializeRun;

var _logger = require("../../lib/logger");

var _package = _interopRequireDefault(require("../../../package.json"));

var _kibanaTemplate = require("../../integration-files/kibana-template");

var _getConfiguration = require("../../lib/get-configuration");

var _os = require("os");

var _fs = _interopRequireDefault(require("fs"));

var _constants = require("../../../common/constants");

var _filesystem = require("../../lib/filesystem");

var _lodash = _interopRequireDefault(require("lodash"));

var _settings = require("../../../common/services/settings");

/*
 * Wazuh app - Module for app initialization
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
function jobInitializeRun(context) {
  const PLUGIN_PLATFORM_INDEX = context.server.config.kibana.index;
  (0, _logger.log)('initialize', `${_constants.PLUGIN_PLATFORM_NAME} index: ${PLUGIN_PLATFORM_INDEX}`, 'info');
  (0, _logger.log)('initialize', `App revision: ${_package.default.revision}`, 'info');
  let configurationFile = {};
  let pattern = null; // Read config from package.json and wazuh.yml

  try {
    configurationFile = (0, _getConfiguration.getConfiguration)();
    pattern = configurationFile && typeof configurationFile.pattern !== 'undefined' ? configurationFile.pattern : (0, _settings.getSettingDefaultValue)('pattern');
  } catch (error) {
    (0, _logger.log)('initialize', error.message || error);
    context.wazuh.logger.error('Something went wrong while reading the configuration.' + (error.message || error));
  }

  try {
    // RAM in MB
    const ram = Math.ceil((0, _os.totalmem)() / 1024 / 1024);
    (0, _logger.log)('initialize', `Total RAM: ${ram}MB`, 'info');
  } catch (error) {
    (0, _logger.log)('initialize', `Could not check total RAM due to: ${error.message || error}`);
  } // Save Wazuh App setup


  const saveConfiguration = async (hosts = {}) => {
    try {
      const commonDate = new Date().toISOString();
      const configuration = {
        name: 'Wazuh App',
        'app-version': _package.default.version,
        revision: _package.default.revision,
        installationDate: commonDate,
        lastRestart: commonDate,
        hosts
      };

      try {
        (0, _filesystem.createDataDirectoryIfNotExists)();
        (0, _filesystem.createDataDirectoryIfNotExists)('config');
        (0, _logger.log)('initialize:saveConfiguration', `Saving configuration in registry file: ${JSON.stringify(configuration)}`, 'debug');
        await _fs.default.writeFileSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH, JSON.stringify(configuration), 'utf8');
        (0, _logger.log)('initialize:saveConfiguration', 'Wazuh configuration registry saved.', 'debug');
      } catch (error) {
        (0, _logger.log)('initialize:saveConfiguration', error.message || error);
        context.wazuh.logger.error('Could not create Wazuh configuration registry');
      }
    } catch (error) {
      (0, _logger.log)('initialize:saveConfiguration', error.message || error);
      context.wazuh.logger.error('Error creating wazuh-registry.json file.');
    }
  };
  /**
   * Checks if the .wazuh-registry.json file exists:
   * - yes: check the plugin version and revision match the values stored in the registry file.
   *  If not, then it migrates the data rebuilding the registry file.
   * - no: create the file with empty hosts
   */


  const checkWazuhRegistry = async () => {
    (0, _logger.log)('initialize:checkwazuhRegistry', 'Checking wazuh-registry.json file.', 'debug');

    if (!_fs.default.existsSync(_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH)) {
      throw new Error(`The data directory is missing in the ${_constants.PLUGIN_PLATFORM_NAME} root instalation. Create the directory in ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH} and give it the required permissions (sudo mkdir ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH};sudo chown -R ${_constants.PLUGIN_PLATFORM_INSTALLATION_USER}:${_constants.PLUGIN_PLATFORM_INSTALLATION_USER_GROUP} ${_constants.WAZUH_DATA_PLUGIN_PLATFORM_BASE_ABSOLUTE_PATH}). After restart the ${_constants.PLUGIN_PLATFORM_NAME} service.`);
    }

    ;

    if (!_fs.default.existsSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH)) {
      (0, _logger.log)('initialize:checkwazuhRegistry', 'wazuh-registry.json file does not exist. Initializing configuration.', 'debug'); // Create the app registry file for the very first time

      await saveConfiguration();
    } else {
      // If this function fails, it throws an exception
      const source = JSON.parse(_fs.default.readFileSync(_constants.WAZUH_DATA_CONFIG_REGISTRY_PATH, 'utf8')); // Check if the stored revision differs from the package.json revision

      const isUpgradedApp = _package.default.revision !== source.revision || _package.default.version !== source['app-version']; // Rebuild the registry file if revision or version fields are differents

      if (isUpgradedApp) {
        (0, _logger.log)('initialize:checkwazuhRegistry', 'Wazuh app revision or version changed, regenerating wazuh-registry.json.', 'info'); // Rebuild the registry file `wazuh-registry.json`
        // Get the supported extensions for the installed plugin

        const supportedDefaultExtensionsConfiguration = Object.entries((0, _settings.getSettingsDefault)()).filter(([setting]) => setting.startsWith('extensions.')).map(([setting, settingValue]) => {
          return [setting.split('.')[1], settingValue];
        }); // Get the supported extensions by ID

        const supportedDefaultExtensionsNames = supportedDefaultExtensionsConfiguration.map(([setting]) => setting); // Generate the hosts data, migrating the extensions.
        // Keep the supported and existent extensions for the installed plugin with the configurated value
        // Add the extensions with default values that didn't exist in the previous configuration
        // Remove the unsupported extensions for the installed plugin

        const registryHostsData = Object.entries(source.hosts).reduce((accum, [hostID, hostData]) => {
          accum[hostID] = hostData;

          if (accum[hostID].extensions) {
            // Migrate extensions to those supported by the installed plugin
            const defaultHostExtentionsConfiguration = Object.fromEntries(supportedDefaultExtensionsConfiguration); // Select of current configuration the extension IDs that are supported in the installed plugin

            const currentHostConfiguration = _lodash.default.pick(accum[hostID].extensions, supportedDefaultExtensionsNames); // Merge the default extensions configuration with the configuration stored in the registry file


            accum[hostID].extensions = _lodash.default.merge(defaultHostExtentionsConfiguration, currentHostConfiguration);
          }

          return accum;
        }, {}); // Rebuild the registry file with the migrated host data (extensions are migrated to these supported by the installed plugin).

        await saveConfiguration(registryHostsData);
        (0, _logger.log)('initialize:checkwazuhRegistry', 'Migrated the registry file.', 'info');
      }
    }
  }; // Init function. Check for wazuh-registry.json file exists.


  const init = async () => {
    await checkWazuhRegistry();
  };

  const createKibanaTemplate = () => {
    (0, _logger.log)('initialize:createKibanaTemplate', `Creating template for ${PLUGIN_PLATFORM_INDEX}`, 'debug');

    try {
      _kibanaTemplate.pluginPlatformTemplate.template = PLUGIN_PLATFORM_INDEX + '*';
    } catch (error) {
      (0, _logger.log)('initialize:createKibanaTemplate', error.message || error);
      context.wazuh.logger.error('Exception: ' + error.message || error);
    }

    return context.core.elasticsearch.client.asInternalUser.indices.putTemplate({
      name: _constants.WAZUH_PLUGIN_PLATFORM_TEMPLATE_NAME,
      order: 0,
      create: true,
      body: _kibanaTemplate.pluginPlatformTemplate
    });
  };

  const createEmptyKibanaIndex = async () => {
    try {
      (0, _logger.log)('initialize:createEmptyKibanaIndex', `Creating ${PLUGIN_PLATFORM_INDEX} index.`, 'info');
      await context.core.elasticsearch.client.asInternalUser.indices.create({
        index: PLUGIN_PLATFORM_INDEX
      });
      (0, _logger.log)('initialize:createEmptyKibanaIndex', `Successfully created ${PLUGIN_PLATFORM_INDEX} index.`, 'debug');
      await init();
    } catch (error) {
      return Promise.reject(new Error(`Error creating ${PLUGIN_PLATFORM_INDEX} index due to ${error.message || error}`));
    }
  };

  const fixKibanaTemplate = async () => {
    try {
      await createKibanaTemplate();
      (0, _logger.log)('initialize:fixKibanaTemplate', `Successfully created ${PLUGIN_PLATFORM_INDEX} template.`, 'debug');
      await createEmptyKibanaIndex();
    } catch (error) {
      return Promise.reject(new Error(`Error creating template for ${PLUGIN_PLATFORM_INDEX} due to ${error.message || error}`));
    }
  };

  const getTemplateByName = async () => {
    try {
      await context.core.elasticsearch.client.asInternalUser.indices.getTemplate({
        name: _constants.WAZUH_PLUGIN_PLATFORM_TEMPLATE_NAME
      });
      (0, _logger.log)('initialize:getTemplateByName', `No need to create the ${PLUGIN_PLATFORM_INDEX} template, already exists.`, 'debug');
      await createEmptyKibanaIndex();
    } catch (error) {
      (0, _logger.log)('initialize:getTemplateByName', error.message || error);
      return fixKibanaTemplate();
    }
  }; // Does Kibana index exist?


  const checkKibanaStatus = async () => {
    try {
      const response = await context.core.elasticsearch.client.asInternalUser.indices.exists({
        index: PLUGIN_PLATFORM_INDEX
      });

      if (response.body) {
        // It exists, initialize!
        await init();
      } else {
        // No Kibana index created...
        (0, _logger.log)('initialize:checkKibanaStatus', `Not found ${PLUGIN_PLATFORM_INDEX} index`, 'info');
        await getTemplateByName();
      }
    } catch (error) {
      (0, _logger.log)('initialize:checkKibanaStatus', error.message || error);
      context.wazuh.logger.error(error.message || error);
    }
  }; // Wait until Elasticsearch js is ready


  const checkStatus = async () => {
    try {
      // TODO: wait until elasticsearch is ready?
      // await server.plugins.elasticsearch.waitUntilReady();
      return await checkKibanaStatus();
    } catch (error) {
      (0, _logger.log)('initialize:checkStatus', 'Waiting for elasticsearch plugin to be ready...', 'debug');
      setTimeout(() => checkStatus(), 3000);
    }
  }; // Check Kibana index and if it is prepared, start the initialization of Wazuh App.


  return checkStatus();
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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