"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WazuhReportingRoutes = WazuhReportingRoutes;

var _controllers = require("../controllers");

var _configSchema = require("@kbn/config-schema");

/*
 * Wazuh app - Module for Wazuh reporting routes
 * Copyright (C) 2015-2022 Wazuh, Inc.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Find more information about this on the LICENSE file.
 */
function WazuhReportingRoutes(router) {
  const ctrl = new _controllers.WazuhReportingCtrl();

  const agentIDValidation = _configSchema.schema.string({
    minLength: 3,
    validate: agentID => /^\d{3,}$/.test(agentID) ? undefined : 'must be 0-9 are allowed'
  });

  const groupIDValidation = _configSchema.schema.string({
    minLength: 1,
    validate: agentID => /^(?!^(\.{1,2}|all)$)[\w\.\-]+$/.test(agentID) ? undefined : 'must be A-z, 0-9, _, . are allowed. It must not be ., .. or all.'
  });

  const ReportFilenameValidation = _configSchema.schema.string({
    validate: agentID => /^[\w\-\.]+\.pdf$/.test(agentID) ? undefined : 'must be A-z, 0-9, _, ., and - are allowed. It must end with .pdf.'
  });

  const moduleIDValidation = _configSchema.schema.oneOf([_configSchema.schema.literal('general'), _configSchema.schema.literal('fim'), _configSchema.schema.literal('aws'), _configSchema.schema.literal('gcp'), _configSchema.schema.literal('pm'), _configSchema.schema.literal('audit'), _configSchema.schema.literal('sca'), _configSchema.schema.literal('office'), _configSchema.schema.literal('github'), _configSchema.schema.literal('ciscat'), _configSchema.schema.literal('vuls'), _configSchema.schema.literal('mitre'), _configSchema.schema.literal('virustotal'), _configSchema.schema.literal('docker'), _configSchema.schema.literal('osquery'), _configSchema.schema.literal('oscap'), _configSchema.schema.literal('pci'), _configSchema.schema.literal('hipaa'), _configSchema.schema.literal('nist'), _configSchema.schema.literal('gdpr'), _configSchema.schema.literal('tsc')]);

  router.post({
    path: '/reports/modules/{moduleID}',
    validate: {
      body: _configSchema.schema.object({
        array: _configSchema.schema.any(),
        browserTimezone: _configSchema.schema.string(),
        serverSideQuery: _configSchema.schema.maybe(_configSchema.schema.any()),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        agents: _configSchema.schema.maybe(_configSchema.schema.oneOf([agentIDValidation, _configSchema.schema.boolean()])),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        searchBar: _configSchema.schema.maybe(_configSchema.schema.string()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        tab: _configSchema.schema.string(),
        tables: _configSchema.schema.maybe(_configSchema.schema.any()),
        time: _configSchema.schema.oneOf([_configSchema.schema.object({
          from: _configSchema.schema.string(),
          to: _configSchema.schema.string()
        }), _configSchema.schema.string()]),
        indexPatternTitle: _configSchema.schema.string(),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        moduleID: moduleIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsModules(context, request, response));
  router.post({
    path: '/reports/groups/{groupID}',
    validate: {
      body: _configSchema.schema.object({
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        groupID: groupIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsGroups(context, request, response));
  router.post({
    path: '/reports/agents/{agentID}',
    validate: {
      body: _configSchema.schema.object({
        browserTimezone: _configSchema.schema.string(),
        filters: _configSchema.schema.any(),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        agentID: agentIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsAgentsConfiguration(context, request, response));
  router.post({
    path: '/reports/agents/{agentID}/inventory',
    validate: {
      body: _configSchema.schema.object({
        array: _configSchema.schema.any(),
        browserTimezone: _configSchema.schema.string(),
        serverSideQuery: _configSchema.schema.maybe(_configSchema.schema.any()),
        filters: _configSchema.schema.maybe(_configSchema.schema.any()),
        agents: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.boolean()])),
        components: _configSchema.schema.maybe(_configSchema.schema.any()),
        searchBar: _configSchema.schema.maybe(_configSchema.schema.oneOf([_configSchema.schema.string(), _configSchema.schema.boolean()])),
        section: _configSchema.schema.maybe(_configSchema.schema.string()),
        tab: _configSchema.schema.string(),
        tables: _configSchema.schema.maybe(_configSchema.schema.any()),
        time: _configSchema.schema.oneOf([_configSchema.schema.object({
          from: _configSchema.schema.string(),
          to: _configSchema.schema.string()
        }), _configSchema.schema.string()]),
        indexPatternTitle: _configSchema.schema.string(),
        apiId: _configSchema.schema.string()
      }),
      params: _configSchema.schema.object({
        agentID: agentIDValidation
      })
    }
  }, (context, request, response) => ctrl.createReportsAgentsInventory(context, request, response)); // Fetch specific report

  router.get({
    path: '/reports/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: ReportFilenameValidation
      })
    }
  }, (context, request, response) => ctrl.getReportByName(context, request, response)); // Delete specific report

  router.delete({
    path: '/reports/{name}',
    validate: {
      params: _configSchema.schema.object({
        name: ReportFilenameValidation
      })
    }
  }, (context, request, response) => ctrl.deleteReportByName(context, request, response)); // Fetch the reports list

  router.get({
    path: '/reports',
    validate: false
  }, (context, request, response) => ctrl.getReports(context, request, response));
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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